﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_DeltaApplier.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/htc/htc_Api.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <nn/fssystem/fs_HostFileSystem.h>
#else
    #include <nn/fssystem/fs_TmFileSystem.h>
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

namespace {

#if defined(NN_BUILD_CONFIG_OS_WIN)
    typedef nn::fssystem::HostFileSystem HostFileSystem;
#else
    typedef nn::fssystem::TmFileSystem HostFileSystem;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

typedef nn::fs::fsa::IFile IFile;

/**
 * @brief   ContentStorage のインターフェースを使ってホストファイルに書き込むテスト用のクラスです。
 */
class ContentStorageMock
{
public:
    struct Option
    {
        bool isNotEnoughSpaceExpected;

        Option() NN_NOEXCEPT
        {
            isNotEnoughSpaceExpected = false;
        }
    };

public:
    explicit ContentStorageMock(IFile* pFile) NN_NOEXCEPT
        : m_pFile(pFile),
          m_IsNotEnoughSpaceExpected(false),
          m_VirtualFileSize(0)
    {
    }

    ContentStorageMock(IFile* pFile, const Option& option) NN_NOEXCEPT
        : m_pFile(pFile),
          m_IsNotEnoughSpaceExpected(option.isNotEnoughSpaceExpected),
          m_VirtualFileSize(0)
    {
    }

    nn::Result GeneratePlaceHolderId(nn::sf::Out<nn::ncm::PlaceHolderId> outValue) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result CreatePlaceHolder(const nn::ncm::PlaceHolderId& placeHolderId, const nn::ncm::ContentId& contentId, std::int64_t size) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    nn::Result DeletePlaceHolder(const nn::ncm::PlaceHolderId& id) NN_NOEXCEPT
    {
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result HasPlaceHolder(nn::sf::Out<bool> outValue, const nn::ncm::PlaceHolderId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result WritePlaceHolder(const nn::ncm::PlaceHolderId& id, std::int64_t offset, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_UNUSED(id);
        if (m_pFile == nullptr)
        {
            if (offset + static_cast<int64_t>(buffer.GetSize()) > m_VirtualFileSize)
            {
                NN_RESULT_THROW(nn::fs::ResultOutOfRange());
            }
            NN_RESULT_SUCCESS;
        }
        return m_pFile->Write(offset, buffer.GetPointerUnsafe(), buffer.GetSize(), nn::fs::WriteOption::MakeValue(0));
    }

    nn::Result Register(const nn::ncm::PlaceHolderId& placeHolderId, const nn::ncm::ContentId& contentId) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);
        NN_RESULT_SUCCESS;
    }

    nn::Result Delete(const nn::ncm::ContentId& id) NN_NOEXCEPT
    {
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result Has(nn::sf::Out<bool> outValue, const nn::ncm::ContentId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPath(nn::sf::Out<nn::ncm::Path> outValue, const nn::ncm::ContentId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetPlaceHolderPath(nn::sf::Out<nn::ncm::Path> outValue, const nn::ncm::PlaceHolderId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result CleanupAllPlaceHolder() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    nn::Result ListPlaceHolder(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::PlaceHolderId>& outList) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_UNUSED(outList);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetContentCount(nn::sf::Out<std::int32_t> outCount) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_RESULT_SUCCESS;
    }

    nn::Result ListContentId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ContentId>& outList, std::int32_t offset) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_UNUSED(outList);
        NN_UNUSED(offset);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSizeFromContentId(nn::sf::Out<std::int64_t> outValue, const nn::ncm::ContentId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSizeFromPlaceHolderId(nn::sf::Out<std::int64_t> outValue, const nn::ncm::PlaceHolderId& id) const NN_NOEXCEPT
    {
        NN_UNUSED(id);

        if (m_pFile == nullptr)
        {
            outValue.Set(m_VirtualFileSize);
            NN_RESULT_SUCCESS;
        }

        int64_t value;
        NN_RESULT_DO(m_pFile->GetSize(&value));

        outValue.Set(value);

        NN_RESULT_SUCCESS;
    }

    nn::Result DisableForcibly() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    nn::Result RevertToPlaceHolder(const nn::ncm::PlaceHolderId& placeHolderId, const nn::ncm::ContentId& contentId, const nn::ncm::ContentId& postContentId) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);
        NN_UNUSED(postContentId);
        NN_RESULT_SUCCESS;
    }

    nn::Result SetPlaceHolderSize(const nn::ncm::PlaceHolderId& placeHolderId, int64_t size) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);

        NN_RESULT_THROW_UNLESS(!m_IsNotEnoughSpaceExpected, nn::fs::ResultUsableSpaceNotEnough());
        if (m_pFile == nullptr)
        {
            m_VirtualFileSize = size;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(m_pFile->SetSize(size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadContentIdFile(nn::sf::OutBuffer buffer, nn::ncm::ContentId id, int64_t offset) const NN_NOEXCEPT
    {
        NN_UNUSED(buffer);
        NN_UNUSED(id);
        NN_UNUSED(offset);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetRightsIdFromPlaceHolderId(nn::sf::Out<nn::ncm::RightsId> outValue, nn::ncm::PlaceHolderId id) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetRightsIdFromContentId(nn::sf::Out<nn::ncm::RightsId> outValue, nn::ncm::ContentId id) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_RESULT_SUCCESS;
    }

    nn::Result WriteContentForDebug(nn::ncm::ContentId contentId, int64_t offset, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_UNUSED(contentId);
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetFreeSpaceSize(nn::sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetTotalSpaceSize(nn::sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result FlushPlaceHolder() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    nn::Result RepairInvalidFileAttribute() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

private:
    IFile* m_pFile;
    bool m_IsNotEnoughSpaceExpected;
    int64_t m_VirtualFileSize;
};

/**
 * @brief   DeltaApplierTest で利用するテストフィクスチャです。
 */
class DeltaApplierTest : public ::testing::Test
{
public:
    void CheckDeltaAppliedFile() NN_NOEXCEPT;

    void CopyPatchFileToWorkDirectory() NN_NOEXCEPT;

    std::unique_ptr<nn::fs::fsa::IFileSystem>& GetHostFs() NN_NOEXCEPT
    {
        return m_pHostFs;
    }

    const nnt::fs::util::String& GetTestPath() NN_NOEXCEPT
    {
        return m_TestPath;
    }

    const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT
    {
        return m_RootPath;
    }

protected:

    /**
    * @brief テスト開始時に毎回呼び出される関数です。
    */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_RootDirectory.Create();

        {
            auto* pHostFs = new HostFileSystem();
            GTEST_ASSERT_NE(nullptr, pHostFs);
#if NN_BUILD_CONFIG_OS_WIN
            NNT_ASSERT_RESULT_SUCCESS(pHostFs->Initialize(""));
#else
            NNT_ASSERT_RESULT_SUCCESS(pHostFs->Initialize());
#endif // NN_BUILD_CONFIG_OS_WIN
            m_pHostFs.reset(pHostFs);
        }

        m_TestPath = nnt::fs::util::String(nnt::fs::util::GetHostTemporaryPath());
        m_RootPath = m_RootDirectory.GetPath();
    }

    /**
    * @brief テスト終了時に毎回呼び出される関数です。
    */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pHostFs.reset(nullptr);
        m_RootDirectory.Delete();
    }

private:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pHostFs;
    nnt::fs::util::String m_TestPath;
    nnt::fs::util::TemporaryHostDirectory m_RootDirectory;
    nnt::fs::util::String m_RootPath;
};

}

void DeltaApplierTest::CopyPatchFileToWorkDirectory() NN_NOEXCEPT
{
    // 元ファイルを結果ファイルにコピー

    std::unique_ptr<IFile> filePatch;
    {
        auto filePath = GetTestPath() + "/source.bin";
        NN_SDK_LOG("open file: %s\n", filePath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &filePatch,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
        ));
    }

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(filePatch->GetSize(&size));

    std::unique_ptr<IFile> fileActual;
    {
        auto filePath = GetRootPath() + "/actual.bin";
        NN_SDK_LOG("create file: %s\n", filePath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->CreateFile(filePath.c_str(), size, 0));
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &fileActual,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
        ));
    }

    {
        const int WorkBufferSize = 64 * 1024;

        int64_t offset = 0;
        std::unique_ptr<char> pBuffer(new char[WorkBufferSize]);
        GTEST_ASSERT_NE(nullptr, pBuffer.get());
        while (offset < size)
        {
            size_t sizeCurrent;
            NNT_ASSERT_RESULT_SUCCESS(filePatch->Read(&sizeCurrent, offset, pBuffer.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
            NNT_ASSERT_RESULT_SUCCESS(fileActual->Write(offset, pBuffer.get(), sizeCurrent, nn::fs::WriteOption::MakeValue(0)));
            offset += sizeCurrent;
        }
    }
}

void DeltaApplierTest::CheckDeltaAppliedFile() NN_NOEXCEPT
{
    // パッチ適用した後のファイルと比較する

    int64_t sizePatch;
    {
        std::unique_ptr<IFile> filePatch;
        auto filePath = GetTestPath() + "/source.bin";
        NN_SDK_LOG("open file: %s\n", filePath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &filePatch,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
        ));
        NNT_ASSERT_RESULT_SUCCESS(filePatch->GetSize(&sizePatch));
    }

    std::unique_ptr<IFile> fileActual;
    {
        auto filePath = GetRootPath() + "/actual.bin";
        NN_SDK_LOG("open file: %s\n", filePath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &fileActual,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
        ));
    }

    std::unique_ptr<IFile> fileExpected;
    {
        auto filePath = GetTestPath() + "/expected.bin";
        NN_SDK_LOG("open file: %s\n", filePath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &fileExpected,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
        ));
    }

    int64_t sizeFileExpected;
    int64_t sizeFileActual;
    NNT_ASSERT_RESULT_SUCCESS(fileExpected->GetSize(&sizeFileExpected));
    NNT_ASSERT_RESULT_SUCCESS(fileActual->GetSize(&sizeFileActual));
    if( sizeFileExpected < sizePatch )
    {
        // shrink はしない。適用側で実行する。
        ASSERT_EQ(sizePatch, sizeFileActual);
    }
    else
    {
        ASSERT_EQ(sizeFileExpected, sizeFileActual);
    }

    {
        const int WorkBufferSize = 64 * 1024;

        int64_t offset = 0;
        std::unique_ptr<char> pBufferAns(new char[WorkBufferSize]);
        GTEST_ASSERT_NE(nullptr, pBufferAns.get());
        std::unique_ptr<char> pBufferResult(new char[WorkBufferSize]);
        GTEST_ASSERT_NE(nullptr, pBufferResult.get());
        while (offset < sizeFileExpected)
        {
            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(fileActual->Read(&sizeRead, offset, pBufferResult.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
            NNT_ASSERT_RESULT_SUCCESS(fileExpected->Read(&sizeRead, offset, pBufferAns.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));

            ASSERT_EQ(0, std::memcmp(pBufferAns.get(), pBufferResult.get(), sizeRead)) << "offset " << offset << " size " << sizeRead << " invalid data";

            offset += sizeRead;
        }
    }
}

TEST_F(DeltaApplierTest, ApplyBasic)
{
    const int WorkBufferSize = 64 * 1024;

    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());
    std::unique_ptr<char> pBuffer(new char[WorkBufferSize]);
    GTEST_ASSERT_NE(nullptr, pBuffer.get());

    CopyPatchFileToWorkDirectory();

    auto filePathPatch = GetRootPath() + "/actual.bin";
    auto filePathDelta = GetTestPath() + "/patch.bdiff";

    // パッチを適用
    {
        std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
            new nn::ncm::DeltaApplier()
        );
        GTEST_ASSERT_NE(nullptr, pApplier.get());

        std::unique_ptr<IFile> filePatch;
        {
            NN_SDK_LOG("open file: %s\n", filePathPatch.c_str());
            NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                &filePatch,
                filePathPatch.c_str(),
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
            ));
        }

        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(filePatch.get()));
        nn::ncm::ContentStorage storage(mockStorage);
        nn::ncm::PlaceHolderId id;

        int64_t offset = 0;

        {
            std::memset(pStatus.get(), 0, nn::ncm::DeltaApplier::StatusSize);
            NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));
            ASSERT_EQ(0, offset);
        }

        int64_t fileSizeDelta = 0;

        {
            std::unique_ptr<IFile> fileDelta;

            NN_SDK_LOG("open file: %s\n", filePathDelta.c_str());
            NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                &fileDelta,
                filePathDelta.c_str(),
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            ));

            NNT_ASSERT_RESULT_SUCCESS(fileDelta->GetSize(&fileSizeDelta));
        }

        {
            while (offset < fileSizeDelta)
            {
                NN_SDK_LOG("%d/%d\n", offset, fileSizeDelta);

                std::unique_ptr<IFile> fileDelta;

                NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                    &fileDelta,
                    filePathDelta.c_str(),
                    static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
                ));

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(fileDelta->Read(&sizeRead, offset, pBuffer.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), sizeRead));
                offset += sizeProcessed;

                fileDelta.reset(nullptr);
            }
        }
    }

    CheckDeltaAppliedFile();
} // NOLINT(impl/function_size)

TEST_F(DeltaApplierTest, ApplyVariousSizesLarge)
{
    const int WorkBufferSize = 1024 * 1024;

    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());
    std::unique_ptr<char> pBuffer(new char[WorkBufferSize]);
    GTEST_ASSERT_NE(nullptr, pBuffer.get());

    // パッチを適用
    {
        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
        nn::ncm::ContentStorage storage(mockStorage);
        nn::ncm::PlaceHolderId id;

        int64_t offset = 0;

        {
            std::memset(pStatus.get(), 0, nn::ncm::DeltaApplier::StatusSize);

            nn::ncm::DeltaApplier applier;
            NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), 0));

            ASSERT_EQ(0, offset);
        }

        {
            auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pBuffer.get());

            {
                pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);
                pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
                pHeader->sizeBody = 0x1000000000;
                pHeader->sizeDestination = 0x70000000000000;
                pHeader->sizeSource = 0x70000000000000;

                nn::ncm::DeltaApplier applier;
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), 0));
                NNT_ASSERT_RESULT_SUCCESS(applier.ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), nn::fssystem::DeltaHeader::Size));
                ASSERT_EQ(sizeProcessed, pHeader->offsetBody);
            }

            {
                nn::ncm::DeltaApplier applier;
                NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
                ASSERT_EQ(pHeader->offsetBody, offset);
            }
        }

        for (int64_t size = 1; size < 0x1000000000; size *= 2)
        {
            NN_LOG("Size %llX\n", size);

            int64_t offsetNew;
            nn::ncm::DeltaApplier applier;
            NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offsetNew, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
            ASSERT_EQ(offset, offsetNew);

            nn::fssystem::DeltaCommandWrite::Create(pBuffer.get(), nn::fssystem::DeltaHeader::Size, 0, size);
            auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, size) + size;
            offset += commandSize;

            int64_t applied = 0;
            while (applied < commandSize)
            {
                auto currentSize = static_cast<size_t>(std::min<int64_t>(commandSize - applied, WorkBufferSize));

                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(applier.ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), currentSize));
                applied += static_cast<int64_t>(sizeProcessed);
            }
        }
    }
}

TEST_F(DeltaApplierTest, ApplyVariousOffsetsLarge)
{
    const int WorkBufferSize = 1024 * 1024;

    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());
    std::unique_ptr<char> pBuffer(new char[WorkBufferSize]);
    GTEST_ASSERT_NE(nullptr, pBuffer.get());

    // パッチを適用
    {
        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
        nn::ncm::ContentStorage storage(mockStorage);
        nn::ncm::PlaceHolderId id;

        int64_t offset = 0;

        {
            std::memset(pStatus.get(), 0, nn::ncm::DeltaApplier::StatusSize);

            nn::ncm::DeltaApplier applier;
            NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), 0));

            ASSERT_EQ(0, offset);
        }

        {
            auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pBuffer.get());

            {
                pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);
                pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
                pHeader->sizeBody = 0x1000000000;
                pHeader->sizeDestination = 0x70000000000000;
                pHeader->sizeSource = 0x70000000000000;

                nn::ncm::DeltaApplier applier;
                NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), 0));
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(applier.ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), nn::fssystem::DeltaHeader::Size));
                ASSERT_EQ(sizeProcessed, pHeader->offsetBody);
            }

            {
                nn::ncm::DeltaApplier applier;
                NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offset, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
                ASSERT_EQ(pHeader->offsetBody, offset);
            }
        }

        for (int64_t commandOffset = 1; commandOffset < 0x1000000000; commandOffset *= 2)
        {
            NN_LOG("Offset %llX\n", commandOffset);

            int64_t offsetNew;
            nn::ncm::DeltaApplier applier;
            NNT_ASSERT_RESULT_SUCCESS(applier.Initialize(&offsetNew, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
            ASSERT_EQ(offset, offsetNew);

            nn::fssystem::DeltaCommandWrite::Create(pBuffer.get(), nn::fssystem::DeltaHeader::Size, commandOffset, 4);
            auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(commandOffset, 4) + 4;
            offset += commandSize;

            int64_t applied = 0;
            while (applied < commandSize)
            {
                auto currentSize = static_cast<size_t>(std::min<int64_t>(commandSize - applied, WorkBufferSize));

                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(applier.ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), currentSize));
                applied += static_cast<int64_t>(sizeProcessed);
            }
        }
    }
}

TEST_F(DeltaApplierTest, ApplySuspendAndResume)
{
    const int WorkBufferSize = 64 * 1024;

    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());
    std::unique_ptr<char> pStatusBackup(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatusBackup.get());
    std::unique_ptr<char> pBuffer(new char[WorkBufferSize]);
    GTEST_ASSERT_NE(nullptr, pBuffer.get());
    nn::ncm::PlaceHolderId id;

    CopyPatchFileToWorkDirectory();

    auto filePathPatch = GetRootPath() + "/actual.bin";
    auto filePathDelta = GetTestPath() + "/patch.bdiff";

    // パッチを適用 (途中でサスペンド)
    {
        std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
            new nn::ncm::DeltaApplier()
        );
        GTEST_ASSERT_NE(nullptr, pApplier.get());

        std::unique_ptr<IFile> filePatch;

        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &filePatch,
            filePathPatch.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
        ));

        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(filePatch.get()));
        nn::ncm::ContentStorage storage(mockStorage);

        int64_t offset = 0;

        {
            std::memset(pStatus.get(), 0, nn::ncm::DeltaApplier::StatusSize);
            NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));
            ASSERT_EQ(0, offset);
        }

        int64_t fileSizeDelta = 0;

        {
            std::unique_ptr<IFile> fileDelta;

            NN_SDK_LOG("open file: %s\n", filePathDelta.c_str());
            NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                &fileDelta,
                filePathDelta.c_str(),
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            ));

            NNT_ASSERT_RESULT_SUCCESS(fileDelta->GetSize(&fileSizeDelta));
        }

        {
            while (offset < fileSizeDelta / 2)
            {
                std::unique_ptr<IFile> fileDelta;

                NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                    &fileDelta,
                    filePathDelta.c_str(),
                    static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
                ));

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(fileDelta->Read(&sizeRead, offset, pBuffer.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), sizeRead));
                offset += sizeProcessed;

                fileDelta.reset(nullptr);
            }
        }
    }

    std::memcpy(pStatusBackup.get(), pStatus.get(), nn::ncm::DeltaApplier::StatusSize);

    // パッチを適用 (再開して電源断を仮定した中断でファイルが巻き戻る)
    {
        std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
            new nn::ncm::DeltaApplier()
        );
        GTEST_ASSERT_NE(nullptr, pApplier.get());

        std::unique_ptr<IFile> filePatch;

        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &filePatch,
            filePathPatch.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
        ));

        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(filePatch.get()));
        nn::ncm::ContentStorage storage(mockStorage);

        int64_t offset = 0;

        {
            NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
            ASSERT_LT(0, offset);
        }

        int64_t fileSizeDelta = 0;

        {
            std::unique_ptr<IFile> fileDelta;

            NN_SDK_LOG("open file: %s\n", filePathDelta.c_str());
            NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                &fileDelta,
                filePathDelta.c_str(),
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            ));

            NNT_ASSERT_RESULT_SUCCESS(fileDelta->GetSize(&fileSizeDelta));
        }

        {
            while (offset < fileSizeDelta * 3 / 4)
            {
                std::unique_ptr<IFile> fileDelta;

                NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                    &fileDelta,
                    filePathDelta.c_str(),
                    static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
                ));

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(fileDelta->Read(&sizeRead, offset, pBuffer.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), sizeRead));
                offset += sizeProcessed;

                fileDelta.reset(nullptr);
            }
        }
    }

    std::memcpy(pStatus.get(), pStatusBackup.get(), nn::ncm::DeltaApplier::StatusSize);

    // パッチを適用 (再開)
    {
        std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
            new nn::ncm::DeltaApplier()
        );
        GTEST_ASSERT_NE(nullptr, pApplier.get());

        std::unique_ptr<IFile> filePatch;

        NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
            &filePatch,
            filePathPatch.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
        ));

        nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(filePatch.get()));
        nn::ncm::ContentStorage storage(mockStorage);

        int64_t offset = 0;

        {
            NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), nn::ncm::DeltaApplier::StatusSize));
            ASSERT_LT(0, offset);
        }

        int64_t fileSizeDelta = 0;

        {
            std::unique_ptr<IFile> fileDelta;

            NN_SDK_LOG("open file: %s\n", filePathDelta.c_str());
            NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                &fileDelta,
                filePathDelta.c_str(),
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            ));

            NNT_ASSERT_RESULT_SUCCESS(fileDelta->GetSize(&fileSizeDelta));
        }

        {
            while (offset < fileSizeDelta)
            {
                std::unique_ptr<IFile> fileDelta;

                NNT_ASSERT_RESULT_SUCCESS(GetHostFs()->OpenFile(
                    &fileDelta,
                    filePathDelta.c_str(),
                    static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
                ));

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(fileDelta->Read(&sizeRead, offset, pBuffer.get(), WorkBufferSize, nn::fs::ReadOption::MakeValue(0)));
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(
                    &sizeProcessed,
                    pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                    pBuffer.get(), sizeRead));
                offset += sizeProcessed;

                fileDelta.reset(nullptr);
            }
        }
    }

    CheckDeltaAppliedFile();
} // NOLINT(impl/function_size)

TEST_F(DeltaApplierTest, InvalidHeader)
{
    std::unique_ptr<char> pHeaderBuffer(new char[nn::fssystem::DeltaHeader::Size]);
    GTEST_ASSERT_NE(nullptr, pHeaderBuffer.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pHeaderBuffer.get());
    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    static const std::function<void ()> tries[] = {
        // ゼロ フィル
        [&]() NN_NOEXCEPT { std::memset(pHeaderBuffer.get(), 0, nn::fssystem::DeltaHeader::Size); },
        // 負値
        [&]() NN_NOEXCEPT { pHeader->sizeSource = -1; },
        [&]() NN_NOEXCEPT { pHeader->sizeDestination = -1; },
        [&]() NN_NOEXCEPT { pHeader->offsetBody = -1; },
        [&]() NN_NOEXCEPT { pHeader->sizeBody = -1; },
        // ヘッダー サイズの異常
        [&]() NN_NOEXCEPT { pHeader->offsetBody = sizeof(*pHeader) - 4; }
    };
    for (auto lambdaTry : tries)
    {
        int64_t offset;
        NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

        // 正常値にする
        pHeader->sizeSource = 4;
        pHeader->sizeDestination = 4;
        pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
        pHeader->sizeBody = 4;

        // 異常値にする
        lambdaTry();

        size_t sizeProcessed;
        NNT_ASSERT_RESULT_FAILURE(
            nn::ncm::ResultInvalidDeltaFormat,
            pApplier->ApplyDelta(
                &sizeProcessed,
                pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));
    }
}

TEST_F(DeltaApplierTest, InvalidStatus)
{
    struct StatusHeader
    {
        int64_t offsetDelta;
        int64_t offsetPatch;

        int32_t offsetHeader;
        int32_t sizeHeader;
    };

    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());
    auto pStatusHeader = reinterpret_cast<StatusHeader*>(pStatus.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pStatus.get() + sizeof(*pStatusHeader));
    size_t statusSize = 0;

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    static const std::function<void ()> tries[] = {
        // ステータス ヘッダー部分 負値
        [&]() NN_NOEXCEPT { pStatusHeader->offsetDelta = -1; },
        [&]() NN_NOEXCEPT { pStatusHeader->offsetPatch = -1; },
        [&]() NN_NOEXCEPT { pStatusHeader->offsetHeader = -1; },
        [&]() NN_NOEXCEPT { pStatusHeader->sizeHeader = -1; },
        // ステータス ヘッダー部分 サイズ異常
        [&]() NN_NOEXCEPT { pStatusHeader->offsetHeader = 0; },
        [&]() NN_NOEXCEPT { pStatusHeader->sizeHeader = sizeof(*pHeader) - 4; },
        // ヘッダー部分 ゼロ フィル
        [&]() NN_NOEXCEPT { std::memset(pHeader, 0, nn::fssystem::DeltaHeader::Size); },
        // ヘッダー部分 負値
        [&]() NN_NOEXCEPT { pHeader->sizeSource = -1; },
        [&]() NN_NOEXCEPT { pHeader->sizeDestination = -1; },
        [&]() NN_NOEXCEPT { pHeader->offsetBody = -1; },
        [&]() NN_NOEXCEPT { pHeader->sizeBody = -1; },
        // ヘッダー部分 ヘッダー サイズの異常
        [&]() NN_NOEXCEPT { pHeader->offsetBody = sizeof(*pHeader) - 4; },
        // ヘッダー部分 ヘッダー バージョンの異常
        [&]() NN_NOEXCEPT { pHeader->signature = 0; }
    };

    for (auto lambdaTry : tries)
    {
        // 正常値にする
        pStatusHeader->sizeHeader = sizeof(*pHeader);
        pStatusHeader->offsetPatch = 0;
        pStatusHeader->offsetHeader = 0;
        pStatusHeader->offsetDelta = 0;

        pHeader->sizeSource = 4;
        pHeader->sizeDestination = 4;
        pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
        pHeader->sizeBody = 4;
        pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

        statusSize = sizeof(*pStatusHeader) + sizeof(*pHeader);

        // 異常値にする
        lambdaTry();

        int64_t offset;
        NNT_ASSERT_RESULT_FAILURE(
            nn::ncm::ResultInvalidApplyDeltaStatus,
            pApplier->Initialize(&offset, &storage, id, pStatus.get(), statusSize));
    }
}

TEST_F(DeltaApplierTest, InvalidBody)
{
    std::unique_ptr<char> pHeaderBuffer(new char[nn::fssystem::DeltaHeader::Size]);
    GTEST_ASSERT_NE(nullptr, pHeaderBuffer.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pHeaderBuffer.get());
    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    int64_t offset;
    NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

    char invalidBodyData[4];
    std::memset(invalidBodyData, 0xFF, sizeof(invalidBodyData));

    pHeader->sizeSource = 4;
    pHeader->sizeDestination = 4;
    pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
    pHeader->sizeBody = sizeof(invalidBodyData);
    pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

    size_t sizeProcessed;
    NNT_ASSERT_RESULT_SUCCESS(
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));

    NNT_ASSERT_RESULT_FAILURE(
        nn::ncm::ResultInvalidDeltaFormat,
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            invalidBodyData, sizeof(invalidBodyData)));
}

TEST_F(DeltaApplierTest, TooLargeBody)
{
    std::unique_ptr<char> pHeaderBuffer(new char[nn::fssystem::DeltaHeader::Size]);
    GTEST_ASSERT_NE(nullptr, pHeaderBuffer.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pHeaderBuffer.get());
    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    int64_t offset;
    NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

    pHeader->sizeSource = 1024;
    pHeader->sizeDestination = 1024;
    pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
    pHeader->sizeBody = 4;
    pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

    size_t sizeProcessed;
    NNT_ASSERT_RESULT_SUCCESS(
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));

    // ヘッダーで指定したサイズより大きなバッファーを書き込んだらエラー
    nn::fssystem::DeltaCommandWrite::Create(pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size, 0, 4);
    auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, 4) + 4;

    NNT_ASSERT_RESULT_FAILURE(
        nn::ncm::ResultInvalidDeltaFormat,
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            pHeaderBuffer.get(), commandSize));
}

TEST_F(DeltaApplierTest, OverDestination)
{
    std::unique_ptr<char> pHeaderBuffer(new char[nn::fssystem::DeltaHeader::Size]);
    GTEST_ASSERT_NE(nullptr, pHeaderBuffer.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pHeaderBuffer.get());
    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    // ヘッダーで指定したサイズをシークが超えるとエラー
    {
        pHeader->sizeSource = 4;
        pHeader->sizeDestination = 4;
        pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
        pHeader->sizeBody = 1024;
        pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

        int64_t offset;
        NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

        size_t sizeProcessed;
        NNT_ASSERT_RESULT_SUCCESS(
            pApplier->ApplyDelta(
                &sizeProcessed,
                pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));
        ASSERT_EQ(sizeProcessed, pHeader->offsetBody);

        // 後続のデータが必要なので Seek x 2 とする
        auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(8, 0);
        nn::fssystem::DeltaCommandWrite::Create(pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size, 8, 0);
        nn::fssystem::DeltaCommandWrite::Create(pHeaderBuffer.get() + commandSize, nn::fssystem::DeltaHeader::Size, 8, 0);

        NNT_ASSERT_RESULT_FAILURE(
            nn::ncm::ResultInvalidDeltaFormat,
            pApplier->ApplyDelta(
                &sizeProcessed,
                pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                pHeaderBuffer.get(), commandSize * 2));
    }

    // ヘッダーで指定したサイズを書き込みが超えるとエラー
    {
        pHeader->sizeSource = 4;
        pHeader->sizeDestination = 4;
        pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
        pHeader->sizeBody = 1024;
        pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

        int64_t offset;
        NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

        size_t sizeProcessed;
        NNT_ASSERT_RESULT_SUCCESS(
            pApplier->ApplyDelta(
                &sizeProcessed,
                pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));
        ASSERT_EQ(sizeProcessed, pHeader->offsetBody);

        nn::fssystem::DeltaCommandWrite::Create(pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size, 0, 8);
        auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, 8) + 8;

        NNT_ASSERT_RESULT_FAILURE(
            nn::ncm::ResultInvalidDeltaFormat,
            pApplier->ApplyDelta(
                &sizeProcessed,
                pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
                pHeaderBuffer.get(), commandSize));
    }
}

TEST_F(DeltaApplierTest, NotEnoughSpace)
{
    std::unique_ptr<char> pHeaderBuffer(new char[nn::fssystem::DeltaHeader::Size]);
    GTEST_ASSERT_NE(nullptr, pHeaderBuffer.get());
    auto pHeader = reinterpret_cast<nn::fssystem::DeltaHeader*>(pHeaderBuffer.get());
    std::unique_ptr<char> pStatus(new char[nn::ncm::DeltaApplier::StatusSize]);
    GTEST_ASSERT_NE(nullptr, pStatus.get());

    ContentStorageMock::Option option;
    option.isNotEnoughSpaceExpected = true;

    nn::sf::SharedPointer<nn::ncm::IContentStorage> mockStorage(nn::sf::CreateSharedObjectEmplaced<nn::ncm::IContentStorage, ContentStorageMock>(nullptr, option));
    nn::ncm::ContentStorage storage(mockStorage);
    nn::ncm::PlaceHolderId id;

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(
        new nn::ncm::DeltaApplier()
    );
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    pHeader->sizeSource = 0;
    pHeader->sizeDestination = 4;
    pHeader->offsetBody = nn::fssystem::DeltaHeader::Size;
    pHeader->sizeBody = 1024;
    pHeader->signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);

    int64_t offset;
    NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, id, pStatus.get(), 0));

    size_t sizeProcessed;
    NNT_ASSERT_RESULT_SUCCESS(
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size));

    nn::fssystem::DeltaCommandWrite::Create(pHeaderBuffer.get(), nn::fssystem::DeltaHeader::Size, 0, 4);
    auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, 4) + 4;

    NNT_ASSERT_RESULT_FAILURE(
        nn::ncm::ResultNotEnoughSpaceToApplyDelta,
        pApplier->ApplyDelta(
            &sizeProcessed,
            pStatus.get(), nn::ncm::DeltaApplier::StatusSize,
            pHeaderBuffer.get(), commandSize));
}

extern "C" void nnMain()
{
    nn::htc::Initialize();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static const size_t HeapSize = 32 * 1024 * 1024;
    void* pHeapStack = ::malloc(HeapSize);

    nnt::fs::util::InitializeTestLibraryHeap(pHeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    ::free(pHeapStack);

    nn::htc::Finalize();

    nnt::Exit(result);
}
