﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>
#include <new>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/crypto.h>
#include <nn/fs.h>
#include <nn/htc.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_StdAllocationPolicy.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include <nnt/result/testResult_Assert.h>

namespace {
#if defined(NN_BUILD_CONFIG_OS_WIN)
    typedef std::string String;
#else
    typedef nnt::fs::util::String String;
#endif
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentStorageFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentStorage> ContentStorageShared;

    String GetNowString()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        using namespace std::chrono;

        auto duration = high_resolution_clock::now().time_since_epoch();
        time_t t = duration_cast<seconds>(duration).count();
        tm now;
        localtime_s(&now, &t);

        char stamp[32];
        sprintf_s(stamp, sizeof(stamp),
            "%04u%02u%02u%02u%02u%02u%03u",
            now.tm_year + 1900, now.tm_mon + 1, now.tm_mday, now.tm_hour, now.tm_min, now.tm_sec,
            static_cast<uint32_t>(duration_cast<milliseconds>(duration).count() % 1000));

        return String(stamp);
#else
        return nnt::fs::util::GetNowString();
#endif
    }

    String CreateTemporaryRootPath()
    {
        return String(nnt::fs::util::GetHostTemporaryPath())
            .append("\\")
            .append("SIGLO_NCM_TEST")
            .append("_")
            .append(GetNowString());
    }

    template<typename HandleT>
    class ScopedHandle
    {
    public:
        typedef void(*CloseFunction)(HandleT);
        explicit ScopedHandle(HandleT handle, CloseFunction func) NN_NOEXCEPT : m_Handle(handle), m_Func(func){}
        ~ScopedHandle()
        {
            m_Func(m_Handle);
        }

    private:
        HandleT         m_Handle;
        CloseFunction   m_Func;
    };

    class ScopedDirectoryHandle : private ScopedHandle<nn::fs::DirectoryHandle>
    {
    public:
        explicit ScopedDirectoryHandle(nn::fs::DirectoryHandle handle) NN_NOEXCEPT : ScopedHandle<nn::fs::DirectoryHandle>(handle, nn::fs::CloseDirectory){}
    };

    nn::Result ShowDirectoryTreeRecursively(const char* rootPath, int indent)
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, rootPath, static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_File | nn::fs::OpenDirectoryMode_Directory)));
        ScopedDirectoryHandle scopedDirectoryHandle(dir);

        for (;;)
        {
            nn::fs::DirectoryEntry entry;
            int64_t numEntries;
            NN_RESULT_DO(nn::fs::ReadDirectory(&numEntries, &entry, dir, 1));
            if (numEntries == 0)
            {
                break;
            }

            for (auto i = 0; i < indent; i++)
            {
                NN_LOG("  ");
            }
            switch (entry.directoryEntryType)
            {
            case nn::fs::DirectoryEntryType_Directory:
                {
                    NN_LOG("%s/\n", entry.name);
                    ShowDirectoryTreeRecursively((std::string(rootPath) + "/" + entry.name).c_str(), indent + 1);
                }
                break;
            case nn::fs::DirectoryEntryType_File:
                {
                    NN_LOG("%s %lld bytes\n", entry.name, entry.fileSize);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ShowDirectoryTree(const char* rootPath)
    {
        NN_LOG("Show directory tree %s\n", rootPath);

        return ShowDirectoryTreeRecursively(rootPath, 0);
    }

    template<typename FuncT>
    void ExecuteAllContentDirectoryHierarchy(FuncT func)
    {
        func(nn::ncm::MakeFlatContentFilePath, nn::ncm::MakeFlatPlaceHolderFilePath);
        func(nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat4KCluster, nn::ncm::MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster);
        func(nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat16KCluster, nn::ncm::MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster);
        func(nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat32KCluster, nn::ncm::MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster);
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return std::free(p);
    }

    struct ContentInfo
    {
        nn::ncm::PlaceHolderId  placeHolderId;
        nn::ncm::ContentId      contentId;
        size_t                  size;
        std::vector<int64_t>    data;
    };

    class ContentStorageTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            auto rootPath = CreateTemporaryRootPath();
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::DeleteFileOrDirectoryIfExists(rootPath.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(rootPath.c_str()));
            m_RootPath = rootPath;
        }

        virtual void TearDown()
        {
            NN_LOG("delete root: %s\n", GetRootPath());
            //NN_ABORT_UNLESS(nn::fs::DeleteDirectoryRecursively(GetRootPath()).IsSuccess());
        }

        static void SetUpTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetRootPath()
        {
            return m_RootPath.c_str();
        }

        std::vector<ContentInfo> MakeRandomContentInfoList(nn::ncm::ContentStorage& contentStorage, int count)
        {

            std::vector<ContentInfo> contentInfoList;
            for (auto i = 0; i < count; i++)
            {
                auto placeHolderId = contentStorage.GeneratePlaceHolderId();
                auto size = GetRandomSize();
                ContentInfo contentInfo = { placeHolderId, GetRandomContentId(), size, GetRandomData(size) };
                contentInfoList.push_back(contentInfo);
            }

            return contentInfoList;
        }

        nn::ncm::ContentId GetRandomContentId()
        {
            nn::ncm::ContentId contentId = {};
            auto low = m_RandomContentIdEngine();
            auto high = m_RandomContentIdEngine();
            std::memcpy(&contentId.data[0], &low, sizeof(low));
            std::memcpy(&contentId.data[8], &high, sizeof(high));

            return contentId;
        }

        size_t GetRandomSize()
        {
            return std::uniform_int_distribution<>(1, 1024 * 1024)(m_RandomSizeEngine);
        }

        std::vector<int64_t> GetRandomData(size_t size)
        {
            size_t count = size / sizeof(int64_t);
            std::vector<int64_t> data;
            data.reserve(count);
            for (size_t i = 0; i < count; i++)
            {
                data.push_back(m_RandomDataEngine());
            }

            return data;
        }


        String          m_RootPath;
        std::mt19937_64 m_RandomContentIdEngine;
        std::mt19937_64 m_RandomSizeEngine;
        std::mt19937_64 m_RandomDataEngine;
    };

    class ContentStorageTestWithMemoryStorage : public testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            m_pStorageBuffer.reset(new char[TotalSize]);
            ASSERT_NE(nullptr, m_pStorageBuffer);

            m_pBaseStorage.reset(new nn::fs::MemoryStorage(m_pStorageBuffer.get(), TotalSize));
            ASSERT_NE(nullptr, m_pBaseStorage.get());

            size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
            m_pFatCacheBuffer.reset(new char[cacheBufferSize]);
            ASSERT_NE(nullptr, m_pFatCacheBuffer.get());

            m_pFatFs.reset(new nn::fat::FatFileSystem());
            ASSERT_NE(nullptr, m_pFatFs.get());
            NNT_ASSERT_RESULT_SUCCESS(m_pFatFs->Initialize(m_pBaseStorage.get(), m_pFatCacheBuffer.get(), cacheBufferSize));
            NNT_ASSERT_RESULT_SUCCESS(m_pFatFs->Format());
            NNT_ASSERT_RESULT_SUCCESS(m_pFatFs->Mount());

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::fsa::Register(MountName, std::move(m_pFatFs)));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(RootPath));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::fsa::Unregister(MountName);
        }

        const char* GetRootPath() const NN_NOEXCEPT
        {
            return RootPath;
        }

    private:
        static const size_t TotalSize = 1 * 1024 * 1024;
        static const char* MountName;
        static const char* RootPath;

    private:
        std::unique_ptr<char[]> m_pStorageBuffer;
        std::unique_ptr<nn::fs::MemoryStorage> m_pBaseStorage;
        std::unique_ptr<nn::fat::FatFileSystem> m_pFatFs;
        std::unique_ptr<char[]> m_pFatCacheBuffer;
    };

    const char* ContentStorageTestWithMemoryStorage::MountName = "test";
    const char* ContentStorageTestWithMemoryStorage::RootPath = "test:/test";

    class ContentStorageIntegrationTest : public testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::ncm::Initialize();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::ncm::Finalize();
        }
    };
}

TEST_F(ContentStorageTestWithMemoryStorage, VerifyInitializeBase)
{
    // 何もないなら ResultContentStorageBaseNotFound
    NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentStorageBaseNotFound, nn::ncm::ContentStorageImpl::VerifyBase(GetRootPath()));

    {
        String path = GetRootPath();
        path += "/registered";
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(path.c_str()));

        // 中途半端なら ResultInvalidContentStorageBase
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultInvalidContentStorageBase, nn::ncm::ContentStorageImpl::VerifyBase(GetRootPath()));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::InitializeBase(GetRootPath()));
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::VerifyBase(GetRootPath()));
    ShowDirectoryTree(GetRootPath());
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::CleanupBase(GetRootPath()));

    // Base に必要なディレクトリは残るので Success する
    ShowDirectoryTree(GetRootPath());
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::VerifyBase(GetRootPath()));
}

#if 0
// TODO: ストレージに空きが無い時のテストを追加・メモリ上に仮想的な FS を作成する機能が欲しい
TEST_F(ContentStorageTest, CreateDeletePlaceHolder)
{
    ExecuteAllContentDirectoryHierarchy(
        [&](nn::ncm::MakeContentPathFunction func, nn::ncm::MakePlaceHolderPathFunction placeHolderFunc)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::InitializeBase(GetRootPath()));

            auto contentStorageShared = ContentStorageFactory::CreateSharedEmplaced<nn::ncm::IContentStorage, nn::ncm::ContentStorageImpl>();
            NNT_EXPECT_RESULT_SUCCESS(ContentStorageFactory::GetEmplacedImplPointer<nn::ncm::ContentStorageImpl>(contentStorageShared)->Initialize(GetRootPath(), func, placeHolderFunc));
            nn::ncm::ContentStorage contentStorage(contentStorageShared);

            auto contentInfoList = MakeRandomContentInfoList(contentStorage, 10);

            for (const auto& i : contentInfoList)
            {
                bool has;
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
                EXPECT_FALSE(has);
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.CreatePlaceHolder(i.placeHolderId, i.contentId, i.size));
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
                EXPECT_TRUE(has);
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderAlreadyExists, contentStorage.CreatePlaceHolder(i.placeHolderId, GetRandomContentId(), i.size));
            }
            ShowDirectoryTree(GetRootPath());

            for (const auto& i : contentInfoList)
            {
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.DeletePlaceHolder(i.placeHolderId));
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderNotFound, contentStorage.DeletePlaceHolder(i.placeHolderId));
            }

            for (const auto& i : contentInfoList)
            {
                bool has;
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
                EXPECT_FALSE(has);
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.CreatePlaceHolder(i.placeHolderId, i.contentId, i.size));
                NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
                EXPECT_TRUE(has);
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderAlreadyExists, contentStorage.CreatePlaceHolder(i.placeHolderId, GetRandomContentId(), i.size));
            }

            nn::ncm::PlaceHolderId buffer[16];
            int count;
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.ListPlaceHolder(&count, buffer, sizeof(buffer) / sizeof(buffer[0])));
            EXPECT_EQ(static_cast<int>(contentInfoList.size()), count);
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.CleanupAllPlaceHolder());
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.ListPlaceHolder(&count, buffer, sizeof(buffer) / sizeof(buffer[0])));
            EXPECT_EQ(0, count);

            NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::CleanupBase(GetRootPath()));
        }
    );
}

TEST_F(ContentStorageTest, RegisterDeleteContents)
{
    ExecuteAllContentDirectoryHierarchy(
        [&](nn::ncm::MakeContentPathFunction func, nn::ncm::MakePlaceHolderPathFunction placeHolderFunc)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::InitializeBase(GetRootPath()));

        auto contentStorageShared = ContentStorageFactory::CreateSharedEmplaced<nn::ncm::IContentStorage, nn::ncm::ContentStorageImpl>();
        NNT_EXPECT_RESULT_SUCCESS(ContentStorageFactory::GetEmplacedImplPointer<nn::ncm::ContentStorageImpl>(contentStorageShared)->Initialize(GetRootPath(), func, placeHolderFunc));
        nn::ncm::ContentStorage contentStorage(contentStorageShared);

        auto contentInfoList = MakeRandomContentInfoList(contentStorage, 10);

        for (const auto& i : contentInfoList)
        {
            auto dummyData = GetRandomData(1024);
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderNotFound, contentStorage.WritePlaceHolder(i.placeHolderId, 0, dummyData.data(), 1024));
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderNotFound, contentStorage.Register(i.placeHolderId, i.contentId));
            bool has;
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_FALSE(has);
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentNotFound, contentStorage.Delete(i.contentId));

            NNT_EXPECT_RESULT_SUCCESS(contentStorage.CreatePlaceHolder(i.placeHolderId, i.contentId, i.size));
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.WritePlaceHolder(i.placeHolderId, 0, i.data.data(), i.size));
        }
        ShowDirectoryTree(GetRootPath());

        for (const auto& i : contentInfoList)
        {
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Register(i.placeHolderId, i.contentId));
            bool has;
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_TRUE(has);
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
            EXPECT_FALSE(has);

            nn::ncm::Path path;
            contentStorage.GetPath(&path, i.contentId);

            nn::fs::FileHandle handle;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path.string, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };
        }
        ShowDirectoryTree(GetRootPath());

        for (const auto& i : contentInfoList)
        {
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Delete(i.contentId));
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentNotFound, contentStorage.Delete(i.contentId));
        }
        ShowDirectoryTree(GetRootPath());

        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::CleanupBase(GetRootPath()));
    }
    );
}
#endif

TEST_F(ContentStorageTest, ListContentId)
{
    ExecuteAllContentDirectoryHierarchy(
        [&](nn::ncm::MakeContentPathFunction func, nn::ncm::MakePlaceHolderPathFunction placeHolderFunc)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::InitializeBase(GetRootPath()));

        auto contentStorageShared = ContentStorageFactory::CreateSharedEmplaced<nn::ncm::IContentStorage, nn::ncm::ContentStorageImpl>();
        NNT_EXPECT_RESULT_SUCCESS(ContentStorageFactory::GetEmplacedImplPointer<nn::ncm::ContentStorageImpl>(contentStorageShared)->Initialize(GetRootPath(), func, placeHolderFunc, false));
        nn::ncm::ContentStorage contentStorage(contentStorageShared);

        auto contentInfoList = MakeRandomContentInfoList(contentStorage, 10);

        for (const auto& i : contentInfoList)
        {
            auto dummyData = GetRandomData(1024);
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderNotFound, contentStorage.WritePlaceHolder(i.placeHolderId, 0, dummyData.data(), 1024));
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultPlaceHolderNotFound, contentStorage.Register(i.placeHolderId, i.contentId));
            bool has;
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_FALSE(has);
            NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentNotFound, contentStorage.Delete(i.contentId));

            NNT_EXPECT_RESULT_SUCCESS(contentStorage.CreatePlaceHolder(i.placeHolderId, i.contentId, i.size));
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.WritePlaceHolder(i.placeHolderId, 0, i.data.data(), i.size));
        }
        ShowDirectoryTree(GetRootPath());

        for (const auto& i : contentInfoList)
        {
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Register(i.placeHolderId, i.contentId));
            bool has;
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_TRUE(has);
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
            EXPECT_FALSE(has);

            NNT_EXPECT_RESULT_SUCCESS(contentStorage.RevertToPlaceHolder(i.placeHolderId, i.contentId, i.contentId));
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_FALSE(has);
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
            EXPECT_TRUE(has);

            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Register(i.placeHolderId, i.contentId));
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.Has(&has, i.contentId));
            EXPECT_TRUE(has);
            NNT_EXPECT_RESULT_SUCCESS(contentStorage.HasPlaceHolder(&has, i.placeHolderId));
            EXPECT_FALSE(has);

            nn::ncm::Path path;
            contentStorage.GetPath(&path, i.contentId);
            const auto pathBody = std::strchr(path.string, '/') + 1; // 先頭の "@Host:/" を無視する

            nn::fs::FileHandle handle;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, pathBody, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };
        }
        ShowDirectoryTree(GetRootPath());

        int count;
        NNT_EXPECT_RESULT_SUCCESS(
            contentStorage.GetContentCount(&count));

        EXPECT_EQ(10, count);

        count = 0;
        std::vector<nn::ncm::ContentId> contentIds(12);
        NNT_EXPECT_RESULT_SUCCESS(
            contentStorage.ListContentId(&count, &contentIds[0], 12, 0));

        EXPECT_EQ(10, count);
        contentIds.resize(count);

        std::vector<nn::ncm::ContentId> expectedIds(10);
        for(int i = 0; i < count; i++)
        {
            expectedIds[i] = contentInfoList[i].contentId;
        }
        std::sort(contentIds.begin(), contentIds.end(), [&](nn::ncm::ContentId a, nn::ncm::ContentId b) -> bool {
            return 0 < std::memcmp(a.data, b.data, 16);
        });
        std::sort(expectedIds.begin(), expectedIds.end(), [&](nn::ncm::ContentId a, nn::ncm::ContentId b) -> bool {
            return 0 < std::memcmp(a.data, b.data, 16);
        });

        for(int i = 0; i < count; i++)
        {
            EXPECT_EQ(expectedIds[i], contentIds[i]);
            NN_LOG("\n");
        }

        count = 0;
        contentIds.resize(12);
        NNT_EXPECT_RESULT_SUCCESS(
            contentStorage.ListContentId(&count, &contentIds[0], 12, 5));
        EXPECT_EQ(5, count);

        ShowDirectoryTree(GetRootPath());

        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ContentStorageImpl::CleanupBase(GetRootPath()));
    }
    );
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
namespace {
    struct Argument
    {
        bool isAlive;
        nn::Result result;
    };

    void ForegroundThread(void* pArgument) NN_NOEXCEPT
    {
        auto pThreadArgument = reinterpret_cast<Argument*>(pArgument);
        pThreadArgument->result = nn::ResultSuccess();

        nn::fs::SetPriorityOnCurrentThread(nn::fs::Priority_Normal);

        static const char* MountName = "test";
        static const char* TestFileName = "test:/test.bin";
        pThreadArgument->result = nn::fs::MountBis(MountName, nn::fs::BisPartitionId::User);
        if( pThreadArgument->result.IsFailure() )
        {
            return;
        }

        static const int64_t FileSize = 10 * 1024 * 1024;
        pThreadArgument->result = nnt::fs::util::DeleteFileOrDirectoryIfExists(TestFileName);
        if( pThreadArgument->result.IsFailure() )
        {
            return;
        }
        pThreadArgument->result = nn::fs::CreateFile(TestFileName, FileSize);
        if( pThreadArgument->result.IsFailure() )
        {
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::DeleteFile(TestFileName);
        };

        nn::fs::FileHandle file;
        pThreadArgument->result = nn::fs::OpenFile(&file, TestFileName, nn::fs::OpenMode_Read);
        if( pThreadArgument->result.IsFailure() )
        {
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        static char buffer[1024 * 1024];

        int64_t offset = 0;
        while( pThreadArgument->isAlive )
        {
            pThreadArgument->result = nn::fs::ReadFile(file, offset, buffer, sizeof(buffer));
            if( pThreadArgument->result.IsFailure() )
            {
                return;
            }

            offset += sizeof(buffer);
            if( offset + sizeof(buffer) > FileSize )
            {
                offset = 0;
            }
        }
    }
}

TEST_F(ContentStorageIntegrationTest, AccessPriority)
{
    static const int64_t ContentSize = 10 * 1024 * 1024;

    nn::ncm::ContentId contentId;
    {
        char mac[nn::crypto::HmacSha256Generator::MacSize];
        nn::crypto::GenerateHmacSha256Mac(
            mac, sizeof(mac),
            "testtesttest", 12,
            "testtesttest", 12
        );
        std::memcpy(&contentId, mac, sizeof(contentId));
    }

    nn::ncm::ContentStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuildInUser));

    auto placeHolder = storage.GeneratePlaceHolderId();
    NNT_ASSERT_RESULT_SUCCESS(storage.CreatePlaceHolder(placeHolder, contentId, ContentSize));
    NN_UTIL_SCOPE_EXIT
    {
        storage.DeletePlaceHolder(placeHolder);
    };

    nn::os::ThreadType thread;
    static char NN_OS_ALIGNAS_THREAD_STACK threadStack[16 * 1024];
    Argument argument = {};
    argument.isAlive = true;

    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
        &thread,
        ForegroundThread,
        &argument,
        threadStack,
        sizeof(threadStack),
        nn::os::DefaultThreadPriority,
        1
    ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::DestroyThread(&thread);
    };
    nn::os::StartThread(&thread);

    static const auto AccessTime = nn::TimeSpan::FromSeconds(1);

    auto doAccess = [&](int* outAccessCount, nn::fs::Priority priority) NN_NOEXCEPT -> nn::Result
    {
        int accessCount = 0;
        nn::fs::SetPriorityOnCurrentThread(priority);

        static char buffer[256 * 1024];
        int64_t offset = 0;

        auto tickStarted = nn::os::GetSystemTick();
        while( true )
        {
            auto tickCurrent = nn::os::GetSystemTick();
            NN_RESULT_DO(storage.WritePlaceHolder(placeHolder, offset, buffer, sizeof(buffer)));

            offset += sizeof(buffer);
            if( offset + sizeof(buffer) > ContentSize )
            {
                offset = 0;
            }

            ++accessCount;

            if( (tickCurrent - tickStarted).ToTimeSpan() >= AccessTime )
            {
                break;
            }
        }

        NN_RESULT_DO(storage.FlushPlaceHolder());
        *outAccessCount = accessCount;

        NN_RESULT_SUCCESS;
    };

    int accessCountRealtime = 0;
    auto resultRealtime = doAccess(&accessCountRealtime, nn::fs::Priority_Realtime);

    int accessCountLow = 0;
    auto resultLow = doAccess(&accessCountLow, nn::fs::Priority_Low);

    nn::fs::SetPriorityOnCurrentThread(nn::fs::Priority_Normal);

    argument.isAlive = false;
    nn::os::WaitThread(&thread);
    NNT_ASSERT_RESULT_SUCCESS(argument.result);
    NNT_ASSERT_RESULT_SUCCESS(resultLow);
    NNT_ASSERT_RESULT_SUCCESS(resultRealtime);

    // low 10, realtime 73 のように大きな差が出るはず
    NN_LOG("low %d, realtime %d\n", accessCountLow, accessCountRealtime);
    ASSERT_LT(accessCountLow, accessCountRealtime);
}
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(Allocate, Deallocate);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // BufferPool 初期化
    static const size_t BufferPoolSize = 1024 * 1024;
    static NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    static const size_t WorkBufferSize = nn::fssystem::BufferPoolWorkSize;
    static NN_ALIGNAS(8) char g_WorkBuffer[WorkBufferSize];
    nn::fssystem::InitializeBufferPool(g_BufferPool, BufferPoolSize, g_WorkBuffer, WorkBufferSize);
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
    nn::fat::FatFileSystem::SetAllocatorForFat(nnt::fs::util::GetTestLibraryAllocator());

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
