﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

namespace {

nn::os::ThreadType g_Thread;
nn::ncm::ApplyPatchDeltaTask::TaskState g_TaskStatus;
nn::Result g_ExecuteResult;
volatile bool g_TaskThreadExecuting;
int g_BufferSize;
int64_t g_ExpectedRequiredSizeMin;
int64_t g_ExpectedRequiredSizeMax;

nn::Result ExecuteInThread(nn::ncm::ApplyDeltaTaskBase* pTask) NN_NOEXCEPT
{
    const int StackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK static nn::Bit8 s_ThreadStack[StackSize];
    nn::os::ThreadFunction f = [](void* pTask) NN_NOEXCEPT
        {
            g_ExecuteResult = reinterpret_cast<nn::ncm::ApplyDeltaTaskBase*>(pTask)->Execute();
            g_TaskThreadExecuting = false;
        };
    NN_RESULT_DO(nn::os::CreateThread(&g_Thread, f, pTask, s_ThreadStack, StackSize, nn::os::DefaultThreadPriority));

    g_TaskThreadExecuting = true;
    nn::os::StartThread(&g_Thread);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   ApplyDeltaTaskTest で利用するテストフィクスチャです。
 */
class ApplyDeltaTaskTest : public ::testing::Test
{
protected:

    /**
    * @brief テスト開始時に毎回呼び出される関数です。
    */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Initialize();
        std::memset(&g_TaskStatus, 0, sizeof(g_TaskStatus));
    }

    /**
    * @brief テスト終了時に毎回呼び出される関数です。
    */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Finalize();
    }
};

}

TEST_F(ApplyDeltaTaskTest, ApplyBasic)
{
    static const nn::Bit64 SourceProgramId = 0x0005000c10000000;
    static const nn::Bit64 DeltaProgramId = 0x0005000c10000000;

    nn::ncm::ApplicationId SourceId;
    SourceId.value = SourceProgramId;

    nn::ncm::ApplicationId DeltaId;
    DeltaId.value = DeltaProgramId;

    nn::ncm::ContentMetaDatabase dbBuildInUser;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&dbBuildInUser, nn::ncm::StorageId::BuildInUser));

    nn::ncm::ContentMetaDatabase dbSd;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&dbSd, nn::ncm::StorageId::SdCard));

    nn::ncm::StorageContentMetaKey source;
    nn::ncm::StorageContentMetaKey delta;

    // 事前チェック。Patch と Delta が存在していることを確認。
    {
        nn::ncm::ListCount count;

        {
            source.storageId = nn::ncm::StorageId::BuildInUser;
            count = dbBuildInUser.ListContentMeta(&source.key, 1, nn::ncm::ContentMetaType::Patch, SourceId);
            if (count.listed == 0)
            {
                source.storageId = nn::ncm::StorageId::SdCard;
                count = dbSd.ListContentMeta(&source.key, 1, nn::ncm::ContentMetaType::Patch, SourceId);
                if (count.listed == 1)
                {
                    NN_SDK_LOG("Source storage is SdCard.\n");
                }
            }
            else
            {
                NN_SDK_LOG("Source storage is Built-In-User.\n");
            }
            ASSERT_EQ(1, count.listed);
        }

        {
            delta.storageId = nn::ncm::StorageId::BuildInUser;
            count = dbBuildInUser.ListContentMeta(&delta.key, 1, nn::ncm::ContentMetaType::Patch, SourceId, 0, 0xffffffffffffffff, nn::ncm::ContentInstallType::FragmentOnly);
            if (count.listed == 0)
            {
                delta.storageId = nn::ncm::StorageId::SdCard;
                count = dbSd.ListContentMeta(&delta.key, 1, nn::ncm::ContentMetaType::Patch, SourceId, 0, 0xffffffffffffffff, nn::ncm::ContentInstallType::FragmentOnly);
                if (count.listed == 1)
                {
                    NN_SDK_LOG("Delta storage is SdCard.\n");
                }
            }
            else
            {
                NN_SDK_LOG("Delta storage is Built-In-User.\n");
            }
            ASSERT_EQ(1, count.listed);
        }
    }

    // 適用できることを確認
    int64_t totalSize;
    nn::os::Tick tick = nn::os::GetSystemTick();
    NN_UNUSED(tick);
    {
        nn::ncm::ApplyPatchDeltaTask task;

        std::unique_ptr<nn::Bit8[]> buffer(new nn::Bit8[g_BufferSize]);
        ASSERT_NE(nullptr, buffer.get());
        NNT_ASSERT_RESULT_SUCCESS(task.SetBuffer(buffer.get(), g_BufferSize));

        NNT_ASSERT_RESULT_SUCCESS(task.Initialize(source, delta, &g_TaskStatus));

        auto getStorageSize = [&]() NN_NOEXCEPT
        {
            int64_t sizeFree = 0;

            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, source.storageId));
            NN_ABORT_UNLESS_RESULT_SUCCESS(storage.GetFreeSpaceSize(&sizeFree));
            NN_LOG("Storage free size: %lld\n", sizeFree);

            return sizeFree;
        };

        int64_t sizeFreeAtFirst = getStorageSize();;

        {
            auto result = task.Prepare();

            int64_t sizeRequiredForPrepare = 0;
            int64_t sizeRequiredForResume = 0;

            if( nn::ncm::ResultNotEnoughSpaceToApplyDelta::Includes(result)
                || result.IsSuccess() )
            {
                NNT_ASSERT_RESULT_SUCCESS(task.CalculateRequiredSizeMaxForPrepare(&sizeRequiredForPrepare));
                NNT_ASSERT_RESULT_SUCCESS(task.CalculateRequiredSizeMaxForResume(&sizeRequiredForResume));
            }

            if( nn::ncm::ResultNotEnoughSpaceToApplyDelta::Includes(result) )
            {
                NN_LOG(
                    "Not enough space error detected on preparing. (%lld + %lld = %lld required)\n",
                    sizeRequiredForPrepare - sizeRequiredForResume,
                    sizeRequiredForResume,
                    sizeRequiredForPrepare
                );
                ASSERT_LE(g_ExpectedRequiredSizeMin, sizeRequiredForPrepare);
                ASSERT_GE(g_ExpectedRequiredSizeMax, sizeRequiredForPrepare);
                return;
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            int64_t sizeFree = getStorageSize();
            NN_LOG(
                "size required for prepare: %lld (%lld required)\n",
                sizeFreeAtFirst - sizeFree,
                sizeRequiredForPrepare - sizeRequiredForResume
            );
            ASSERT_LE(sizeFreeAtFirst - sizeFree, sizeRequiredForPrepare - sizeRequiredForResume);
        }
        NNT_ASSERT_RESULT_SUCCESS(ExecuteInThread(&task));
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::WaitThread(&g_Thread);
            nn::os::DestroyThread(&g_Thread);
        };

        auto progress = task.GetProgress();
        while(progress.state != nn::ncm::ApplyDeltaProgressState_DeltaApplied && g_TaskThreadExecuting)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            progress = task.GetProgress();
            totalSize = progress.totalDeltaSize;
            NN_SDK_LOG("[Progress] %d %lld / %lld\n", progress.state, progress.appliedDeltaSize, progress.totalDeltaSize);
        }
        if( nn::ncm::ResultNotEnoughSpaceToApplyDelta::Includes(g_ExecuteResult) )
        {
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(task.CalculateRequiredSizeMaxForResume(&size));
            NN_LOG("Not enough space error detected on applying. (%lld required)\n", size);
            ASSERT_LE(g_ExpectedRequiredSizeMin, size);
            ASSERT_GE(g_ExpectedRequiredSizeMax, size);
            return;
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(g_ExecuteResult);
            ASSERT_EQ(-1, g_ExpectedRequiredSizeMin);
            ASSERT_EQ(-1, g_ExpectedRequiredSizeMax);
        }
        NNT_ASSERT_RESULT_SUCCESS(task.Commit());

        auto timeSeconds = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetSeconds();
        NN_UNUSED(timeSeconds);
        NN_SDK_LOG("[Result] total size: %lld byte, time %lld s, buffer size %d byte, speed %lld KB/s\n",
            totalSize,
            timeSeconds,
            g_BufferSize,
            totalSize / 1024 / timeSeconds);
    }

    // 適用後に Patch が存在していることを確認
    {
        nn::ncm::StorageContentMetaKey keyList;
        nn::ncm::ListCount count;

        keyList.storageId = source.storageId;

        if (source.storageId == nn::ncm::StorageId::BuildInUser)
        {
            count = dbBuildInUser.ListContentMeta(&keyList.key, 1, nn::ncm::ContentMetaType::Patch, SourceId);
        }
        else
        {
            count = dbSd.ListContentMeta(&keyList.key, 1, nn::ncm::ContentMetaType::Patch, SourceId);
        }
        ASSERT_EQ(1, count.listed);

        nn::ns::Initialize();
        NN_UTIL_SCOPE_EXIT{nn::ns::Finalize();};

        NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushApplicationRecord(SourceId, nn::ns::ApplicationEvent::LocalInstalled, &keyList, 1));
    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static const size_t HeapSize = 32 * 1024 * 1024;
    void* pHeapStack = ::malloc(HeapSize);

    nnt::fs::util::InitializeTestLibraryHeap(pHeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);

    g_BufferSize = 256 * 1024;
    g_ExpectedRequiredSizeMin = -1;
    g_ExpectedRequiredSizeMax = -1;

    for( int argumentIndex = 1; argumentIndex < argc; ++argumentIndex )
    {
        auto pCurrentArgument = argv[argumentIndex];
        if( std::strcmp(pCurrentArgument, "--buffer-size") == 0 )
        {
            ++argumentIndex;
            if( argumentIndex >= argc )
            {
                NN_LOG("buffer size not specified\n");
                nnt::Exit(1);
            }

            g_BufferSize = std::atoi(argv[argumentIndex]);
        }
        else if( std::strcmp(pCurrentArgument, "--expected-required-size") == 0 )
        {
            ++argumentIndex;
            if( argumentIndex + 1 >= argc )
            {
                NN_LOG("expected required size not specified\n");
                nnt::Exit(1);
            }

            g_ExpectedRequiredSizeMin = std::strtoll(argv[argumentIndex + 0], nullptr, 10);
            g_ExpectedRequiredSizeMax = std::strtoll(argv[argumentIndex + 1], nullptr, 10);
            NN_LOG("expected required size set to %lld - %lld\n", g_ExpectedRequiredSizeMin, g_ExpectedRequiredSizeMax);

            argumentIndex += (2 - 1);
        }
    }

    int result = RUN_ALL_TESTS();

    ::free(pHeapStack);

    nnt::Exit(result);
}
