﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <limits>
#include <cstdint>
#include <cinttypes>
#include <vector>
#include <string>
#include <array>
#include <utility>
#include <algorithm>

#include <movie/Utils.h>
#include <nn/nn_SdkLog.h>

namespace {

    struct GatherSomeValues : movie::MediaData::Visitor {

        mutable std::pair<std::string,int32_t>                  i32;
        mutable std::pair<std::string,int64_t>                  i64;
        mutable std::pair<std::string,size_t>                    sz;
        mutable std::pair<std::string,float>                    flt;
        mutable std::pair<std::string,double>                   dbl;
        mutable std::pair<std::string,void*>               void_ptr;
        mutable std::pair<std::string,const char*>         char_ptr;
        mutable std::pair<std::string,movie::Buffer const*> buf_ptr;
        mutable std::pair<std::string,std::array<int32_t,4>>   rect;

        void accept(const char* name_, int32_t i) const noexcept override {
            i32.first = name_;
            i32.second = i;
        }
        void accept(const char* name_, int64_t i) const noexcept override {
            i64.first = name_;
            i64.second = i;
        }
        void accept(const char* name_, size_t i) const noexcept override {
            sz.first = name_;
            sz.second = i;
        }
        void accept(const char* name_, float f) const noexcept override {
            flt.first = name_;
            flt.second = f;
        }
        void accept(const char* name_, double d) const noexcept override {
            dbl.first = name_;
            dbl.second = d;
        }
        void accept(const char* name_, void* ptr) const noexcept override {
            void_ptr.first = name_;
            void_ptr.second = ptr;
        }
        void accept(const char* name_, const char* c) const noexcept override {
            char_ptr.first = name_;
            char_ptr.second = c;
        }
        void accept(const char* name_, movie::Buffer const& b) const noexcept override {
            buf_ptr.first = name_;
            buf_ptr.second = &b;
        }
        void accept(const char* name_, int32_t left, int32_t top, int32_t right, int32_t bottom) const noexcept override {
            rect.first = name_;
            rect.second[0] = left;
            rect.second[1] = top;
            rect.second[2] = right;
            rect.second[3] = bottom;
        }
    };
}


TEST(MediaData_test,Visitor) {

    movie::MediaData media_data_;

    std::string string_{"Just a test string."};
    std::vector<char> buffer_backing___{std::begin(string_),
                                            std::end(string_)};
    movie::Buffer buffer_{buffer_backing___.data(),
                            buffer_backing___.size()};
    // Store some things
    media_data_.SetSize   ("test_size",   std::numeric_limits<size_t>::max());
    media_data_.SetInt32  ("test_int32",  std::numeric_limits<int32_t>::max());
    media_data_.SetInt64  ("test_int64",  std::numeric_limits<int64_t>::max());
    media_data_.SetFloat  ("test_float",  std::numeric_limits<float>::max());
    media_data_.SetDouble ("test_double", std::numeric_limits<double>::max());
    media_data_.SetPointer("test_ptr",    reinterpret_cast<void*>(&string_)); // point to something..
    media_data_.SetString ("test_string", string_.c_str());
    media_data_.SetBuffer ("test_buffer", &buffer_);
    media_data_.SetRect   ("test_rect",   -1,-2,3,4);

    // Let's collect what we've just stored
    int32_t i32;
    int64_t i64;
    size_t  sz;
    float   f;
    double  d;
    void*   void_ptr;
    movie::Buffer* buf_ptr;
    std::array<int32_t,4> rect;
    const char* char_ptr;

    ASSERT_TRUE(media_data_.FindSize   ("test_size",   &sz));
    ASSERT_TRUE(media_data_.FindInt32  ("test_int32",  &i32));
    ASSERT_TRUE(media_data_.FindInt64  ("test_int64",  &i64));
    ASSERT_TRUE(media_data_.FindFloat  ("test_float",  &f));
    ASSERT_TRUE(media_data_.FindDouble ("test_double", &d));
    ASSERT_TRUE(media_data_.FindPointer("test_ptr",    &void_ptr));
    ASSERT_TRUE(media_data_.FindString ("test_string", &char_ptr));
    ASSERT_TRUE(media_data_.FindBuffer ("test_buffer", &buf_ptr));
    ASSERT_TRUE(media_data_.FindRect   ("test_rect",   &rect[0], &rect[1], &rect[2], &rect[3]));
    //
    ASSERT_FALSE(media_data_.FindInt32("not_included", &i32));
    ASSERT_FALSE(media_data_.FindInt64("not_included", &i64));
    //
    ASSERT_EQ(sz,  std::numeric_limits<size_t>::max());
    ASSERT_EQ(i32, std::numeric_limits<int32_t>::max());
    ASSERT_EQ(i64, std::numeric_limits<int64_t>::max());
    ASSERT_EQ(f,   std::numeric_limits<float>::max());
    ASSERT_EQ(d,   std::numeric_limits<double>::max());

    ASSERT_EQ(void_ptr, reinterpret_cast<void*>(&string_)); // our earlier target
    ASSERT_EQ(char_ptr, string_);

    ASSERT_TRUE((rect == std::array<int32_t,4>{-1,-2,3,4}));
    ASSERT_TRUE(std::equal(buffer_.Data(),  buffer_.Data() + buffer_.Size(),  // Confirm the buffer data matches
                           buf_ptr->Data(), buf_ptr->Data() + buf_ptr->Size()));

    GatherSomeValues gather_;
    media_data_.Visit(gather_);

    // Confirm the names are the same.
    ASSERT_EQ(gather_.i32.first,      "test_int32");
    ASSERT_EQ(gather_.i64.first,      "test_int64");
    ASSERT_EQ(gather_.sz.first,       "test_size");
    ASSERT_EQ(gather_.flt.first,      "test_float");
    ASSERT_EQ(gather_.dbl.first,      "test_double");
    ASSERT_EQ(gather_.void_ptr.first, "test_ptr");
    ASSERT_EQ(gather_.char_ptr.first, "test_string");
    ASSERT_EQ(gather_.buf_ptr.first,  "test_buffer");
    ASSERT_EQ(gather_.rect.first,     "test_rect");

    // Confirm the values match what we got through the FindXX APIs
    ASSERT_EQ(gather_.i32.second,      i32);
    ASSERT_EQ(gather_.i64.second,      i64);
    ASSERT_EQ(gather_.sz.second,       sz);
    ASSERT_EQ(gather_.flt.second,      f);
    ASSERT_EQ(gather_.dbl.second,      d);
    ASSERT_EQ(gather_.void_ptr.second, void_ptr);
    ASSERT_EQ(gather_.char_ptr.second, char_ptr);
    ASSERT_EQ(gather_.buf_ptr.second,  buf_ptr);
    ASSERT_EQ(gather_.rect.second,     rect);
}


