﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define LOG_TAG "ScoopVideo"
#define ATRACE_TAG ATRACE_TAG_GRAPHICS

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nnt/nntest.h>

#include <cerrno>
#include <cstdlib>
#include <cstdio>
#include <fcntl.h>
#include <cstring>
#include <map>

#define DRV
#define ENABLE_AUDIO_RECORD

#include <nn/lmem/lmem_ExpHeap.h>
#include <binder/Parcel.h>
#include <utils/Log.h>
#include <utils/Errors.h>
#include <utils/Timers.h>
#include <utils/Trace.h>
#include <gui/Surface.h>
#include <gui/SurfaceComposerClient.h>
#include <gui/ISurfaceComposer.h>
#include <ui/DisplayInfo.h>
#include <media/openmax/OMX_IVCommon.h>
#include <media/stagefright/foundation/ABuffer.h>
#include <media/stagefright/foundation/AMessage.h>
#include <media/stagefright/MediaCodec.h>
#include <media/stagefright/MediaErrors.h>
#include <media/stagefright/MediaMuxer.h>
#include <media/ICrypto.h>
#include <SfNvnUtil.h>

#include <nn/nn_Log.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <mm_MemoryManagement.h>
#include <mm_Threads.h>

#include "NvMediaRecorder.h"
#include "NvAudioEncoder.h"
#include "NvVideoEncoder.h"
#include "../common/HeapTracker.h"

#define Now() ns2ms( systemTime() )

#define CHECK_MEMORY_LEAK

using namespace android;

const int HOURS_IN_SECS = 60 * 60;
const int MAX_HOURS = 4 * HOURS_IN_SECS;
uint32_t gFrameRate = 30;
uint32_t gNumFrameRecord;
const int NUMFRAMECAPTURE = 200;
bool gVideoDone = false;
bool gAudioDone = false;
Mutex mLock;
Condition mCondition;

static const void *sRefBaseOwner;

#ifdef DRV
NvMediaRecorder *mNvMediaRecorder;
#ifdef ENABLE_AUDIO_RECORD
NvAudioEncoder *mAudioEncoder;
#endif
NvVideoEncoder *mVideoEncoder;
#endif


//#ifdef RAPTOR_ENHANCEMENT
extern void initializeLooperRoster();
extern void initializeAtomizer();
extern void initializeDataSource();
extern void shutdownLooperRoster();
extern void shutdownAtomizer();
extern void shutdownDataSource();
//#endif

//#define ENABLE_LOG

#ifdef ENABLE_LOG
    #define NN_LOG_V NN_LOG
#else
    #define NN_LOG_V(format, args...) ((void)0)
#endif

#define NN_LOG_E NN_LOG

enum {
    SV_VIDEO_END = 1,
    SV_AUDIO_END = 2
};

namespace{

    const int FsHeapSize = 512 * 1024;
    const int mmHeapSize = 256 << 20;
    const int CoreHeapSize = 8 << 20;
    const int mmFirmwareMemorySize = 8 << 20;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;
    char                        g_mmHeapBuffer[mmHeapSize];
    nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    char                        g_CoreHeapBuffer[CoreHeapSize];
    nn::mem::StandardAllocator  g_CoreAllocator(g_CoreHeapBuffer, sizeof(g_CoreHeapBuffer));
    char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));
    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }
static nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    return g_MultimediaAllocator;
}

static HeapTracker &MMHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &CoreHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &ThreadHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

void *mmAllocate(size_t size, size_t align, void *userPtr)
{
    if(size == 0)
        return NULL;
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    void *address = GlobalAllocator().Allocate(size, align);
    static size_t totalAllocatedSize = 0;

    totalAllocatedSize += size;
    static_cast<HeapTracker*>(userPtr)->Track(address, size);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return address;
}

void mmDeallocate(void *address, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();

    GlobalAllocator().Free(address);
    static_cast<HeapTracker*>(userPtr)->Untrack(address);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
}

void *mmReallocate(void* address, size_t newSize, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    if (address)
       static_cast<HeapTracker*>(userPtr)->Untrack(address);
    void *memory = GlobalAllocator().Reallocate(address, newSize);
    static_cast<HeapTracker*>(userPtr)->Track(memory, newSize);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return memory;
}
}
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    const size_t mallocSize = 32 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    NN_SDK_LOG("\ninit done: \n");
}

void EventCallback(uint32_t event) {
    Mutex::Autolock autoLock(mLock);
    status_t err;
    switch (event) {
        case SV_VIDEO_END:
            //Video ends
#ifdef ENABLE_AUDIO_RECORD
            NN_LOG_E("Stop event received from video !!!!!!!!!!!!!!!!!\n");
            gVideoDone = true;
            NN_LOG_E("ScoopVideo:: calling stop of audio encoder\n");
            mAudioEncoder->stop();
#else
            err = mNvMediaRecorder->writeData();
            if (err != NO_ERROR) {
                NN_LOG_E("ScoopAudio:: Failed writing data to video recorder (err=%d)\n", err);
            }
            NN_LOG_V("ScoopVideo:: File write complete\n");
            gVideoDone = true;
#endif
            mVideoEncoder->stop();

            /*if (gAudioDone && gVideoDone) {
                err = mNvMediaRecorder->writeData();
                if (err != NO_ERROR) {
                    NN_LOG_E("ScoopVideo:: Failed writing data to video recorder (err=%d)\n", err);
                    //return false;
                }
                //mCondition.signal();
            }*/
            break;
        case SV_AUDIO_END:
            //Audio ends
            NN_LOG_E("Stop event received from audio !!!!!!!!!!!!!!!!!\n");

            if (gVideoDone) {
                err = mNvMediaRecorder->writeData();
                if (err != NO_ERROR) {
                    NN_LOG_E("ScoopAudio:: Failed writing data to video recorder (err=%d)\n", err);
                    //return false;
                }
                NN_LOG_V("ScoopVideo:: File write complete\n");

                //mCondition.signal();
            }
            gAudioDone = true;
            break;
        default:
            break;
    }
}

void setVirtualDisplayPolicy(const char *vdpyPolicy)
{
    sp<SurfaceComposerClient> client = new SurfaceComposerClient();
    sp<IBinder> binder = client->connection();
    Parcel data;
    data.writeInterfaceToken(binder->getInterfaceDescriptor());
    data.writeCString(vdpyPolicy);
    binder->transact(1009, data, NULL);
}

int ScoopVideo_test() {
    status_t err = NO_ERROR;
    fnEvent mFunEvent;
    mFunEvent = EventCallback;

    NN_LOG_V("Entering main thread\n");
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    initializeLooperRoster();
    initializeAtomizer();
    initializeDataSource();
    if (argc >= 2) {
        gNumFrameRecord = atoi(argv[1]);
        if(gNumFrameRecord <= 0 || gNumFrameRecord > (MAX_HOURS *gFrameRate))
            gNumFrameRecord = NUMFRAMECAPTURE;
    }

    const char *dispPolicy = "30hz";
    setVirtualDisplayPolicy(dispPolicy);
    NN_LOG_V("Setting display policy to 30 Hz\n");

    mNvMediaRecorder = new NvMediaRecorder();
    mNvMediaRecorder->setIFrameInterval(gFrameRate);
    mNvMediaRecorder->init();

    NN_LOG_V("ScoopVideo:: Creating video thread\n");
    mVideoEncoder = new NvVideoEncoder(1280,720,gFrameRate,gNumFrameRecord);
    mVideoEncoder->start(mNvMediaRecorder, mFunEvent);

#ifdef ENABLE_AUDIO_RECORD
    // thread looper
    NN_LOG_V("ScoopAudio:: Creating audio thread\n");

    mAudioEncoder = new NvAudioEncoder();
    mAudioEncoder->start(mNvMediaRecorder, mFunEvent);
#endif

    NN_LOG_E("ScoopVideo:: check for completion\n");
    //mCondition.wait(mLock);
#ifdef ENABLE_AUDIO_RECORD
    while (!gAudioDone) {
        if (gVideoDone) {
           usleep(1000);
        } else {
           usleep(100000);
       }
    }
#else
    while (!gVideoDone) {
        usleep(1000);
    }
#endif

    NN_LOG_E("ScoopVideo:: calling stop of video encoder...\n");
    mVideoEncoder->stop();

    delete mVideoEncoder;
#ifdef ENABLE_AUDIO_RECORD
    delete mAudioEncoder;
#endif
    delete mNvMediaRecorder;
    shutdownLooperRoster();
    shutdownAtomizer();
    shutdownDataSource();
    return 0;
}
extern "C" int nnMain(){
     /* Set allocator callback functions */
    nv::mm::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    nv::mm::SetThreadStackAllocator(mmAllocate, mmDeallocate, &ThreadHeap());
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsServiceName("nvdrv:s");
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));

    ScoopVideo_test();

    nv::FinalizeGraphics();

#ifdef CHECK_MEMORY_LEAK
    NN_LOG("[MM HEAP]\n");
    MMHeap().OutputUsage();
    NN_LOG("[CORE HEAP]\n");
    CoreHeap().OutputUsage();

    NN_LOG("[THREAD HEAP]\n");
    ThreadHeap().OutputUsage();
#endif

    return 0;
}

//HACK: need to find out if there's an __assert2 and __errno somewhere

volatile int*   __errno()
{
    static int i = 0;
    return &i;
}
