﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#define LOG_TAG "NvMediaRecorder"
#include "NvMediaRecorder.h"
//#define LOG_NDEBUG = 0;
#define ENABLE_AUDIO_RECORD

const size_t HEAP_SIZE_VIDEO  = 27 * 1024 * 1024 ; //27 MB
const size_t HEAP_SIZE_AUDIO  = 2 * 1024 * 1024 ; //5 MB
const size_t MAX_INPUT_BUFFER_SIZE  = 200000; //460800 // (1280 x 720)/2

NvMediaRecorder::NvMediaRecorder() {
    mIFrameInterval = 5;
    mStartIndex = -1;
    mEndIndex = 0;
    mFrameIndex = 0;
    mMediaMuxer = NULL;
    mVideotimestamp  = 0;
    mFirstFrame = true;
    mMemHandleVideo = NULL;
    mMemHandleAudio = NULL;
    mAudioFrameCount = 0;
    //mAudiotimestamp = 0;
    mAudioTrackIdx = -1;
    mVideoTrackIdx = -1;
    mStartIndexInc = false;
}

NvMediaRecorder::~NvMediaRecorder() {
    NN_LOG_V("NvMediaRecorder instance destroying\n\n");
    status_t err;

    NN_LOG_V("ScoopVideo:: video chunks in file mStartIndex - %d vector size - %d\n",mStartIndex, mVideoChunk.size());
    while (mVideoChunk.size() != 0) {
        videoChunk *chunk = mVideoChunk.editItemAt(0);
        while (chunk->videoframes.size() != 0) {
            FrameData *videoData = chunk->videoframes.editItemAt(0);
            videoData->aBuf.clear();
            chunk->videoframes.erase(chunk->videoframes.begin());
        }
        delete chunk;
        mVideoChunk.erase(mVideoChunk.begin());
    }

    while (mAudioChunk.size() != 0) {
         audioChunk *chunk = mAudioChunk.editItemAt(0);
         delete chunk->audioframes;
         //delete chunk;
         mAudioChunk.erase(mAudioChunk.begin());
    }
    NN_LOG_V("ScoopVideo:: delete temp memory");
    delete mMemAllocatorVideo;
    delete mMemAllocatorAudio;
    NN_LOG_V("ScoopVideo:: delete recorder instance");
}

status_t NvMediaRecorder::initVideo() {
    mMemAllocatorVideo = new MemoryAllocator();

    mMemHandleVideo = mMemAllocatorVideo->init(HEAP_SIZE_VIDEO);
    if (mMemHandleVideo == NULL) {
        NN_LOG_E("ScoopVideo:: NvMediaRecorder : Memory allocation fails...");
    }
    NN_LOG_V("NvMediaRecorder :: mMemHandle - %p",mMemHandleVideo);
    for (int i = 0; i < MAX_COUNT; i++) {
        videoChunk *chunk = new videoChunk();

        for (int j = 0; j < mIFrameInterval; j++) {
            FrameData *mVideoData = new FrameData(); //instead use malloc
            //mVideoData->aBuf = new ABuffer(MAX_INPUT_BUFFER_SIZE);
            //mVideoData->aBuf->setRange(0, 0);
            mVideoData->flags = 0;
            mVideoData->timestamp = -1;
            mVideoData->offset = 0;
            mVideoData->size = 0;
            mVideoData->state = FRAME_INITIALIZE;
            chunk->videoframes.push(mVideoData);
            NN_LOG_V("ScoopVideo:: %d video frame allocated with size - %d\n", i, chunk->videoframes.size());
        }
        chunk->state = MEDIA_CHUNK_INIT;
        mVideoChunk.push(chunk);
        NN_LOG_V("ScoopVideo:: %d video chunk allocated with size - %d\n", i, mVideoChunk.size());
    }
    mStartIndex = 0;
    return NO_ERROR;
}

status_t NvMediaRecorder::initAudio() {
    mMemAllocatorAudio = new MemoryAllocator();

    mMemHandleAudio = mMemAllocatorAudio->init(HEAP_SIZE_AUDIO);
    if (mMemHandleAudio == NULL) {
        NN_LOG_E("ScoopAudio:: NvMediaRecorder : initAudio: Memory allocation fails...");
    }
    NN_LOG_V("ScoopAudio:: NvMediaRecorder :: mMemHandleAudio - %p",mMemHandleAudio);
#if 0
    for (int i = 0; i < MAX_COUNT; i++) {
        audioChunk *chunk = new audioChunk();

        for (int j = 0; j < 47; j++) { //(47 = 48000/1024)
            FrameData *mAudioData = new FrameData(); //instead use malloc
            //mAudioData->aBuf = new ABuffer(MAX_INPUT_BUFFER_SIZE);
            //mAudioData->aBuf->setRange(0, 0);
            mAudioData->flags = 0;
            mAudioData->timestamp = -1;
            mAudioData->offset = 0;
            mAudioData->size = 0;
            mAudioData->state = FRAME_INITIALIZE;
            chunk->audioframes.push(mAudioData);
            NN_LOG_V("ScoopAudio:: %d audio frame allocated with size - %d\n", i, chunk->videoframes.size());
        }
        chunk->state = MEDIA_CHUNK_INIT;
        mAudioChunk.push(chunk);
        NN_LOG_V("ScoopAudio:: %d audio chunk allocated with size - %d\n", i, mVideoChunk.size());
    }
    //mStartIndex = 0;
#endif
    return NO_ERROR;
}


status_t NvMediaRecorder::init() {
    NN_LOG_V("NvMediaRecorder::init In mIFrameInterval - %d\n",mIFrameInterval);
    if (initVideo() != NO_ERROR) {
        NN_LOG_E("NvMediaRecorder::initVideo failed !!!");
    }

    if (initAudio() != NO_ERROR) {
        NN_LOG_E("NvMediaRecorder::initAudio failed !!!");
    }

    // Initialise muxer
    char fileName[] = "c:/temp/VideoRec_01_DVR.mp4";
    mMediaMuxer = new MediaMuxer(fileName, MediaMuxer::OUTPUT_FORMAT_MPEG_4);
    NN_LOG_V("VideoRecoder::MediaMuxer set to MP4\n\n");
    return NO_ERROR;
}

status_t NvMediaRecorder::feedVideoData(sp<ABuffer> abuf, int64_t ts, uint32_t flags) {
    NN_LOG_V("ScoopVideo:: feed video data with ts - %lld\n",ts);

    NN_LOG_V("ScoopVideo:: feed data get the chunk mEndIndex - %d\n",mEndIndex);
    videoChunk *chunk = mVideoChunk.editItemAt(mEndIndex);
    NN_LOG_V("ScoopVideo:: feed data get the chunk success mFrameIndex - %d\n",mFrameIndex);
    FrameData *mVideoData = chunk->videoframes.editItemAt(mFrameIndex);
    NN_LOG_V("ScoopVideo:: feed data complete..");

    if (abuf != NULL && mVideoData != NULL) {
#if 0
        if (mFirstFrame) {
            NN_LOG_V("ScoopVideo:: Its first frame of size - %d\n", abuf->size());
            memcpy(mConfigData->data(), abuf->data(), abuf->size());
            mConfigData->setRange(0, abuf->size());
            status_t err = mMediaMuxer->writeSampleData(mConfigData, mVideoTrackIdx,
                           /*ptsUsec*/ 0, flags);
             if (err != NO_ERROR) {
                NN_LOG_V("ScoopVideo:: Failed writing config data to muxer (err=%d)\n", err);
                return err;
             }
            mFirstFrame = false;
        }
#endif
        chunk->state = MEDIA_CHUNK_PROCESSING;
        uint8_t *memPtr;
        int32_t status;
        uint64_t offset;
        bool repeat;

        do {
        repeat = false;
        NN_LOG_V("ScoopVideo:: get memory\n");
        memPtr = mMemAllocatorVideo->getMemory(abuf->size(), &offset, &status);
        if (memPtr == NULL) {
            NN_LOG_E("ScoopVideo:: get memory failed !!!!!  Please stop processing\n");
            //TODO: return error
        } else if (status == MemoryAllocator::MEMORY_FULL) {
            NN_LOG_E("ScoopVideo:: Memory full return will clear mStartIndex - %d\n",mStartIndex);
            int32_t freeSize = 0;
            uint8_t *basePtr = NULL;
            // free the oldest sec data in queue
            videoChunk *cleanChunk = mVideoChunk.editItemAt(mStartIndex);
            for (int j = 0; j < mIFrameInterval; j++) {
                FrameData *videoData = cleanChunk->videoframes.editItemAt(j);
                freeSize += videoData->size;
                NN_LOG_V("ScoopVideo:: freeing mStartIndex - %d frame no - %d videoData->size - %lld freeSize - %d\n",mStartIndex, j, videoData->size, freeSize);
                videoData->state = FRAME_EMPTY;
                if (j == 0) {
                    basePtr = (uint8_t *)videoData->aBuf->data();
                }
            }
            cleanChunk->state = MEDIA_CHUNK_INIT;
            if (mStartIndex >= (MAX_COUNT - 1)){
                mStartIndex = 0;
            } else {
                mStartIndex++;
                mStartIndexInc = true;
            }
            //NN_LOG_V("freeing mStartIndex - %d memset", mStartIndex);
            //memset(basePtr,0 ,freeSize); // TODO: remove memset
            //NN_LOG_V("memset removed");
            mMemAllocatorVideo->freeMemory(freeSize);
            repeat = true;
            NN_LOG_V("ScoopVideo:: try to allocate memory again\n");
        }
        }while (repeat);

        NN_LOG_V("ScoopVideo:: creating ABuffer with memHandle - %p and offset - %p abuf->size() - %d\n",mMemHandleVideo, memPtr, abuf->size());
        mVideoData->aBuf = new ABuffer((void *)memPtr, abuf->size());
        NN_LOG_V("ScoopVideo:: memcpy abuffer data of size - %d\n", abuf->size());
        memcpy(mVideoData->aBuf->data(), abuf->data(), abuf->size());
        mVideoData->aBuf->setRange(0, abuf->size());
        NN_LOG_V("ScoopVideo:: memcpy completed..\n");

        mVideoData->flags = flags;
        mVideoData->timestamp = ts;
        mVideoData->offset = offset;
        mVideoData->size = abuf->size();
        mVideoData->state = FRAME_FILLED;

        mFrameIndex++;
    }

    if (mFrameIndex >= mIFrameInterval) {
        if (mEndIndex < (MAX_COUNT - 1)) {
            NN_LOG_V("ScoopVideo:: start new video chunk\n");
            mEndIndex++;
            if (mStartIndex == mEndIndex) {
                if (mStartIndex == (MAX_COUNT - 1)) {
                    mStartIndex = 0;
                } else {
                    mStartIndex++;
                }
                NN_LOG_V("mStartIndex - %d mEndIndex - %d\n",mStartIndex, mEndIndex);
            }
        } else {
            if (mStartIndex == mEndIndex) {
                NN_LOG_V("ScoopVideo:: start index and end index are same - %d\n\n",mStartIndex);
                mStartIndex = 0;
            } else {
                if (mStartIndexInc == false) {
                    mStartIndex++;
                }
                mStartIndexInc = false;
            }
            mEndIndex = 0;
        }
        NN_LOG_V("ScoopVideo:: mStartIndex - %d mEndIndex - %d\n",mStartIndex, mEndIndex);
        NN_LOG_E("ScoopVideo:: start new frame\n");
        chunk->state = MEDIA_CHUNK_COMPLETE;
        mFrameIndex = 0;

        videoChunk *cleanChunk = mVideoChunk.editItemAt(mEndIndex);
        for (int j = 0; j < mIFrameInterval; j++) {
            FrameData *videoData = cleanChunk->videoframes.editItemAt(j);
            videoData->aBuf.clear();
            videoData->state = FRAME_EMPTY;
            videoData->flags = 0;
            videoData->timestamp = -1;
            videoData->offset = 0;
            videoData->size = 0;
        }
    }
    NN_LOG_V("ScoopVideo:: feed data returns\n");
    return NO_ERROR;
}//NOLINT(impl/function_size)

// buffer audio data
status_t NvMediaRecorder::feedAudioData(sp<ABuffer> abuf, int64_t ts, uint32_t flags) {
    NN_LOG_V("ScoopAudio:: feed audio data with ts - %lld\n",ts);

    //NN_LOG_V("feed data get the chunk mEndIndex - %d\n",mEndIndex);
    //audioChunk *chunk = mAudioChunk.editItemAt(mEndIndex);
    //NN_LOG_V("feed data get the chunk success mFrameIndex - %d\n",mFrameIndex);
    //FrameData *mVideoData = chunk->videoframes.editItemAt(mFrameIndex);

    audioChunk *chunk = new audioChunk();
    FrameData *mAudioData = new FrameData(); //instead use malloc
    mAudioData->flags = 0;
    mAudioData->timestamp = -1;
    mAudioData->offset = 0;
    mAudioData->size = 0;
    mAudioData->state = FRAME_INITIALIZE;
    chunk->audioframes = mAudioData;

    if (abuf != NULL && mAudioData != NULL) {

        mAudioData->state = MEDIA_CHUNK_PROCESSING;
        uint8_t *memPtr;
        int32_t status;
        uint64_t offset;
        bool repeat;

        do {
        repeat = false;
        memPtr = mMemAllocatorAudio->getMemory(abuf->size(), &offset, &status);
        if (memPtr == NULL) {
            NN_LOG_E("ScoopAudio:: get memory failed !!!!!  Please stop processing\n");
            //TODO: return error
        } else if (status == MemoryAllocator::MEMORY_FULL) {
            NN_LOG_E("ScoopAudio:: Memory full return will clear mStartIndex - %d\n",mStartIndex);
            //int32_t freeSize = 0;
            //uint8_t *basePtr = NULL;
            // free the oldest data in queue
            audioChunk *cleanChunk = mAudioChunk.editItemAt(0);
            FrameData *audioData = cleanChunk->audioframes;
            //audioData->state = FRAME_EMPTY;
            //cleanChunk->state = MEDIA_CHUNK_INIT;

            mMemAllocatorAudio->freeMemory(audioData->size);
            repeat = true;
            delete audioData;
            mAudioChunk.removeAt(0);
            NN_LOG_V("ScoopAudio:: try to allocate memory again\n");
        }
        }while (repeat);

        NN_LOG_V("ScoopAudio:: audio creating ABuffer with memHandle - %p and offset - %p abuf->size() - %d\n",mMemAllocatorAudio, memPtr, abuf->size());
        mAudioData->aBuf = new ABuffer((void *)memPtr, abuf->size());
        NN_LOG_V("ScoopAudio:: memcpy abuffer data of size - %d\n", abuf->size());
        memcpy(mAudioData->aBuf->data(), abuf->data(), abuf->size());
        mAudioData->aBuf->setRange(0, abuf->size());
        NN_LOG_V("ScoopAudio:: memcpy completed..\n");

        mAudioData->flags = flags;
        mAudioData->timestamp = ts;
        mAudioData->offset = offset;
        mAudioData->size = abuf->size();
        mAudioData->state = FRAME_FILLED;
        chunk->sequence_no = mAudioFrameCount;
        mAudioChunk.push(chunk);

        //mFrameIndex++;
    }

    NN_LOG_V("ScoopAudio:: feed data returns\n");
    return NO_ERROR;
}//NOLINT(impl/function_size)

status_t NvMediaRecorder::setEncoderFormat(sp<AMessage> msg, bool isAudio, int *trackId) {
    status_t err;
    Mutex::Autolock autoLock(mLock);

    if (mMediaMuxer != NULL) {
        NN_LOG_V("added track\n");
        if (isAudio) {
            mAudioTrackIdx = mMediaMuxer->addTrack(msg);
            *trackId = mAudioTrackIdx;
            NN_LOG_V("added audio track with index mAudioTrackIdx - %d\n", mAudioTrackIdx);
        } else {
            mVideoTrackIdx = mMediaMuxer->addTrack(msg);
            *trackId = mVideoTrackIdx;
        }
#ifdef ENABLE_AUDIO_RECORD
        if (mAudioTrackIdx >= 0 && mVideoTrackIdx >= 0) {
#else
        if (mVideoTrackIdx >= 0) {
#endif
        ALOGV("Starting muxer\n");
        err = mMediaMuxer->start();
        if (err != NO_ERROR) {
            NN_LOG_E("Unable to start muxer (err=%d)\n", err);
            return err;
        }
        }
    }
    return NO_ERROR;
}

status_t NvMediaRecorder::writeData() {
    int index = 0;
    int currentIndex = mStartIndex;
    bool isWriteFirstFrame = true;
    int64_t baseTimestamp;
    int startAudioIndex = 0;
    int64_t audioBaseTs = 0;
    status_t err;
    Mutex::Autolock autoLock(mLock);

    NN_LOG_V("start writing  the audio/video chunks in file mStartIndex - %d size - %d\n",mStartIndex, mVideoChunk.size());
    for (int i = 0; i < MAX_COUNT; i++) {
        //if (index > 0) {
        NN_LOG_V("ScoopVideo:: currentIndex - %d mVideoChunk.size() - %d index - %d\n", currentIndex, mVideoChunk.size(), index);
        index = (currentIndex) % mVideoChunk.size();
        NN_LOG_V("ScoopVideo:: get video chunk at index - %d\n", index);
        videoChunk *chunk = mVideoChunk.editItemAt(index);

        if (chunk->state == MEDIA_CHUNK_INIT) {
            NN_LOG_E("ScoopVideo:: This video chunk is just initialise.. no data\n");
            break;
        }
#ifdef ENABLE_AUDIO_RECORD
         if (isWriteFirstFrame) {
             int64_t firstTs, secondTs;
             audioChunk *audioSecond, *audioFirst;

             FrameData *mVideoData = chunk->videoframes.editItemAt(0);
             baseTimestamp = mVideoData->timestamp;
              NN_LOG_E("ScoopVideo:: First video frame timestamp - %lld base ts - %lld\n", mVideoData->timestamp, baseTimestamp);
              audioFirst = mAudioChunk.editItemAt(0);
              firstTs = audioFirst->audioframes->timestamp;
              NN_LOG_V("ScoopAudio:: First audio frame timestamp - %lld mAudioChunk.size() - %d\n", firstTs, mAudioChunk.size());
              for (int k = 1;k < (mAudioChunk.size() - 1);k++) {
                  audioSecond = mAudioChunk.editItemAt(k);
                  int64_t secondTs = audioSecond->audioframes->timestamp;
                  NN_LOG_V("ScoopAudio:: First audio frame ts calculation secondTs - %lld\n", secondTs);
                  if (firstTs <= baseTimestamp && secondTs >= baseTimestamp) {
                      if (secondTs == baseTimestamp) {
                          NN_LOG_E("ScoopAudio:: First audio frame timestamp - %lld \n", secondTs);
                          audioBaseTs = secondTs;
                          startAudioIndex = k;
                      } else {
                          NN_LOG_E("ScoopAudio:: First audio frame timestamp - %lld \n", firstTs);
                          audioBaseTs = firstTs;
                          startAudioIndex = (k - 1);
                      }
                      break;
                  }
                  firstTs = secondTs;
              }
              isWriteFirstFrame = false;
         }
#else
        if (isWriteFirstFrame) {
            FrameData *mVideoData = chunk->videoframes.editItemAt(0);
            baseTimestamp = mVideoData->timestamp;
            NN_LOG_E("ScoopVideo:: First video frame timestamp - %lld base ts - %lld\n", mVideoData->timestamp, baseTimestamp);
            isWriteFirstFrame = false;
        }
#endif

         //write video chunk
         for (int j = 0; j < mIFrameInterval; j++) {
             FrameData *mVideoData = chunk->videoframes.editItemAt(j);
             if (mVideoData->state != FRAME_FILLED) {
                 NN_LOG_E("ScoopVideo:: Last sec data is not completely written\n\n");
                 break;
             }
             NN_LOG_V("ScoopVideo:: write video frame at index - %d mVideoData->flags - %d mCurrentOffset - %lld memptr - %p\n", j, mVideoData->flags, mVideoData->offset, mVideoData->aBuf->data());
             mVideotimestamp = mVideoData->timestamp - baseTimestamp;
             NN_LOG_E("ScoopVideo:: mVideoData->timestamp - %lld baseTimestamp - %lld timestamp - %lld frameIndex - %d currentIndex - %d\n",mVideoData->timestamp, baseTimestamp, mVideotimestamp, j, currentIndex);
             NN_LOG_V("ScoopVideo:: Size of Abuffer sending for write - %d ts - %lld\n",mVideoData->aBuf->size(), mVideotimestamp);

             err = mMediaMuxer->writeSampleData(mVideoData->aBuf, mVideoTrackIdx,
                           /*ptsUsec*/ mVideotimestamp, mVideoData->flags);
             if (err != NO_ERROR) {
                NN_LOG_E("ScoopVideo:: Failed writing data to muxer (err=%d)\n\n", err);
                return err;
             }
             mVideoData->aBuf.clear();
             NN_LOG_V("ScoopVideo:: writeSampleData complete with size  %d\n",mVideoData->aBuf->size());
         }
         currentIndex++;

#ifdef ENABLE_AUDIO_RECORD
         // write audio chunk
         for (int k = startAudioIndex; k < mAudioChunk.size(); k++) {
             int64_t audioTs;
            audioChunk *audioFrame = mAudioChunk.editItemAt(k);
            FrameData *audioData = audioFrame->audioframes;
            if (audioData->state != FRAME_FILLED) {
                 NN_LOG_E("ScoopAudio:: Last sec data is not completely written\n\n");
                 break;
             }
             NN_LOG_V("ScoopAudio:: write audio frame at index - %d audioTs - %lld audioBaseTs - %lld memptr - %p\n", k, audioData->timestamp, audioBaseTs, audioData->aBuf->data());
             audioTs = audioData->timestamp - audioBaseTs;
             NN_LOG_V("ScoopAudio:: Size of Abuffer sending for write - %d ts - %lld mAudioTrackIdx - %d\n",audioData->aBuf->size(), audioTs, mAudioTrackIdx);

             if (audioTs > mVideotimestamp) {
                 NN_LOG_E("ScoopAudio:: partial audio data completely written\n\n");
                 break;
             }

             err = mMediaMuxer->writeSampleData(audioData->aBuf, mAudioTrackIdx,
                           /*ptsUsec*/ audioTs, audioData->flags);
             if (err != NO_ERROR) {
                NN_LOG_E("ScoopAudio:: Failed writing data to muxer (err=%d)\n\n", err);
                return err;
             }
              NN_LOG_V("ScoopAudio:: writeSampleData complete with size  %d\n",audioData->aBuf->size());
              startAudioIndex++;
         }
#endif
    }

#ifdef ENABLE_AUDIO_RECORD
    for (int k = startAudioIndex; k < mAudioChunk.size(); k++) {
        int64_t audioTs;
        audioChunk *audioFrame = mAudioChunk.editItemAt(k);
        FrameData *audioData = audioFrame->audioframes;
        if (audioData->state == FRAME_FILLED) {
            audioTs = audioData->timestamp - audioBaseTs;
            NN_LOG_V("ScoopAudio:: extra write audio frame at index - %d audioTs - %lld audioBaseTs - %lld memptr - %p\n", k, audioData->timestamp, audioBaseTs, audioData->aBuf->data());
            err = mMediaMuxer->writeSampleData(audioData->aBuf, mAudioTrackIdx,
                                        /*ptsUsec*/ audioTs, audioData->flags);
            if (err != NO_ERROR) {
                NN_LOG_E("ScoopAudio:: Failed writing data to muxer (err=%d)\n\n", err);
                return err;
            }
            NN_LOG_V("ScoopAudio:: writeSampleData complete with size  %d\n", audioData->aBuf->size());
            startAudioIndex++;
        }
    }
#endif
    NN_LOG_E("end writing  the chunks in file\n\n");
    mMediaMuxer.clear();
    return NO_ERROR;
}

status_t NvMediaRecorder::setIFrameInterval(int fps) {
    Mutex::Autolock autoLock(mLock);
    NN_LOG_V("setIFrameInterval - %d\n",fps);
    mIFrameInterval = fps;
    return NO_ERROR;
}
