﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
 * Copyright (c) 2007 NVIDIA Corporation.  All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property
 * and proprietary rights in and to this software, related documentation
 * and any modifications thereto.  Any use, reproduction, disclosure or
 * distribution of this software and related documentation without an express
 * license agreement from NVIDIA Corporation is strictly prohibited.
 */

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nn/audioctrl.h>


#include <nn/lmem/lmem_ExpHeap.h>

#include <cerrno>
#include <cstdlib>
#include <cstdio>
#include <fcntl.h>
#include <cstring>
#include <map>

#include <gui/SurfaceComposerClient.h>
#include <gui/Surface.h>
#include <gui/ISurfaceComposer.h>
#include <ui/DisplayInfo.h>
#include <binder/IPCThreadState.h>
#include <cutils/properties.h>
#include <nuplayer/NuPlayerDriver.h>
#include <media/IMediaPlayerClient.h>
#include <media/IMediaPlayerService.h>
#include <media/IMediaHTTPService.h>
#include <utils/Log.h>
#include <ALooper.h>
#include "nvgr.h"
#include "SfAudioSink.h"
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <mm_MemoryManagement.h>
#include <mm_Threads.h>
#include <nn/nn_Assert.h>
#include <ui/GraphicBufferAllocator.h>
#include <ui/GraphicBufferMapper.h>
#include <private/gui/ComposerService.h>

#include "../common/HeapTracker.h"

const int NUM_INSTANCE = 2;


_Noreturn void _Exit (int);
extern void initializeLooperRoster();
extern void initializeAtomizer();
extern void initializeDataSource();
extern void shutdownLooperRoster();
extern void shutdownAtomizer();
extern void shutdownDataSource();

using namespace android;
static char         *inputFileName;
int                 width;
int                 height;
bool                isComplete;
volatile bool       isSetVideoSize;
Mutex               *mLock;
Condition           *mCondition;


namespace{
    const int FsHeapSize = 512 * 1024;
    const int mmHeapSize = 512 << 20;
    const int CoreHeapSize = 8 << 20;
    const int mmFirmwareMemorySize = 8 << 20;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;
    char                        g_mmHeapBuffer[mmHeapSize];
    nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    char                        g_CoreHeapBuffer[CoreHeapSize];
    nn::mem::StandardAllocator  g_CoreAllocator(g_CoreHeapBuffer, sizeof(g_CoreHeapBuffer));
    char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }

static nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    return g_MultimediaAllocator;
}

static HeapTracker &MMHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &CoreHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

void *mmAllocate(size_t size, size_t align, void *userPtr)
{
    if(size == 0)
        return NULL;
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    void *address = GlobalAllocator().Allocate(size, align);
    static size_t totalAllocatedSize = 0;

    totalAllocatedSize += size;
    static_cast<HeapTracker*>(userPtr)->Track(address, size);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return address;
}

void mmDeallocate(void *address, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();

    GlobalAllocator().Free(address);
    static_cast<HeapTracker*>(userPtr)->Untrack(address);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
}

void *mmReallocate(void* address, size_t newSize, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    if (address)
       static_cast<HeapTracker*>(userPtr)->Untrack(address);
    void *memory = GlobalAllocator().Reallocate(address, newSize);
    static_cast<HeapTracker*>(userPtr)->Track(memory, newSize);
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return memory;
}
}

static bool  getDisplaySize(sp<SurfaceComposerClient> &client,
               int32_t id,
               int &width,
               int &height)
{
    sp<IBinder> dpy = client->getBuiltInDisplay(id);
    if (dpy == NULL)
        return false;

    DisplayInfo info;
    status_t err = client->getDisplayInfo(dpy, &info);
    if (err != NO_ERROR)
        return false;

    width = info.w;
    height = info.h;
    return true;
}


static void usage(const char *s)
{
    NN_SDK_LOG("Usage - nvsfplayer.kip  <inputFile> \n");
}

static void notify(void *cookie, int msg, int ext1, int ext2, const Parcel *obj)
{
    switch(msg)
    {
    case MEDIA_PLAYBACK_COMPLETE:
             isComplete = true;
             NN_LOG("MEDIA_PLAYBACK_COMPLETE\n");
             mCondition->signal();
    break;
    case MEDIA_SET_VIDEO_SIZE:
             width = ext1;
             height = ext2;
             isSetVideoSize = true;
             NN_SDK_LOG("MEDIA_SET_VIDEO_SIZE\n");
             mCondition->signal();

    break;
    case MEDIA_PREPARED:
    case MEDIA_SEEK_COMPLETE:
    case MEDIA_BUFFERING_UPDATE:
    case MEDIA_ERROR:
    case MEDIA_INFO:
    default:
    break;
    }
}

extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    const size_t mallocSize = 32 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    NN_SDK_LOG("\ninit done: \n");
}

void NvSfPlayer_MultiInstance()
{
    android::sp<android::NuPlayerDriver> nvSFPlayer[NUM_INSTANCE];
    sp<SurfaceControl> videoControl[NUM_INSTANCE] ;
    int flags = ISurfaceComposerClient::eOpaque;
    int nState = 0;
    int retCode = 0;
    int displayWidth = 256;
    int displayHeight = 256;
    sp<SurfaceComposerClient> videoClient = new SurfaceComposerClient;
    bool  ok = getDisplaySize(videoClient,
                        ISurfaceComposer::eDisplayIdHdmi,
                        displayWidth,
                        displayHeight);
    if (ok == false) {
        NN_SDK_LOG("Please connect HDMI for multi-instance test\n");
        FAIL();
    }

    char clientName[60];
    for (int i = 0; i< NUM_INSTANCE; i++) {
        NN_SDK_LOG("Multi-instance video %d\n",i);
        nvSFPlayer[i] = new NuPlayerDriver;
        nvSFPlayer[i]->setNotifyCallback(0, notify);
        nvSFPlayer[i]->setDataSource((const sp<IMediaHTTPService>&)NULL,
                                     (const char *)inputFileName,
                                     (const KeyedVector<String8, String8> *)NULL);
        retCode = nvSFPlayer[i]->prepare();
        if(retCode < 0)
        {
            NN_SDK_LOG("\nPrepare failed: %d\n", retCode);
            delete mLock;
            delete mCondition;
            FAIL();
        };
        nvSFPlayer[i]->setLooping(0);
        nvSFPlayer[i]->prepareAsync();
        if (!isSetVideoSize) {
            Mutex::Autolock _l(mLock);
            mCondition->wait(*mLock);  // wait for MEDIA_SET_VIDEO_SIZE
            isSetVideoSize = false;
        }
        sprintf(clientName,"VideoNew%d",i);
        videoControl[i] = videoClient->createSurface(String8(clientName),
                    width, height, NVGR_PIXEL_FORMAT_NV12, flags);
        sp<Surface> videoSurface2[NUM_INSTANCE];
        videoClient->openGlobalTransaction();
        if (i == 0) {
            NN_SDK_LOG("[%d]setPosition x:%d y:%d\n",i, 0, 0);
            videoControl[i]->setPosition(0, 0);
        } else if (i == 1) {
            NN_SDK_LOG("[%d]setPosition x:%d y:%d\n",i, displayWidth / 2, 0);
            videoControl[i]->setPosition(displayWidth / 2, 0);
        } else {
            NN_SDK_LOG("[%d]setPosition x:%d y:%d\n",i, 0, displayHeight / 2);
            videoControl[i]->setPosition(0, displayHeight / 2);
        }

        videoControl[i]->setSize(displayWidth / 2, displayHeight / 2);
        videoControl[i]->setLayer(2);
        nState = videoControl[i]->show();
        videoClient->closeGlobalTransaction();
        videoSurface2[i] = videoControl[i]->getSurface();
        nvSFPlayer[i]->setVideoSurfaceTexture(videoSurface2[i]->getIGraphicBufferProducer());
        nvSFPlayer[i]->start();
        sleep(1);
    }

    while (!isComplete) {
        Mutex::Autolock _l(mLock);
        mCondition->wait(*mLock);  // wait for MEDIA_PLAYBACK_COMPLETE
    }

    for (int i=0; i < NUM_INSTANCE; i++) {
       nvSFPlayer[i]->setNotifyCallback(0, 0);
       nvSFPlayer[i]->stop();
       nvSFPlayer[i]->reset();
       videoControl[i]->clear();
    }
    videoClient->dispose();
}

void NvSfplayer_VideoPlayback(android::sp<android::NuPlayerDriver> nvSFPlayer,bool actualRes)
{
    int flags = ISurfaceComposerClient::eOpaque;
    bool isDispAcutalResolution = actualRes;
    if (!isSetVideoSize) {
        Mutex::Autolock _l(mLock);
        mCondition->wait(*mLock);  // wait for MEDIA_SET_VIDEO_SIZE
    }
    //Setup video surface
    NN_SDK_LOG("Setup video surface\n");
    int nState = 0;
    sp<SurfaceComposerClient> videoClient = new SurfaceComposerClient;
    int displayWidth = 256;
    int displayHeight = 256;
    bool  ok = getDisplaySize(videoClient,
                        ISurfaceComposer::eDisplayIdHdmi,
                        displayWidth,
                        displayHeight);
    if (ok == false)
    {
        getDisplaySize(videoClient,
                       ISurfaceComposer::eDisplayIdMain,
                       displayWidth,
                       displayHeight);
    }

    NN_SDK_LOG("display is %ld x %ld\n", displayWidth, displayHeight);
    if (!isDispAcutalResolution)
        flags |= ISurfaceComposerClient::eFullscreen;

    NN_SDK_LOG("CreateSurface with width %d height %d\n",width, height);
    sp<SurfaceControl> videoControl = videoClient->createSurface(String8("Video"),
                    width, height, NVGR_PIXEL_FORMAT_NV12, flags);
    sp<Surface> videoSurface;
    videoClient->openGlobalTransaction();
    videoControl->setPosition(0,0);
    videoControl->setSize(width, height);
    videoControl->setLayer(2);  //set zdepth greater than 1 as devmenu set it 1.
    nState = videoControl->show();
    videoClient->closeGlobalTransaction();
    NN_SDK_LOG("getSurface\n");
    videoSurface = videoControl->getSurface();
    NN_SDK_LOG("setVideoSurfaceTexture\n");
    nvSFPlayer->setVideoSurfaceTexture(videoSurface->getIGraphicBufferProducer());

    NN_SDK_LOG("\nStart Playback\n");
    nvSFPlayer->start();
    NN_SDK_LOG("\nWait for the playback to be completed\n");

    while (!isComplete) {
        Mutex::Autolock _l(mLock);
        mCondition->wait(*mLock);  // wait for MEDIA_PLAYBACK_COMPLETE
    }
    videoControl->clear();
    videoClient->dispose();

}

void NvSfplayer_test()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    bool isMultiInstance = false;
    bool isDispAcutalResolution = false;
    bool AudioTargetSpeaker = false;
    bool AudioTargetHeadphone = false;
    bool isMemProfiling = false;
    NN_SDK_LOG("Entering Main\n");

    if(argc <= 1)
    {
       usage(argv[0]);
       FAIL();
    }
    if (argc >= 3)
    {
        for (int i=2; i < argc; i++) {
            if (!strcmp(argv[i],"multi-instance")) {
                isMultiInstance = true;
            }
            if (!strcmp(argv[i],"actual-resolution")) {
                isDispAcutalResolution = true;
            }
            if (!strcmp(argv[i],"memprofile")) {
                isMemProfiling = true;
            }
            if (!strcmp(argv[i],"forcespeaker")) {
                AudioTargetSpeaker = true;
            }
            if (!strcmp(argv[i],"forceheadphone")) {
                AudioTargetHeadphone = true;
            }
        }
    }
#ifdef ENABLE_NVMEMPROFILE
    if (isMemProfiling) {
        if (!NvMemoryProfilerConnect()) {
            NN_SDK_LOG("Failed to enable memory profiling\n");
        }
    }
#endif
    isComplete = false;
    height = 0;
    width =0;
    isSetVideoSize = false;

    inputFileName = argv[1];
    NN_SDK_LOG("\nInput File Name %s\n",inputFileName);
    bool sdcardMounted = false;
    const char* token = "sdcard";
    int compare = strncmp(inputFileName, token, strlen(token));
    if (compare == 0)
    {
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if (resultSdcardMount.IsFailure())
        {
            NN_SDK_LOG("nn::fs::SD card Mount failure. Module:%d, Description:%d\n",
                    resultSdcardMount.GetModule(),
                    resultSdcardMount.GetDescription());
            FAIL();
        }
        sdcardMounted = true;
    }
    else
    {
        nn::Result resultHostMount = nn::fs::MountHostRoot();
        if (resultHostMount.IsFailure())
        {
            NN_SDK_LOG("nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                    resultHostMount.GetModule(),
                    resultHostMount.GetDescription());
            FAIL();
        }
    }

    initializeLooperRoster();
    initializeAtomizer();
    initializeDataSource();
    sp<ALooper> looper = new ALooper;
    looper->start();
    NN_SDK_LOG("Create NuPlayer\n");
    mLock = new Mutex;
    mCondition = new Condition;

    if (isMultiInstance) {
        NvSfPlayer_MultiInstance();
    }

    android::sp<android::NuPlayerDriver> nvSFPlayer = new NuPlayerDriver;
    nvSFPlayer->setNotifyCallback(0, notify);
    android::sp<SfAudioSink> nvsfAudioSink = new SfAudioSink;
    nvSFPlayer->setAudioSink(nvsfAudioSink);
    nvSFPlayer->setDataSource((const sp<IMediaHTTPService>&)NULL,
                                (const char *)inputFileName,
                                (const KeyedVector<String8, String8> *)NULL);

    if (AudioTargetSpeaker) {
        nn::audioctrl::SetDefaultTarget(nn::audioctrl::AudioTarget_Speaker, nn::TimeSpan::FromMilliSeconds(10), nn::TimeSpan::FromMilliSeconds(10));
        nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Speaker, nn::audioctrl::GetTargetVolumeMax());
    }

    if (AudioTargetHeadphone) {
        nn::audioctrl::SetDefaultTarget(nn::audioctrl::AudioTarget_Speaker, nn::TimeSpan::FromMilliSeconds(10), nn::TimeSpan::FromMilliSeconds(10));
        // Note: SetDefaultTarget can't be set AudioTarget_Headphone (only Speaker or TV)
        nn::audioctrl::SetTargetVolume(nn::audioctrl::AudioTarget_Headphone, nn::audioctrl::GetTargetVolumeMax());
    }

    status_t retCode = nvSFPlayer->prepare();
    if(retCode < 0)
    {
        NN_SDK_LOG("\nPrepare failed: %d\n", retCode);
        delete mLock;
        delete mCondition;
        FAIL();
    };
    nvSFPlayer->setLooping(0);
    nvSFPlayer->prepareAsync();

    if (nvSFPlayer->hasVideoTrack()) {
        NvSfplayer_VideoPlayback(nvSFPlayer,isDispAcutalResolution);
    } else {
        NN_SDK_LOG("\nStart Audio only Playback\n");
        nvSFPlayer->start();
        NN_SDK_LOG("\nWait for the playback to be completed\n");

        while (!isComplete) {
           Mutex::Autolock _l(mLock);
           mCondition->wait(*mLock);  // wait for MEDIA_PLAYBACK_COMPLETE
        }
    }
    nvSFPlayer->setNotifyCallback(0, 0);
    NN_LOG("%s",nvSFPlayer->getStats());
    nvSFPlayer->stop();
    NN_SDK_LOG("reset \n");
    nvSFPlayer->reset();

    delete mLock;
    delete mCondition;
    if (sdcardMounted == true)
    {
        nn::fs::Unmount("sdcard");
    }
    else
    {
        nn::fs::UnmountHostRoot();
    }
    NN_SDK_LOG("\nPlayback ends...\n");
    looper->stop();
    shutdownLooperRoster();
    shutdownAtomizer();
    shutdownDataSource();

}//NOLINT(impl/function_size)

TEST(NvSfPlayerTest, Video)
{
     /* Set allocator callback functions */
    nv::mm::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    nv::mm::SetThreadCoreMask(nn::os::IdealCoreDontCare, 0x4);  //Set core mask to run all MM threads to particular core
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsServiceName("nvdrv:t");
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));
#ifdef ENABLE_NVMEMPROFILE
    g_mmHeap = NvMemoryProfilerRegisterHeap("multimedia");
    g_coreHeap =  NvMemoryProfilerRegisterHeap("CORE");
#endif

    NvSfplayer_test();

    nv::FinalizeGraphics();


    NN_LOG("[MM HEAP]\n");
    MMHeap().OutputUsage();
    NN_LOG("[CORE HEAP]\n");
    CoreHeap().OutputUsage();

    SUCCEED();
}

volatile int*   __errno()
{
    static int i = 0;
    return &i;
}
