﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/* Copyright (c) 2015-16, NVIDIA CORPORATION.  All rights reserved.
*
* NVIDIA Corporation and its licensors retain all intellectual property
* and proprietary rights in and to this software, related documentation
* and any modifications thereto.  Any use, reproduction, disclosure or
* distribution of this software and related documentation without an
* express license agreement from NVIDIA Corporation is strictly prohibited.
*/

#include <cstdio>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <map>
#include "jpeglib.h"
#include "jpegint.h"
#include "nvos.h"
#include "Utilities.h"
#include <nnt/nntest.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nnt/nnt_Argument.h>
#include "mm_MemoryManagement.h"

const int BUFF_HEIGHT = 8;
const int MAX_WIDTH = 4 * 1024;
const int MAX_HEIGHT = 4 * 1024;

/* Create the add-on message string table. */
namespace nn { namespace nvjpg {

namespace
{
#define PROF_START(t)   (t = NvOsGetTimeUS())
#define PROF_END(t)     (t = NvOsGetTimeUS() - t)
const size_t heapSize = 200 * 1024 * 1024;
const unsigned long int CRC32_POLYNOMIAL = 0xEDB88320L;

static NvU32 CRCTable[256];

void ImageJpegDecodingNvjpeg_test();

static void BuildCRCTable()
{
    NvU16 i;
    NvU16 j;
    NvU32 crc;
    for (i = 0; i <= 255; i++)
    {
        crc = i;
        for (j = 8; j > 0; j--)
        {
            if (crc & 1)
            {
                crc = (crc >> 1) ^ CRC32_POLYNOMIAL;
            }
            else
            {
                crc >>= 1;
            }
        }
        CRCTable[i] = crc;
    }
}

static int32_t CalculateBufferCRC(uint32_t count, uint32_t crc, uint8_t *buffer)
{
    uint8_t *p;
    uint32_t temp1;
    uint32_t temp2;
    p = (uint8_t*)buffer;
    while (count-- != 0)
    {
        temp1 = (crc >> 8) & 0x00FFFFFFL;
        temp2 = CRCTable[((uint32_t)crc ^ *p++) & 0xFF];
        crc = temp1 ^ temp2;
    }
    return crc;
}

//---------------------------------------------------------------------------
// This unnamed namespace includes file system heap allocator and deallcator
//---------------------------------------------------------------------------

namespace{

    const int FsHeapSize = 512 * 1024;

    const int mmHeapSize = 90 * 1024 * 1024;
    const int mmFirmwareMemorySize = 8 * 1024 * 1024;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;

    char                        g_mmHeapBuffer[mmHeapSize];
    nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));

    char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }

    std::map<void*, size_t> g_Allocs;
    nn::os::Mutex g_AllocMutex(true);

    void* MultimediaAllocate(size_t size, size_t alignment, void *userPtr)
    {
        g_AllocMutex.Lock();
        void *ptr = g_MultimediaAllocator.Allocate(size, alignment);
        g_Allocs.insert(std::pair<void*, size_t>(ptr, size));
        g_AllocMutex.Unlock();

        return ptr;
    }

    void MultimediaFree(void *addr, void *userPtr)
    {
        if(!addr)
            return;

        g_AllocMutex.Lock();
        std::map<void*, size_t>::iterator it = g_Allocs.find(addr);
        NN_ASSERT(it != g_Allocs.end());
        g_Allocs.erase(it);
        g_MultimediaAllocator.Free(addr);
        g_AllocMutex.Unlock();
    }

    void *MultimediaReallocate(void* addr, size_t newSize, void *userPtr)
    {
        g_AllocMutex.Lock();

        if(addr)
        {
            std::map<void*, size_t>::iterator it = g_Allocs.find(addr);
            NN_ASSERT(it != g_Allocs.end());
            g_Allocs.erase(it);
        }

        void *ptr = g_MultimediaAllocator.Reallocate(addr, newSize);
        g_Allocs[ptr] = newSize;
        g_AllocMutex.Unlock();

        return ptr;
    }
}

//----------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//----------------------------------------------------------
extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 150 * 1024 * 1024;
    uintptr_t address;
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(heapSize);
    if (!result.IsSuccess()) {
        NN_SDK_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocMemorySize);

    // Set file system allocator and deallocator
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

static void displayUsage()
{
    NN_LOG("usage: %s --in <file> --out <file>\n", nnt::GetHostArgv()[0]);
    NN_LOG("Optinal params: \n");
    NN_LOG("\t --incrc value        value refers to reference CRC value of input file\n");
    NN_LOG("\t --outcrc value       value refers to reference CRC value of decoded data\n");
    NN_LOG("\t --rgb value          0:Decodes in YUV format, 1:Decodes in RGB format. Default is YUV format\n");
    NN_LOG("\t --swpath value       0:Decodes using HW path, 1:Decodes using SW path. Default is HW path\n");
    NN_LOG("\t --perf value         0:Profiling is disabled, 1:Profiling is enabled\n");
}

static void GetYuvData(jpeg_decompress_struct *pCinfo, uint8_t *pixelLine, uint32_t *pSize)
{
    uint16_t LumaWidth, ChromaWidth = 0;
    uint16_t LumaHeight, ChromaHeight = 0;
    JSAMPLE *line;
    uint16_t i;
    uint16_t Pitch;

    Pitch = pCinfo->jpegTegraMgr->pitch[0];
    LumaWidth = pCinfo->output_width;
    LumaHeight = pCinfo->output_height;
    for (i = 0; i < LumaHeight; i++)
    {
        line = pCinfo->jpegTegraMgr->buff[0] + (i * Pitch);
        for (uint16_t j = 0; j < LumaWidth; j++)
        {
            *pixelLine++ = *(line + j); // Y
        }
    }
    *pSize = LumaWidth * LumaHeight;

    switch (pCinfo->jpegTegraMgr->mcu_type)
    {
    case 0:      // YUV 420
        ChromaWidth  = ((pCinfo->output_width + 1) & ~1) / 2;
        ChromaHeight = ((pCinfo->output_height + 1) & ~1) / 2;
        break;

    case 1:     // YUV 422
        ChromaWidth  = ((pCinfo->output_width + 1) & ~1) / 2;
        ChromaHeight = pCinfo->output_height;
        break;

    case 2:     // YUV 422 Transposed
        ChromaWidth  = pCinfo->output_width;
        ChromaHeight = ((pCinfo->output_height + 1) & ~1) / 2;
        break;

    case 3:     // YUV 444
        ChromaWidth  = pCinfo->output_width;
        ChromaHeight = pCinfo->output_height;
        break;

    case 4:     // YUV 400
        ChromaWidth  = 0;
        ChromaHeight = 0;
        break;

    default:
        break;
    }

    *pSize += ChromaWidth * ChromaHeight * 2;

    Pitch = pCinfo->jpegTegraMgr->pitch[2];
    for (i = 0; i < ChromaHeight; i++)
    {
        line = pCinfo->jpegTegraMgr->buff[2] + (i * Pitch);
        for (uint16_t j = 0; j < ChromaWidth; j++)
        {
            *pixelLine++ = *(line + j); // V
        }
    }

    Pitch = pCinfo->jpegTegraMgr->pitch[1];
    for (i = 0; i < ChromaHeight; i++)
    {
        line = pCinfo->jpegTegraMgr->buff[1] + (i * Pitch);
        for (uint16_t j = 0; j < ChromaWidth; j++)
        {
            *pixelLine++ = *(line + j); // U
        }
    }
}

static void GetRgbData(jpeg_decompress_struct *pCinfo, uint8_t *pixelLine, uint32_t *pSize)
{
    uint16_t DumpWidth;
    uint16_t DumpHeight;
    JSAMPLE *line;
    uint16_t i;
    uint16_t Pitch;
    int32_t numOfBytes = 4;

    Pitch = pCinfo->jpegTegraMgr->pitch[0];
    DumpWidth = pCinfo->output_width;
    DumpHeight = pCinfo->output_height;
    for (i = 0; i < DumpHeight; i++)
    {
        line = pCinfo->jpegTegraMgr->buff[0] + (i * Pitch);
        memcpy(pixelLine, line, (DumpWidth * numOfBytes));
        line += Pitch;
        pixelLine += DumpWidth * numOfBytes;
    }

    *pSize = DumpWidth * DumpHeight * 4;
}


void ImageJpegDecodingNvjpeg_test()
{
    struct jpeg_error_mgr errorInfo;
    int kWorkBufSize;
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    const char *inFile = NULL;
    const char *outFile = NULL;
    JDIMENSION readCt;
    JSAMPROW lines[BUFF_HEIGHT] = {}; // This is actually not used by the HW path
    uint8_t OutCrcEnabled = 0, InCrcEnabled = 0;
    uint32_t OutRefCrc = 0, InRefCrc = 0;
    uint32_t Crc = 0;
    uint32_t Loop, LoopCount = 1;
    uint32_t MaxBlockPerRow, MaxRowsPerChunk;
    uint32_t RgbMode = 0;
    uint32_t SwPath = 0;
    uint32_t Perf = 0;
    uint64_t StartTime = 0;
    uint64_t EndTime = 0;
    uint64_t MountTime = 0;
    uint64_t ArgParsingTime = 0;
    uint64_t WorkBufAllocTime = 0;
    uint64_t CreateDecompressTime = 0;
    uint64_t JpegFileReadTime = 0;
    uint64_t JinitTime = 0;
    uint64_t MemSrcTime = 0;
    uint64_t ReadHdrTime = 0;
    uint64_t StartDecompressTime = 0;
    uint64_t PixBufAllocTime = 0;
    uint64_t ReadScanlineTime = 0;
    uint64_t GetYuvDataTime = 0;
    uint64_t GetRgbDataTime = 0;
    uint64_t FinishDecompressTime = 0;
    uint64_t DestroyDecompressTime = 0;
    uint64_t UnMountTime = 0;

    StartTime = NvOsGetTimeUS();

    PROF_START(ArgParsingTime);

    for (int i = 1; i < argc; i++)
    {
        if(argc - i < 2)
            break;
        else if(!strcmp(argv[i], "--in"))
            inFile = argv[++i];
        else if(!strcmp(argv[i], "--out"))
            outFile = argv[++i];
        else if(!strcmp(argv[i], "--outcrc"))
        {
            OutCrcEnabled = 1;
            sscanf(argv[++i], "%8x\n", &OutRefCrc);
        }
        else if(!strcmp(argv[i], "--incrc"))
        {
            InCrcEnabled = 1;
            sscanf(argv[++i], "%8x\n", &InRefCrc);
        }
        else if(!strcmp(argv[i], "--loop"))
        {
            sscanf(argv[++i], "%d\n", &LoopCount);
        }
        else if (!strcmp(argv[i], "--rgb"))
        {
            sscanf(argv[++i], "%d\n", &RgbMode);
        }
        else if (!strcmp(argv[i], "--swpath"))
        {
            sscanf(argv[++i], "%d\n", &SwPath);
        }
        else if (!strcmp(argv[i], "--perf"))
        {
            sscanf(argv[++i], "%d\n", &Perf);
        }
        else
        {
            NN_SDK_LOG("Usage 1\n");
            displayUsage();
            return;
        }
    }
    if(!inFile)
    {
        NN_SDK_LOG("Usage 2\n");
        displayUsage();
        return;
    }
    PROF_END(ArgParsingTime);

    PROF_START(MountTime);
    NN_SDK_LOG("The file path is :%s\n",inFile);
    bool sdcardMounted = false;
    const char* token = "sdcard";
    int compare = strncmp(inFile, token, strlen(token));

    if (InCrcEnabled || OutCrcEnabled)
        BuildCRCTable();

    if (compare == 0)
    {
        NN_SDK_LOG("in sd card  :%s\n",inFile);
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if( resultSdcardMount.IsFailure() )
        {
            NN_SDK_LOG( "\n nn::fs::SD card mount failure. Module:%d, Description:%d\n" ,
                    resultSdcardMount.GetModule(),
                    resultSdcardMount.GetDescription());
            return;
        }
        sdcardMounted = true;
    }
    else {
        nn::Result resultHostMount = nn::fs::MountHostRoot();
        if (resultHostMount.IsFailure())
        {
            NN_SDK_LOG("nn::fs::Host root mount failure.\n",
                    resultHostMount.GetModule(),
                    resultHostMount.GetDescription());
            return;
        }

    }
    PROF_END(MountTime);

    MaxBlockPerRow = (MAX_WIDTH + 15) / 16;
    MaxRowsPerChunk = (MAX_HEIGHT + 15) / 16 * 2;

    // Work buffer
    kWorkBufSize =  (1600 +             /* first PERMANENT pool */
                     16000 +            /* first IMAGE pool */
                     5000 +             /* additional IMAGE pools */
                     MaxBlockPerRow * MaxRowsPerChunk * DCTSIZE2 * 2);  /* Required if SW path is used */
    PROF_START(WorkBufAllocTime);
    Buffer workBuf(kWorkBufSize);
    PROF_END(WorkBufAllocTime);

    jpeg_workbuf wbMgr = {};
    wbMgr.ptr = reinterpret_cast<JSAMPLE*>(workBuf.GetDataPtr());
    wbMgr.total = workBuf.GetDataSize();

    if (!wbMgr.ptr)
    {
        NN_SDK_LOG("%s:%d Unable to allocate memory for workbuffer\n", __func__, __LINE__);
        FAIL();
    }

    PROF_START(CreateDecompressTime);
    jpeg_decompress_struct cinfo = {};
    cinfo.workbuf = &wbMgr;

    cinfo.err = jpeg_std_error(&errorInfo);

    jpeg_create_decompress(&cinfo);
    PROF_END(CreateDecompressTime);

    for (Loop = 0; Loop < LoopCount; Loop++)
    {
        PROF_START(JpegFileReadTime);
        FileData jpegData(inFile);
        PROF_END(JpegFileReadTime);

        if (!jpegData.GetDataPtr())
        {
            NN_SDK_LOG("%s:%d Unable to allocate memory for input buffer\n", __func__, __LINE__);
            FAIL();
        }

        if (InCrcEnabled)
        {
            Crc = CalculateBufferCRC(jpegData.GetDataSize(), 0, (uint8_t *)(jpegData.GetDataPtr()));
            if (InRefCrc == Crc)
                NN_SDK_LOG("%s:%d Iput CRC PASSED\n", __func__, __LINE__);
            else
                NN_SDK_LOG("%s:%d Input CRC FAILED\n", __func__, __LINE__);
        }
        if (Loop)
        {
            /*
             * Initialization of memory mgr and marker reader is done from second iteration on wards as these are
             * already called in the function jpeg_create_decompress
            */
            PROF_START(JinitTime);
            jinit_memory_mgr((j_common_ptr) &cinfo);
            jinit_marker_reader(&cinfo);
            cinfo.global_state = 200; //DSTATE_START;
            PROF_END(JinitTime);
        }

        PROF_START(MemSrcTime);
        jpeg_mem_src(&cinfo, (unsigned char * )jpegData.GetDataPtr(), jpegData.GetDataSize());
        PROF_END(MemSrcTime);

        PROF_START(ReadHdrTime);
        jpeg_read_header(&cinfo, false);
        PROF_END(ReadHdrTime);
        // Initialize input/output parameters
        cinfo.scale_num = 1;
        cinfo.scale_denom = 1;

        if ((cinfo.tegra_acceleration == TRUE) && cinfo.is_baseline && !SwPath)
        {
            if (RgbMode)
                cinfo.out_color_space = JCS_RGBA_8888;
            else
                cinfo.out_color_space = JCS_YCbCr;

        }
        else
        {
            cinfo.out_color_space = JCS_RGB;
        }

        cinfo.dct_method = JDCT_ISLOW;
        cinfo.do_fancy_upsampling = FALSE;
        cinfo.do_block_smoothing = FALSE;

        // Start Decoding
        PROF_START(StartDecompressTime);
        jpeg_start_decompress(&cinfo);
        PROF_END(StartDecompressTime);

        if (cinfo.tegra_acceleration == FALSE)
        {
            const uint32_t kRowStride = cinfo.output_width * 3;
            auto pLineHead = reinterpret_cast<uintptr_t>(workBuf.GetDataPtr()) + (int)((kWorkBufSize - sizeof(JSAMPLE) * BUFF_HEIGHT * kRowStride));
            for (int i = 0; i < BUFF_HEIGHT; i++)
            {
                lines[i] = reinterpret_cast<JSAMPLE*>(pLineHead) + i * kRowStride;
            }
        }
        // Allocate memory
        // pixBuf buffer required for writing decoded pixels. Allocate worst case size for RGB
        PROF_START(PixBufAllocTime);
        Buffer pixBuf(cinfo.output_width * cinfo.output_height * 4); // RGBA 32bit
        PROF_END(PixBufAllocTime);

        uint8_t *pixelLine = reinterpret_cast<uint8_t*>(pixBuf.GetDataPtr());
        JDIMENSION linesToDecode = cinfo.output_height;
        JSAMPLE *line;
        uint32_t Size = 0;

        if (!pixelLine)
        {
            NN_SDK_LOG("%s:%d Unable to allocate memory for output buffer\n", __func__, __LINE__);
            FAIL();
        }

        while (cinfo.output_scanline < cinfo.output_height)
        {
            if (cinfo.tegra_acceleration)
            {
                PROF_START(ReadScanlineTime);
                readCt = jpeg_read_scanlines(&cinfo, lines, linesToDecode);
                PROF_END(ReadScanlineTime);
                if (readCt && (cinfo.out_color_space == JCS_YCbCr))
                {
                    PROF_START(GetYuvDataTime);
                    GetYuvData(&cinfo, pixelLine, &Size);
                    PROF_END(GetYuvDataTime);
                }
                else if (cinfo.out_color_space == JCS_RGBA_8888)
                {
                    PROF_START(GetRgbDataTime);
                    GetRgbData(&cinfo, pixelLine, &Size);
                    PROF_END(GetRgbDataTime);
                }
            }
            else
            {
                JDIMENSION currentLine = cinfo.output_scanline;
                linesToDecode = cinfo.output_height - currentLine;
                linesToDecode = (linesToDecode < BUFF_HEIGHT)?
                    linesToDecode: BUFF_HEIGHT;
                readCt = jpeg_read_scanlines(&cinfo, lines, linesToDecode);

                for (JDIMENSION i = 0; i < readCt; i++)
                {
                    pixelLine = reinterpret_cast<uint8_t*>(pixBuf.GetDataPtr()) + ((currentLine + i) * cinfo.output_width * sizeof(nn::Bit32));

                    line = lines[i];
                    for (uint16_t j = 0; j < cinfo.output_width; j++, line += 3)
                    {
                        int pixelIndex = j * sizeof(nn::Bit32);

                        pixelLine[pixelIndex + 0] = line[0]; // R
                        pixelLine[pixelIndex + 1] = line[1]; // G
                        pixelLine[pixelIndex + 2] = line[2]; // B
                        pixelLine[pixelIndex + 3] = 0xFF; // A
                    }
                }
                Size = cinfo.output_width * cinfo.output_height * 4;
            }
        }
        if (OutCrcEnabled)
        {
            uint8_t *pBuffer;

            Crc = 0;

            pBuffer = (uint8_t *)pixBuf.GetDataPtr();

            Crc = CalculateBufferCRC(Size, Crc, pBuffer);

            if (OutRefCrc == Crc)
                NN_SDK_LOG("%s:%d Test %s PASSED\n", __func__, __LINE__, inFile);
            else
            {
                NN_SDK_LOG("%s:%d Test %s FAILED\n", __func__, __LINE__, inFile);
                FAIL();
            }
        }

        if (outFile != NULL)
        {
            nn::Result result;
            nn::fs::FileHandle fileHandle;
            nn::fs::CreateFile(outFile, Size).IsSuccess();
            result = nn::fs::OpenFile(&fileHandle, outFile, nn::fs::OpenMode_Write);
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::WriteFile(fileHandle, 0, pixBuf.GetDataPtr(), Size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            nn::fs::CloseFile(fileHandle);
        }

        if (LoopCount > 1)
            memset(pixBuf.GetDataPtr(), 0, ((cinfo.output_width + 15) & ~15) * cinfo.output_height * 1.5);
    }

    PROF_START(FinishDecompressTime);
    (void) jpeg_finish_decompress(&cinfo);
    PROF_END(FinishDecompressTime);

    PROF_START(DestroyDecompressTime);
    jpeg_destroy_decompress(&cinfo);
    PROF_END(DestroyDecompressTime);

    PROF_START(UnMountTime);
    if( sdcardMounted == true )
    {
        nn::fs::Unmount("sdcard");
    }else{
        nn::fs::UnmountHostRoot();
    }
    PROF_END(UnMountTime);

    EndTime = NvOsGetTimeUS();
    if (Perf)
    {
        NN_LOG("JpegFilename, Width, Height, MountTime, ArgParsingTime, WorkBufAllocTime, CreateDecompressTime, JpegFileReadTime, JinitTime, MemSrcTime, ReadHdrTime, StartDecompressTime, PixBufAllocTime, ReadScanlineTime, GetYuvDataTime, GetRgbDataTime, FinishDecompressTime, DestroyDecompressTime, UnMountTime, TotalDecodeTime\n");
        NN_LOG("%s, %d, %d, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld, %lld\n",
               inFile, cinfo.output_width, cinfo.output_height, MountTime, ArgParsingTime, WorkBufAllocTime, CreateDecompressTime, JpegFileReadTime, JinitTime, MemSrcTime, ReadHdrTime, StartDecompressTime, PixBufAllocTime, ReadScanlineTime, GetYuvDataTime, GetRgbDataTime, FinishDecompressTime, DestroyDecompressTime, UnMountTime, (EndTime - StartTime));
    }

}//NOLINT(impl/function_size)

TEST(ImageJpegDecodingNvjpeg, Decoder)
{
    /* Set allocator callback functions */
    nv::mm::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsServiceName("nvdrv:t");
    nv::SetGraphicsDevtoolsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));

    ImageJpegDecodingNvjpeg_test();

    nv::FinalizeGraphics();

    NN_LOG("Memory left: %i allocations\n", g_Allocs.size());
    int memory = 0;
    for(std::map<void*, size_t>::iterator it = g_Allocs.begin(); it != g_Allocs.end(); ++it)
    {
        NN_SDK_LOG("address: %p size: %i\n", it->first, it->second);
        memory += it->second;
    }
    NN_LOG("Total not deallocated: %i\n", memory);
    SUCCEED();
}

}
}}

