﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_MediaPlayerTest.h"
#include <getopt.h>
#include "MediaPlayerSample.h"
#include "MediaPlayerObserver.h"
#include "MediaPlayerUtilities.h"
#include "HidHandler.h"
#include "HeapTracker.h"
#include <movie/Utils.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/util/util_ScopeExit.h>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>
#include <sys/time.h>

#include <nn/nifm.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <curl/curl.h>
#include <string>
#include <cinttypes>
#include <nn/fgm.h>
#include <nn/fgm/fgm_Debugger.h>
#include <nn/fgm/fgm_Result.public.h>

const nn::fgm::Module modules[] =
{
    nn::fgm::Module_Cpu,
    nn::fgm::Module_Gpu,
    nn::fgm::Module_Emc,
    nn::fgm::Module_SysBus,
    nn::fgm::Module_Mselect,
    nn::fgm::Module_Nvdec,
    nn::fgm::Module_Nvenc,
    nn::fgm::Module_Nvjpg,
};

#include <limits>
#include <cmath>

#ifdef MOUNT_SDCARD
    #include <nn/fs/fs_SdCardForDebug.h>
#endif

#include <nn/profiler.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

// Instantiate Globals
uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__(( aligned(4096) ));
uint8_t g_FsHeapBuffer[FsHeapSize];
nn::lmem::HeapHandle g_FsHeap;
bool USE_PROXY = false;
int PROXY_SETTING_PORT = 8080;
unsigned long PROXY_SETTING_AUTH = CURLAUTH_BASIC;

// If these two strings get too long, heap tracker will report a leak since std::string
// allocates space for the string from the heap and heap tracker prints statistics before
// these globals destruct. On 32-bit, these will always allocate and cause heap tracking noise.
std::string PROXY_SETTING_HOST = "";
std::string PROXY_SETTING_USERPWD = "";

bool USE_HEAP_TRACKING = true;
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize,
        nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

/* MediaPlayer Sample Usage */
static void Usage()
{
    NN_SDK_LOG( "\n ERROR:: Invalid options See Readme.MediaPlayerSample.txt \n");
}

/* Process startup setup */
extern "C" void nninitStartup()
{
    const size_t heapSize = 256 << 20;
    const size_t mallocSize = 128 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int* __errno()
{
    static int i = 0;
    return &i;
}

struct MediaPlayerTest : public ::testing::Test
{
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp() {}
    virtual void TearDown() {}
};

void MediaPlayerTest::SetUpTestCase()
{
    // Set allocator callback functions
    movie::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    // Donate memory for graphics driver to work in
    nv::SetGraphicsServiceName("nvdrv:t");
    nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
}

void MediaPlayerTest::TearDownTestCase()
{
    nv::FinalizeGraphics();

    // Print out heap tracking statistics
    NN_SDK_LOG("mm heap\n");
    MMHeap().OutputUsage();
    NN_SDK_LOG("core heap\n");
    CoreHeap().OutputUsage();
    NN_SDK_LOG("malloc heap\n");
    MallocHeap().OutputUsage();
    NN_SDK_LOG("new heap\n");
    NewHeap().OutputUsage();
}

void RunTest(bool enableProfiler) {
    bool isNetworkFile = false;

    nn::hid::InitializeDebugPad();

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    if (argc <= 1)
    {
        Usage();
        return;
    }

    bool playAudio = true;
    bool playVideo = true;
    float playRate = 1.0; //Normal;
    bool setLooping = false;
    bool invalidOptions = false;
    bool isVideoTrack = false;
    const char *configFile = NULL;
    movie::PlayerResMax maxPlayerRes = movie::PlayerResMax_720;
    movie::VideoDecodeMode videoDecodeMode = movie::VideoDecodeMode_NativeTexture;
    movie::OutputFormat videoOutputFormat = movie::OutputFormat_VideoColorAbgr;

    bool enableClockLog = false;
    bool readClock = false;
    bool returnAudioDataToClient = false;
    bool returnVideoDataToClient = true;
    nn::fgm::Request requestList[8];
    nn::Result result;
    int i;
    nn::fgm::Setting actual;


    //    while ((option = getopt(argc, argv, "avp:")) != -1)
    for (int options = 0; options < argc; options++)
    {
        if (!argv[options])
            continue;

        if (!strcmp(argv[options], "-a"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    playAudio = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-v"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    playVideo = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-r"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0.25")) || (!strcmp(argv[options], "0.5")) ||
                    (!strcmp(argv[options], "0.75")) || (!strcmp(argv[options], "1.0")) ||
                    (!strcmp(argv[options], "1.25")) || (!strcmp(argv[options], "1.5")) ||
                    (!strcmp(argv[options], "2.0")))
                {
                    playRate = atof(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-l"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    setLooping = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-c"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                configFile = argv[options];
                invalidOptions = false;
                continue;
            }
            break;
        }

        if (!strcmp(argv[options], "-clock"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    enableClockLog = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
        if (!strcmp(argv[options], "-return-audio-data-to-client"))
        {
            invalidOptions = false;
            returnAudioDataToClient = true;
        }
        //if (!strcmp(argv[options], "-return-video-data-to-client"))
        //{
        //    invalidOptions = false;
        //    returnVideoDataToClient = true;
        //}
    }

    if (invalidOptions == true)
    {
        Usage();
        return;
    }

    timeval playerTimeVal;
    int64_t startTime = 0ll;
    int64_t endTime = 0ll;

    char* inputFileName = inputFileName = argv[1];
    std::string path = inputFileName;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    bool sdcardMounted = false;
    if (token == "sdcard")
    {
#ifdef MOUNT_SDCARD
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if (resultSdcardMount.IsFailure())
        {
            NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                resultSdcardMount.GetModule(),
                resultSdcardMount.GetDescription());
            return;
        }
        sdcardMounted = true;
#else
        NN_SDK_LOG("\n SD card Not supported \n");
        return;
#endif
    }
    else if (token == "http" || token == "https")
    {
        isNetworkFile = true;
        nn::nifm::Initialize();
        nn::nifm::SubmitNetworkRequest();
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            NN_SDK_LOG("Network request on hold\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        if (!nn::nifm::IsNetworkAvailable())
        {
            NN_SDK_LOG("Network initialization failed\n");
            nn::nifm::CancelNetworkRequest();
            return;
        }
        else
        {
            nn::Result res = nn::socket::Initialize(g_SocketConfigWithMemory);
            if (res.IsFailure())
            {
                nn::nifm::CancelNetworkRequest();
                NN_SDK_LOG("nn::socket::Initialize failed!\n");
                return;
            }
        }

        CURLcode res = CURLE_OK;

        /* initialize using system malloc for libcurl for testing */
        res = curl_global_init(CURL_GLOBAL_WIN32);
        if (res != CURLE_OK)
        {
            NN_LOG("\nError! curl_global_init failed. Err: %d\n\n", res);
            return;
        }
    }
    nn::Result resultHostMount = nn::fs::MountHostRoot();
    if (resultHostMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription());
        return;
    }
    if (configFile && !ProcessConfigFile(configFile))
        return;
    /* Check whether media exists if it is a local file */
    if (!isNetworkFile && token != "file")
    {
        nn::fs::FileHandle fileHandle;
        fileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaPlayerSample\n", inputFileName);
            return;
        }
        else
        {
            nn::fs::CloseFile(fileHandle);
        }
    }

    /* initialize CPU mode */
    if (enableClockLog == true)
    {
        for (i = 0; i<8; i++){
            result = requestList[i].Initialize(modules[i], nn::fgm::Priority_Default);
            if (result.IsFailure()){
                NN_SDK_LOG("nn::fgm Initialize request failed for :%d module\n", i );
            }
            else
                readClock = true;
        }
    }

    NN_SDK_LOG("\n MediaPlayerSample:: Input File Name : %s\n", inputFileName);

    movie::PlayerConfig config;
    config.returnAudioDataToClient = returnAudioDataToClient;
    config.returnVideoDataToClient = returnVideoDataToClient;
    config.videoDecodeMode = videoDecodeMode;
    config.returnSyncedData = true;
    config.useClientMemoryForOutput = true;
    config.autoReleaseOutputBuffers = true;
    config.audioFormat = movie::OutputFormat_AudioPcm16;
    config.videoFormat = videoOutputFormat;
    config.maxResolution = maxPlayerRes;
    movie::MediaData bufferAlignment;
    int nv12YAlignment = 32;
    int nv12UvAlignment = 32;
    int pcmAlignment = 32;
    bufferAlignment.SetInt32("nv12-y-alignment", nv12YAlignment);
    bufferAlignment.SetInt32("nv12-uv-alignment", nv12UvAlignment);
    bufferAlignment.SetInt32("pcm-alignment", pcmAlignment);


    /*Set browser config, below are some sample values set to test this feature*/
    movie::BrowserConfig browserConfig;
    browserConfig.coreMask = 0x00;
    browserConfig.sfThreadPriority = 16;
    browserConfig.audioRendererThreadPriority = 16;
    browserConfig.videoRendererThreadPriority = 16;

    MediaPlayerObserver* mediaPlayerObserver = new MediaPlayerObserver;
    if (mediaPlayerObserver == NULL)
    {
        NN_SDK_LOG("\n Faile to Create MediaPlayerSample - Terminating PlayerSample \n");
        return;
    }
    mediaPlayerObserver->Initialize(&config);

    NN_SDK_LOG("\n MediaPlayerSample:: Create \n");
    movie::BrowserPlayer* player = nullptr;
    movie::BrowserPlayer::Create(&player, &config, &browserConfig);
    if (player == nullptr)
    {
        NN_SDK_LOG("\n MediaPlayerSample:: Failed to Create Player  - Terminating PlayerSample \n");
        mediaPlayerObserver->Finalize(&config);
        if (isNetworkFile == true)
        {
            nn::socket::Finalize();
            nn::nifm::CancelNetworkRequest();
        }
        else if (sdcardMounted == true)
        {
#ifdef MOUNT_SDCARD
            nn::fs::Unmount("sdcard");
#endif
        }
        else
        {
            nn::fs::UnmountHostRoot();
        }
        delete mediaPlayerObserver;
        EXPECT_NE(nullptr, player);
        return;
    }
    mediaPlayerObserver->SetMoviePlayer(player);

    HidHandler hidHandler(player, mediaPlayerObserver);
    hidHandler.Initialize();

    movie::Status movieStatus = movie::Status_Success;
    bool errorCleanUp = false;

    if (player != NULL)
    {
        NN_SDK_LOG("\n MediaPlayerSample:: Set SetObserver \n");
        movieStatus = player->SetObserver(mediaPlayerObserver);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Set SetObserver: Error : 0x%x  - Terminating PlayerSample \n", movieStatus);
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG("\n MediaPlayerSample :: Set Output Buffer Alignment \n");
        movieStatus = player->SetOutputBufferAlignment(&bufferAlignment);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Set Output Buffer Alignment Error : 0x%x \n", movieStatus);
        }

        NN_SDK_LOG("\n MediaPlayerSample :: Set data Source \n");
        movieStatus = player->SetDataSource(inputFileName);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Set data Source: Error : 0x%x  - Terminating PlayerSample \n", movieStatus);
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG("\n MediaPlayerSample:: Prepare \n");
        movieStatus = player->Prepare();
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Prepare: Error : 0x%x  - Terminating PlayerSample \n", movieStatus);
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: GetPlaybackDuration() + \n");
        int64_t trackDuration = 0;
        movieStatus = player->GetPlaybackDuration(&trackDuration);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetPlaybackDuration: Error : 0x%x \n", movieStatus);
        }

        trackDuration = trackDuration / 1000;
        NN_SDK_LOG("   GetPlaybackDuration (In msec) : %" PRId64 "\n", trackDuration);
        NN_SDK_LOG(" MediaPlayerSample:: Test API: GetPlaybackDuration() - \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: GetTrackCount() + \n");
        int32_t trackCount = 0;
        movieStatus = player->GetTrackCount(&trackCount);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackCount: Error : 0x%x  - Terminating PlayerSample \n", movieStatus);
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        NN_SDK_LOG("   GetTrackCount: %d \n", trackCount);
        NN_SDK_LOG(" MediaPlayerSample:: Test API: GetTrackCount() - \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: GetVideoDimensions() + \n");
        int32_t width = 0;
        int32_t height = 0;
        movieStatus = player->GetVideoDimensions(&width, &height);
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetVideoDimensions: Error : 0x%x, Detail: %s\n", movieStatus);
        }
        NN_SDK_LOG("   GetVideoDimensions: width: %d height: %d \n", width, height);
        NN_SDK_LOG(" MediaPlayerSample:: Test API: GetVideoDimensions() - \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: GetTrackType(), GetTrackInfo() + \n");

        for (int32_t track = 0; track < trackCount; track++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movie::TrackInfo trackInfo;
            movieStatus = player->GetTrackType(track, &trackType);
            if (movieStatus != movie::Status_Success)
            {
                NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus);
            }

            switch (trackType)
            {
            case movie::TrackType_Audio:
                NN_SDK_LOG("   GetTrackType() Track Index : %d Track Type: TrackType_Audio \n", track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if (movieStatus != movie::Status_Success)
                {
                    NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus);
                }
                else
                {
                    NN_SDK_LOG("\n GetTrackInfo() Track Index : %d Track Type: TrackType_Audio \n", track);
                    NN_SDK_LOG("                  mime        : %s \n", trackInfo.audioTrackInfo.mime);
                    NN_SDK_LOG("                  sampleRate  : %d \n", trackInfo.audioTrackInfo.sampleRate);
                    NN_SDK_LOG("                  channels    : %d \n", trackInfo.audioTrackInfo.channels);
                    NN_SDK_LOG("                  bitRate     : %d \n", trackInfo.audioTrackInfo.bitRate);
                    NN_SDK_LOG("                  codec       : 0x%x \n", trackInfo.audioTrackInfo.codecType);
                    NN_SDK_LOG("                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Video:
                isVideoTrack = true;
                NN_SDK_LOG("   GetTrackType() Track Index : %d Track Type: TrackType_Video \n", track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if (movieStatus != movie::Status_Success)
                {
                    NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus);
                }
                else
                {
                    NN_SDK_LOG("\n GetTrackInfo() Track Index : %d Track Type: TrackType_Video \n", track);
                    NN_SDK_LOG("                  mime        : %s \n", trackInfo.videoTrackInfo.mime);
                    NN_SDK_LOG("                  frameRate   : %d \n", trackInfo.videoTrackInfo.frameRate);
                    NN_SDK_LOG("                  height      : %d \n", trackInfo.videoTrackInfo.height);
                    NN_SDK_LOG("                  width       : %d \n", trackInfo.videoTrackInfo.width);
                    NN_SDK_LOG("                  bitRate     : %d \n", trackInfo.videoTrackInfo.bitRate);
                    NN_SDK_LOG("                  codec       : 0x%x \n", trackInfo.videoTrackInfo.codecType);
                    NN_SDK_LOG("                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Subtitle:
                NN_SDK_LOG("   GetTrackType() Track Index : %d Track Type: TrackType_Subtitle \n", track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if (movieStatus != movie::Status_Success)
                {
                    NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus);
                }
                else
                {
                    NN_SDK_LOG("\n GetTrackInfo() Track Index : %d Track Type: TrackType_Subtitle \n", track);
                    NN_SDK_LOG("                  mime        : %s \n", trackInfo.subtitleTrackInfo.mime);
                }
                break;

            case movie::TrackType_ClosedCaption:
                NN_SDK_LOG("   GetTrackType() Track Index : %d Track Type: TrackType_ClosedCaption \n", track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if (movieStatus != movie::Status_Success)
                {
                    NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus);
                }
                else
                {
                    NN_SDK_LOG("\n GetTrackInfo() Track Index : %d Track Type: TrackType_ClosedCaption \n", track);
                    NN_SDK_LOG("                  mime        : %s \n", trackInfo.closedCaptionTrackInfo.mime);
                }
                break;

            default:
                NN_SDK_LOG("   GetTrackType() Track Index : %d Track Type: TrackType_Unknown \n", track);
                break;
            }
        }
        NN_SDK_LOG(" MediaPlayerSample:: Test API: GetTrackType(), GetTrackInfo() - \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: SetPlaybackRate() + \n");
        NN_SDK_LOG("   MediaPlayerSample:: Set rate [PlaybackRate_1_25x] \n");
        float setPlaybackRate = 0.25f;
        player->SetPlaybackRate(setPlaybackRate);

        NN_SDK_LOG("   MediaPlayerSample:: Verify by GetPlaybackRate() \n");
        float verifyPlaybackRate = player->GetPlaybackRate();

        if (verifyPlaybackRate == setPlaybackRate)
        {
            NN_SDK_LOG("   MediaPlayerSample:: Test API: SetPlaybackRate() Success \n");
        }
        else
        {
            NN_SDK_LOG("   MediaPlayerSample:: Test API: SetPlaybackRate() Fail \n");
        }
        NN_SDK_LOG(" MediaPlayerSample:: Test API: SetPlaybackRate() - \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Test API: GetPlaybackRate() + \n");
        NN_SDK_LOG("   MediaPlayerSample:: Test API: GetPlaybackRate(): First Set [PlaybackRate_2x] rate \n");
        player->SetPlaybackRate(2.0f);

        float getPlaybackRate = player->GetPlaybackRate();

        if (getPlaybackRate == 2.0f)
        {
            NN_SDK_LOG("   GetPlaybackRate() : PlaybackRate_2x Success\n");
        }
        else
        {
            NN_SDK_LOG("   GetPlaybackRate(): Unknown Rate, Fail \n");
        }

        NN_SDK_LOG("   MediaPlayerSample:: Test API: GetPlaybackRate() reset playback to normal \n");
        player->SetPlaybackRate(1.0f);
        NN_SDK_LOG(" MediaPlayerSample:: Test API: GetPlaybackRate() - \n");

        /* Check whether Playback rate is set */
        float playbackRate = playRate;

        player->SetPlaybackRate(playbackRate);

        /* Play only selected track */
        for (int32_t track = 0; track < trackCount; track++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movieStatus = player->GetTrackType(track, &trackType);
            if (movieStatus != movie::Status_Success)
            {
                NN_SDK_LOG("\n MediaPlayerSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus);
            }
            switch (trackType)
            {
            case movie::TrackType_Audio:
                if (playAudio == false)
                {
                    movieStatus = player->DeSelectTrack(track);
                    if (movieStatus != movie::Status_Success)
                    {
                        NN_SDK_LOG("\n MediaPlayerSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus);
                    }
                }
                mediaPlayerObserver->SetAudioTrackNumber(track);
                break;

            case movie::TrackType_Video:
                if (playVideo == false)
                {
                    movieStatus = player->DeSelectTrack(track);
                    if (movieStatus != movie::Status_Success)
                    {
                        NN_SDK_LOG("\n MediaPlayerSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus);
                    }
                }
                mediaPlayerObserver->SetVideoTrackNumber(track);
                break;

            default:
                break;
            }
        }

        /* Set loop mode */
        if (setLooping == true)
        {
            movieStatus = player->SetLooping(setLooping);
            if (movieStatus != movie::Status_Success)
            {
                NN_SDK_LOG("\n MediaPlayerSample:: Failed to SetLooping: Error : 0x%x \n", movieStatus);
            }
        }

        NN_SDK_LOG("\n MediaPlayerSample:: start \n");
        movieStatus = player->Start();
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Start: Error : 0x%x  - Terminating PlayerSample \n", movieStatus);
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        if (isVideoTrack)
        {
            nn::os::WaitEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            nn::os::ClearEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            player->GetVideoDimensions(&width, &height);
            if ((config.returnVideoDataToClient == true) && (config.useClientMemoryForOutput == true))
            {
                size_t videoOutputBufferSize = width * height;
                size_t extraForAlignment = 1024 * 4;
                if (config.videoFormat == movie::OutputFormat_VideoColorAbgr )
                {
                    videoOutputBufferSize = videoOutputBufferSize * 4 + extraForAlignment;
                }
                else if (config.videoFormat == movie::OutputFormat_VideoColorNv12)
                {
                    videoOutputBufferSize = videoOutputBufferSize + (2 * ((width / 2) * (height / 2))) + extraForAlignment;
                }

                mediaPlayerObserver->m_VideoOutputBufferSize = videoOutputBufferSize;
            }

        }

        gettimeofday(&playerTimeVal, NULL);
        startTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        if (playAudio == false && playVideo == false)
        {
            mediaPlayerObserver->SetPlaybackComplete();
        }

        int64_t lastPlaybackPosition = 0;
        int64_t getPlaybackPosition = 0;
        while (!mediaPlayerObserver->IsPlaybackComplete())
        {


            if(enableProfiler)
                nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);

            getPlaybackPosition = 0;
            movieStatus = player->GetCurrentPlaybackPosition(&getPlaybackPosition);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
            }
            getPlaybackPosition = getPlaybackPosition / (1000 * 1000);
            if( ( (getPlaybackPosition % 5 ) == 0ll) && ( lastPlaybackPosition != getPlaybackPosition))
            {
                NN_SDK_LOG( "Track Duration : %" PRId64 " Seconds ", trackDuration / 1000);
                NN_SDK_LOG( " Playback Position : %" PRId64 " Seconds\n", getPlaybackPosition);
                lastPlaybackPosition = getPlaybackPosition;
            }

            hidHandler.Update();

            if( isVideoTrack )
            {
                int yOffset = 0;
                int uvOffset = 0;
                int ystride = 0;
                int nativeWidth = 0;
                int nativeHeight = 0;
                int cropWidth = 0;
                int cropHeight = 0;
                int colorSpace = 0;
                int32_t bufferIndex = -1;
                int32_t trackNumber = -1;
                int64_t presentationTimeUs = 0;
                uint32_t flags = 0;
                int32_t bufferIndexForProperties = -1;

                if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
                {
                    nn::os::LockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    uint32_t videoListSize = mediaPlayerObserver->m_VideoOutputBuffersList.size();

                    if( videoListSize > 0 )
                    {
                        bool videoAvailableForRendering = false;
                        MediaPlayerObserver::OutputBufferInfo videoBuffer;

                        videoBuffer = mediaPlayerObserver->m_VideoOutputBuffersList.front();
                        mediaPlayerObserver->m_VideoOutputBuffersList.erase( mediaPlayerObserver->m_VideoOutputBuffersList.begin());

                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );

                        bufferIndex = videoBuffer.bufferIndex;
                        trackNumber = videoBuffer.trackNumber;

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            bufferIndexForProperties = bufferIndex;
                        }
                        else
                        {
                            //make sure the BufferIndex is one of the valid ones
                            for( int i = 0; i < mediaPlayerObserver->m_videoBufferCount; ++i )
                            {
                                if( bufferIndex == mediaPlayerObserver->m_videoBufferIndexArray[ i ] )
                                {
                                    bufferIndexForProperties = bufferIndex;
                                    mediaPlayerObserver->m_PresentationIndex = bufferIndex;
                                    videoAvailableForRendering = true;
                                    break;
                                }
                            }
                        }

                        player->GetOutputBufferProperties(bufferIndexForProperties, &mediaPlayerObserver->m_BufferProperty);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-stride", &ystride);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-offset", &yOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-uv-offset", &uvOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("width", &nativeWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("height", &nativeHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-width", &cropWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-height", &cropHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-colorspace", &colorSpace);
                        if( mediaPlayerObserver->m_CropWidth != cropWidth || mediaPlayerObserver->m_CropHeight != cropHeight )
                        {
                            mediaPlayerObserver->m_CropWidth = cropWidth;
                            mediaPlayerObserver->m_CropHeight = cropHeight;
                            if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                            {
                                NN_SDK_LOG("Video resolution changed, ResizeTextures : WxH %dx%d ystride %d\n", cropWidth, cropHeight, ystride);
                                if( ( cropWidth > 0 ) && ( cropHeight > 0 ) )
                                {
                                    mediaPlayerObserver->ResizeTextures(cropWidth, cropHeight);
                                }
                            }
                        }

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            movie::Buffer buffer;
                            mediaPlayerObserver->m_PresentationIndex = (mediaPlayerObserver->m_PresentationIndex + 1) % mediaPlayerObserver->m_videoBufferCount;
                            buffer.SetDataAndCapacity(mediaPlayerObserver->m_videoBufferYUV[mediaPlayerObserver->m_PresentationIndex].Map(), mediaPlayerObserver->m_VideoOutputBufferSize);
                            buffer.SetRange(0, mediaPlayerObserver->m_VideoOutputBufferSize);
                            movie::Status movieStatus = player->FillOutputBuffer(trackNumber, bufferIndex, &buffer, &presentationTimeUs, &flags);
                            if( movieStatus == movie::Status_Success )
                            {
                                videoAvailableForRendering = true;
                            }
                        }

                        if( videoAvailableForRendering )
                        {
                            mediaPlayerObserver->DrawVideoNvn(mediaPlayerObserver->m_PresentationIndex, cropWidth, cropHeight, yOffset, uvOffset, ystride, colorSpace);
                        }
                    }
                    else
                    {
                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    }
                }
            }

            /* measure clock values */
            if (enableClockLog == true && readClock == true)
            {

                for (i = 0; i<8; i++)
                {
                    switch(modules[i]){
                    case nn::fgm::Module_Cpu:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_Cpu failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Cpu is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Gpu:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_Gpu failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Gpu is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Emc:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get Module_Emc request failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Emc  is :%u\n", actual);
                        break;
                    case nn::fgm::Module_SysBus:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_SysBus failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_SysBus  is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Mselect:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_Mselect failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Mselect is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Nvdec:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_Nvdec failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Nvdec is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Nvenc:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request Module_Nvenc failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Nvenc is :%u\n", actual);
                        break;
                    case nn::fgm::Module_Nvjpg:
                        result = requestList[i].Get(&actual);
                        if (result.IsFailure()){
                            NN_SDK_LOG("nn::fgm get request for Module_Nvjpg failed\n");
                        }
                        NN_SDK_LOG("nn::fgm clock value for Module_Nvjpg is :%u\n", actual);
                        break;
                    default:
                        NN_SDK_LOG("nn::fgm default case \n");

                    }

                }
                readClock = false;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_SDK_LOG("\n MediaPlayerSample:: Playback is complete! \n");

        NN_SDK_LOG("\n MediaPlayerSample:: Playback Stop \n");
        movieStatus = player->Stop();
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Stop: Error : 0x%x \n", movieStatus);
        }

        EXPECT_TRUE(movieStatus == movie::Status_Success);

        gettimeofday(&playerTimeVal, NULL);
        endTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        NN_SDK_LOG(" \n Track Duration : %" PRId64 " msec ******* Total Playback Time : %" PRId64 " msec", trackDuration, (endTime - startTime));

        player->Reset();
        movie::Player::Destroy(player);
        NN_SDK_LOG("\n MediaPlayerSample:: Player destroyed \n");
    }
    else
    {
        NN_SDK_LOG("\n MediaPlayerSample:: Could not create player \n");
    }

Exit_PlayerSample:

    if (errorCleanUp == true)
    {
        movieStatus = player->Stop();
        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Stop: Error : 0x%x \n", movieStatus);
            player->Reset();
        }
        movie::Player::Destroy(player);
        NN_SDK_LOG("\n MediaPlayerSample:: Player destroyed \n");
    }

    mediaPlayerObserver->Finalize(&config);
    if (enableClockLog == true)
    {
        for (i = 0; i<8; i++){
            requestList[i].Finalize();
        }
    }

    if (isNetworkFile == true)
    {
        nn::socket::Finalize();
        nn::nifm::CancelNetworkRequest();
        curl_global_cleanup();
    }
    else if (sdcardMounted == true)
    {
#ifdef MOUNT_SDCARD
        nn::fs::Unmount("sdcard");
#endif
    }
    nn::fs::UnmountHostRoot();
    delete mediaPlayerObserver;
}//NOLINT(impl/function_size)

TEST_F(MediaPlayerTest, NormalPlayback)
{
    { // We use an embedded scope to make sure local destructors have been called
      // before we print our final heap tracking statistics at the end of the test.

        int argc = nn::os::GetHostArgc();
        char** argv = nn::os::GetHostArgv();
        bool enableProfiler = false;

        for (int options = 0; options < argc; options++)
        {
            if (!argv[options])
                continue;

            if (!strcmp(argv[options], "-enable-profiler"))
                enableProfiler = true;
        }

        if(enableProfiler)
        {
            NN_SDK_LOG("\n MediaPlayerTest:: NormalPlayback Profiler Enabled\n");
            std::vector<char> profiler_buffer__(nn::profiler::MinimumBufferSize);
            nn::profiler::Initialize(profiler_buffer__.data(), profiler_buffer__.size());
            RunTest(true);
            nn::profiler::Finalize();
        }
        else
        {
            NN_SDK_LOG("\n MediaPlayerTest:: NormalPlayback\n");
            RunTest(false);
        }
    } // End of embedded scope
}
