﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
//Siglo networking is unstable, causing many false problems
//#define ENABLE_SOCKET
#include <cinttypes>
#include <cstdarg>
#include <cstring>
#include <memory>
#include <sstream>
#include <string>
#include <vector>
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunused-local-typedef"
#include <boost/algorithm/string.hpp>
#include <boost/fusion/include/define_struct.hpp>
#include <boost/spirit/include/qi_alternative.hpp>
#include <boost/spirit/include/qi_as.hpp>
#include <boost/spirit/include/qi_char.hpp>
#include <boost/spirit/include/qi_char_class.hpp>
#include <boost/spirit/include/qi_eol.hpp>
#include <boost/spirit/include/qi_kleene.hpp>
#include <boost/spirit/include/qi_list.hpp>
#include <boost/spirit/include/qi_lit.hpp>
#include <boost/spirit/include/qi_no_case.hpp>
#include <boost/spirit/include/qi_omit.hpp>
#include <boost/spirit/include/qi_optional.hpp>
#include <boost/spirit/include/qi_parse.hpp>
#include <boost/spirit/include/qi_plus.hpp>
#include <boost/spirit/include/qi_sequence.hpp>
#include <boost/spirit/include/qi_copy.hpp>
#pragma clang diagnostic pop
#include <mm_MemoryManagement.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <movie/Player.h>
#include <nn/os.h>
#include <nn/os/os_TimerEvent.h>
#include <nnt/nntest.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_SOCKET
#include <nn/nifm.h>
#include <nn/socket.h>
#endif

// System heap setup
namespace // 'unnamed'
{
nn::lmem::HeapHandle g_FsHeap;
}

void *FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void *p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

void FsCreateHeap()
{
    const int heapSize = 512 * 1024;
    static uint8_t s_FsHeapBuffer[heapSize];
    g_FsHeap = nn::lmem::CreateExpHeap(s_FsHeapBuffer, sizeof(s_FsHeapBuffer), nn::lmem::CreationOption_DebugFill);
}

void FsDestroyHeap()
{
    nn::lmem::DestroyExpHeap(g_FsHeap);
    g_FsHeap = 0;
}

namespace // 'unnamed'
{
nn::mem::StandardAllocator g_MultimediaAllocator;
}

void MMCreateAllocator()
{
    const int allocatorSize = 512 << 20;
    static char s_MMAllocatorBuffer[allocatorSize];
    // Fill memory with a garbage value to help detect uninitialized variable bugs.
    // What's the equivalent to CafeSDK's OSBlockSet?
    std::memset(s_MMAllocatorBuffer, 0xC3C3C3C3, sizeof(s_MMAllocatorBuffer));
    g_MultimediaAllocator.Initialize(s_MMAllocatorBuffer, sizeof(s_MMAllocatorBuffer));
}

void MMDestroyAllocator()
{
    g_MultimediaAllocator.Finalize();
}

void *MMAlloc(size_t size, size_t alignment, void*)
{
    return g_MultimediaAllocator.Allocate(size, alignment);
}

void MMFree(void *p, void*)
{
    g_MultimediaAllocator.Free(p);
}

void *MMRealloc(void *p, size_t newSize, void*)
{
    return g_MultimediaAllocator.Reallocate(p, newSize);
}

// Process startup setup
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    nn::Result result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    uintptr_t raw_address;
    const size_t mallocSize = 32 << 20;
    result = nn::os::AllocateMemoryBlock(&raw_address, mallocSize);
    void * const address = reinterpret_cast<void*>(raw_address);
    // Fill memory with a garbage value to help detect uninitialized variable bugs.
    // What's the equivalent to CafeSDK's OSBlockSet?
    std::memset(address, 0xC3C3C3C3, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(address, mallocSize);
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int *__errno()
{
    static int i = 0;
    return &i;
}

namespace // 'unnamed'
{
movie::Player *CreatePlayer()
{
    movie::Player *ret = 0;
    movie::PlayerConfig conf{};
    conf.returnAudioDataToClient = false;
    conf.returnVideoDataToClient = false;
    conf.returnSyncedData = false;
    conf.audioFormat = movie::OutputFormat_AudioPcm16;
    conf.videoFormat = movie::OutputFormat_VideoColorNv12;
    conf.useNativeBuffers = false;
    EXPECT_TRUE((movie::Status_Success == movie::Player::Create(&ret, &conf)));
    return ret;
}
}

BOOST_FUSION_DEFINE_STRUCT
(
    (test),
    command,
    (std::string, d)
    (std::vector<std::string>, p)
)

// Test "fixture"
class ScriptedTest : public ::testing::Test, public movie::PlayerObserver
{
public:
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp();
    virtual void TearDown();

    void log(const char *format, ...);

    void WaitForPlaybackCompletion(nn::TimeSpan waitTime);
    void WaitForSeekCompletion(nn::TimeSpan waitTime);
    void ResetPlaybackCompletion();

    virtual void OnError(movie::Status status) NN_NOEXCEPT;
    virtual void OnStateChange(movie::PlayerState state) NN_NOEXCEPT;
    virtual void OnBufferingUpdate(float startTime, float endTime) NN_NOEXCEPT;
    virtual void OnAudioOutputFrameAvailable(movie::AudioFrameInfo *frameInfo) NN_NOEXCEPT;
    virtual void OnVideoOutputFrameAvailable(movie::VideoFrameInfo *frameInfo) NN_NOEXCEPT;
    virtual int32_t OnHttpRequest(CURL *easyRequest, const char *uri) NN_NOEXCEPT;
    virtual int32_t OnMultiConfig(CURLM *multiRequest, const char *uri) NN_NOEXCEPT;
    virtual int32_t OnHttpResponse(CURLM *multiRequest, const char *uri) NN_NOEXCEPT;
    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType) NN_NOEXCEPT;
    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index) NN_NOEXCEPT;
    virtual void OnFormatChanged(movie::TrackType eTrackType) NN_NOEXCEPT;

    static bool sdcardMounted;
    static bool hostMounted;
    static const char *scriptPath;
    static const char *baseDirectory;
    static std::vector<test::command> commands;
    const nn::os::Tick startTick = nn::os::GetSystemTick();
    nn::os::TimerEvent timer{nn::os::EventClearMode_AutoClear};
    std::unique_ptr<movie::Player, decltype(&movie::Player::Destroy)> player{nullptr, &movie::Player::Destroy};
    const int64_t seekDelayMS = 1000;

private:
    nn::os::Event playBackCompleteEvent{nn::os::EventClearMode_AutoClear};
    nn::os::Event seekCompleteEvent{nn::os::EventClearMode_AutoClear};
};

bool ScriptedTest::sdcardMounted = false;
bool ScriptedTest::hostMounted = false;
const char *ScriptedTest::scriptPath = 0;
const char *ScriptedTest::baseDirectory = 0;
std::vector<test::command> ScriptedTest::commands;
#ifdef ENABLE_SOCKET
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

void ScriptedTest::SetUpTestCase()
{
    FsCreateHeap();
    MMCreateAllocator();
    NN_SDK_LOG("Calling nv::mm::SetAllocator\n");
    nv::mm::SetAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    NN_SDK_LOG("Calling nv::SetGraphicsAllocator\n");
    nv::SetGraphicsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    nv::SetGraphicsServiceName("nvdrv:t");
    NN_SDK_LOG("Calling nv::SetGraphicsDevtoolsAllocator\n");
    nv::SetGraphicsDevtoolsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    const int mmFirmwareMemorySize = 8 << 20;
    NN_ALIGNAS(4096) static char s_mmFirmwareMemory[mmFirmwareMemorySize];
    // Fill memory with a garbage value to help detect uninitialized variable bugs.
    // What's the equivalent to CafeSDK's OSBlockSet?
    std::memset(s_mmFirmwareMemory, 0xC3C3C3C3, sizeof(s_mmFirmwareMemory));
    NN_SDK_LOG("Calling nv::InitializeGraphics\n");
    nv::InitializeGraphics(s_mmFirmwareMemory, sizeof(s_mmFirmwareMemory));
    const int argc = nn::os::GetHostArgc();
    const char * const * const argv = nn::os::GetHostArgv();
    if(argc <= 1)
    {
        NN_SDK_LOG("\n MediaPlayerScripted::Usage - %s  <script-file-path> [base-directory]\n", argv[0]);
        return;
    }
    scriptPath = argv[1];
    if(argc > 2)
    {
        baseDirectory = argv[2];
    }
#ifdef MOUNT_SDCARD
    nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
    if(resultSdcardMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
            resultSdcardMount.GetModule(),
            resultSdcardMount.GetDescription());
        return;
    }
    sdcardMounted = true;
#endif

#ifdef ENABLE_SOCKET
    nn::nifm::Initialize();
    nn::nifm::SubmitNetworkRequest();
    while(nn::nifm::IsNetworkRequestOnHold())
    {
        NN_SDK_LOG("Network request on hold\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    if(!nn::nifm::IsNetworkAvailable())
    {
        NN_SDK_LOG("Network initialization failed\n");
        nn::nifm::CancelNetworkRequest();
        return;
    }
    else
    {
        nn::Result res = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                                nn::socket::DefaultSocketMemoryPoolSize,
                                                nn::socket::DefaultSocketAllocatorSize,
                                                nn::socket::DefaultConcurrencyLimit);
        if(res.IsFailure())
        {
            nn::nifm::CancelNetworkRequest();
            NN_SDK_LOG("nn::socket::Initialize failed!\n");
            return;
        }
    }
    NN_SDK_LOG("Network initialized (supposedly)\n");
#endif

    const nn::Result resultHostMount = nn::fs::MountHostRoot();
    if(resultHostMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription());
        return;
    }
    hostMounted = true;

    std::vector<char> data;
    const std::string token(scriptPath, 0, std::string(scriptPath).find(':'));
    if((token == "http") || (token == "https") || (token == "file"))
    {
    }
    // Check whether media exists if it is a local file
    else
    {
        nn::fs::FileHandle fileHandle{};
        const nn::Result result = nn::fs::OpenFile(&fileHandle, scriptPath, nn::fs::OpenMode_Read);
        if(result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s\n\n Exiting MediaPlayerScripted Test\n", scriptPath);
        }
        else
        {
            int64_t fileSize = 0;
            nn::fs::GetFileSize(&fileSize, fileHandle);
            data.resize(fileSize);
            nn::fs::ReadFile(fileHandle, 0, &data[0], fileSize);
            nn::fs::CloseFile(fileHandle);
            NN_SDK_LOG("\n DUMP (%s):\n%.*s\n\n", scriptPath, int(fileSize), &data[0]);
        }
    }
    if(!data.empty())
    {
        namespace qi = boost::spirit::qi;
        auto begin = data.begin();
        auto end = data.end();
        auto directive = qi::copy(qi::no_case[qi::string("CREATE")
                                                | qi::string("DESTROY")
                                                | qi::string("EXIT")
                                                | qi::string("OPEN")
                                                | qi::string("URI")
                                                | qi::string("PREPARE")
                                                | qi::string("PLAY")
                                                | qi::string("SEEK")
                                                | qi::string("WAIT")
                                                | qi::string("ECHO")
                                                | qi::string("CAPTURE")
                                                | qi::string("VOLUME")
                                                | qi::string("SPEED")
                                                | qi::string("RESET")
                                                | qi::string("STOP")
                                                | qi::string("PAUSE")
                                                | qi::string("LOOP")
                                                | qi::string("TRACK")
                                                | qi::string("POSITION")]);
        auto parameter = qi::copy(qi::as_string[+qi::graph]);
        auto action = qi::copy(directive >> -(qi::omit[+qi::blank] >> *(parameter % qi::omit[+qi::blank])));
        auto comment = qi::copy(qi::char_('#') >> *(qi::graph | qi::blank));
        auto line = qi::copy(qi::omit[*qi::blank] >> -(action | qi::omit[comment]));
        auto program = qi::copy(line % qi::eol);

        const bool r = qi::parse(begin, end, program, commands);

        EXPECT_EQ(begin, end); // Didn't fully match
        EXPECT_TRUE(r); // Unsuccessful parse
        for(auto i : commands)
        {
            NN_SDK_LOG("%s", i.d.c_str());
            if(!i.p.empty())
            {
                NN_SDK_LOG("(");
            }
            for(std::size_t p = 0; p < i.p.size(); ++p)
            {
                if(p)
                {
                    NN_SDK_LOG(", ");
                }
                NN_SDK_LOG("%s", i.p[p].c_str());
            }
            if(!i.p.empty())
            {
                NN_SDK_LOG(")");
            }
            NN_SDK_LOG("\n");
        }
    }
    NN_SDK_LOG("\n MediaPlayerScripted:: Input File Name : %s\n", scriptPath);
}//NOLINT(impl/function_size)

void ScriptedTest::TearDownTestCase()
{
#ifdef ENABLE_SOCKET
    nn::socket::Finalize();
    nn::nifm::CancelNetworkRequest();
#endif

#ifdef MOUNT_SDCARD
    nn::fs::Unmount("sdcard");
#endif

    if(hostMounted)
        nn::fs::UnmountHostRoot();

    MMDestroyAllocator();
    FsDestroyHeap();
}

void ScriptedTest::SetUp()
{
}

void ScriptedTest::TearDown()
{
    player.reset();
    NN_SDK_LOG("\n MediaPlayerScripted:: Player destroyed\n");
}

void ScriptedTest::WaitForPlaybackCompletion(nn::TimeSpan waitTime)
{
    if(playBackCompleteEvent.TimedWait(waitTime))
    {
        ResetPlaybackCompletion();
    }
    else
    {
        log(" timed out waiting for playback completion\n");
    }
}

void ScriptedTest::WaitForSeekCompletion(nn::TimeSpan waitTime)
{
    if(seekCompleteEvent.TimedWait(waitTime))
    {
        seekCompleteEvent.Clear();
    }
    else
    {
        log(" timed out waiting for seek completion\n");
    }
}

void ScriptedTest::ResetPlaybackCompletion()
{
    playBackCompleteEvent.Clear();
}

// Called when there is any error
void ScriptedTest::OnError(movie::Status status) NN_NOEXCEPT
{
    NN_UNUSED(status);
    log(" OnError\n");
}

const char *PlayerStateToStr(movie::PlayerState state)
{
    const char *ret = "PlayerState UNKNOWN";
    switch(state)
    {
    case movie::PlayerState_UnInitialized:
        ret = "PlayerState_UnInitialized";
        break;
    case movie::PlayerState_Initialized:
        ret = "PlayerState_Initialized";
        break;
    case movie::PlayerState_Preparing:
        ret = "PlayerState_Preparing";
        break;
    case movie::PlayerState_Prepared:
        ret = "PlayerState_Prepared";
        break;
    case movie::PlayerState_Started:
        ret = "PlayerState_Started";
        break;
    case movie::PlayerState_Stopped:
        ret = "PlayerState_Stopped";
        break;
    case movie::PlayerState_Paused:
        ret = "PlayerState_Paused";
        break;
    case movie::PlayerState_Seeking:
        ret = "PlayerState_Seeking";
        break;
    case movie::PlayerState_SeekCompleted:
        ret = "PlayerState_SeekCompleted";
        break;
    case movie::PlayerState_PlaybackCompleted:
        ret = "PlayerState_PlaybackCompleted";
        break;
    case movie::PlayerState_Error:
        ret = "PlayerState_Error";
        break;
    default:
        break;
    }
    return ret;
}

// Called when player state changes
void ScriptedTest::OnStateChange(movie::PlayerState state) NN_NOEXCEPT
{
    const char * const stateStr = PlayerStateToStr(state);

    log(" %s\n", stateStr);
    switch(state)
    {
    case movie::PlayerState_PlaybackCompleted:
        playBackCompleteEvent.Signal();
        break;
    case movie::PlayerState_SeekCompleted:
        seekCompleteEvent.Signal();
        break;
    default:
        break;
    }
}

// Called when the user pressed Play/Pause or the player rebuffers
void ScriptedTest::OnBufferingUpdate(float startTime, float endTime) NN_NOEXCEPT
{
    NN_UNUSED(startTime);
    NN_UNUSED(endTime);
    log(" OnBufferingUpdate\n");
}

// Called when decoded audio data is available and client has requested data back
void ScriptedTest::OnAudioOutputFrameAvailable(movie::AudioFrameInfo *frameInfo) NN_NOEXCEPT
{
    NN_UNUSED(frameInfo);
    log(" OnAudioOutputFrameAvailable\n");
}

// Called when decoded video data is available and client has requested data back
void ScriptedTest::OnVideoOutputFrameAvailable(movie::VideoFrameInfo *frameInfo) NN_NOEXCEPT
{
    NN_UNUSED(frameInfo);
    log(" OnVideoOutputFrameAvailable\n");
}

// Called before curl_easy_perform() for an HTTP request. Modify any CURL option (proxy, cookies, authentication…) and return 0 to proceed
int32_t ScriptedTest::OnHttpRequest(CURL *easyRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(easyRequest);
    NN_UNUSED(uri);
    log(" OnHttpRequest\n");
    return 0;
}

// Called before curl_multi_perform() for an HTTP request. Modify any CURLM option (sockets cache…) and return 0 to proceed
int32_t ScriptedTest::OnMultiConfig(CURLM *multiRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(multiRequest);
    NN_UNUSED(uri);
    log(" OnMultiConfig\n");
    return 0;
}

// Called when an error was detected on an HTTP request managed with curl.
int32_t ScriptedTest::OnHttpResponse(CURLM *multiRequest, const char *uri) NN_NOEXCEPT
{
    NN_UNUSED(multiRequest);
    NN_UNUSED(uri);
    log(" OnHttpResponse\n");
    return 0;
}

// Called when client provided memory is used for output buffers.
void ScriptedTest::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index)  NN_NOEXCEPT
{
    NN_UNUSED(trackNumber);
    NN_UNUSED(eTrackType);
    NN_UNUSED(index);
    NN_SDK_LOG("\n OnOutputBufferAvailable with index\n");
}

// Called when client provided memory is used for output buffers.
void ScriptedTest::OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType) NN_NOEXCEPT
{
    NN_UNUSED(trackNumber);
    NN_UNUSED(eTrackType);
    NN_SDK_LOG("\n OnOutputBufferAvailable\n");
}

// Called when audio or video output format is changed.
void ScriptedTest::OnFormatChanged(movie::TrackType eTrackType) NN_NOEXCEPT
{
    NN_UNUSED(eTrackType);
    NN_SDK_LOG("\n OnFormatChanged\n");
}

void ScriptedTest::log(const char *format, ...)
{
    const nn::os::Tick tick = nn::os::GetSystemTick() - startTick;
    const nn::TimeSpan time = nn::os::ConvertToTimeSpan(tick);
    const int64_t hours = time.GetHours();
    const int64_t minutes = time.GetMinutes() % 60;
    const int64_t seconds = time.GetSeconds() % 60;
    const int64_t milliseconds = time.GetMilliSeconds() % 1000;
    NN_UNUSED(hours);
    NN_UNUSED(minutes);
    NN_UNUSED(seconds);
    NN_UNUSED(milliseconds);
    NN_SDK_LOG("[%" PRId64 ":%02" PRId64 ":%02" PRId64 ".%03" PRId64 "]", hours, minutes, seconds, milliseconds);
    va_list args;
    va_start(args, format);
    NN_SDK_VLOG(format, args);
    va_end(args);
}

bool timespec_to_us(const std::string& timespec, int64_t *us, const int64_t current_time)
{
    bool ret = false;
    std::istringstream stream(timespec);
    int64_t time = 0;
    int peek = stream.peek();
    bool has_plus_sign = peek == '+';
    if(stream >> time)
    {
        int64_t offset = 0;
        if(has_plus_sign || (time < 0))
        {
            offset = current_time;
        }
        std::string suffix;
        int64_t multiplier = 1000; // assumes milliseconds
        if(stream >> suffix)
        {
            if(boost::iequals(suffix, "f"))
            {
                ///@todo need frame rate to convert frames to microseconds
            }
            else if(boost::iequals(suffix, u8"μs") || boost::iequals(suffix, "us"))
            {
                multiplier = 1;
            }
            else if(boost::iequals(suffix, "ms"))
            {
                // Nothing to do
            }
            else if(boost::iequals(suffix, "s"))
            {
                multiplier = 1000 * 1000;
            }
            else
            {
                multiplier = 0;
            }
        }
        if(multiplier)
        {
            *us = offset + time*multiplier;
            ret = true;
        }
    }
    else
    {
        NN_SDK_LOG("Couldn't parse \"%s\"", timespec.c_str());
    }
    return ret;
}

bool volumespec_to_float(const std::string& volumespec, float *volume, const float current_volume)
{
    bool ret = false;
    std::istringstream stream(volumespec);
    float vol;
    int peek = stream.peek();
    bool has_plus_sign = peek == '+';
    if(stream >> vol)
    {
        float offset = 0;
        if(has_plus_sign || (vol < 0))
        {
            offset = current_volume;
        }
        *volume = offset + vol;
        ret = true;
    }
    else
    {
        NN_SDK_LOG("Couldn't parse \"%s\"", volumespec.c_str());
    }
    return ret;
}

bool playbackrate_to_float(const movie::PlaybackRate rate, float *speed)
{
    bool ret = true;
    switch(rate)
    {
    case movie::PlaybackRate_0_25x:
        *speed = 0.25f;
        break;
    case movie::PlaybackRate_0_5x:
        *speed = 0.5f;
        break;
    case movie::PlaybackRate_0_75x:
        *speed = 0.75f;
        break;
    case movie::PlaybackRate_Normal:
        *speed = 1.0f;
        break;
    case movie::PlaybackRate_1_25x:
        *speed = 1.25f;
        break;
    case movie::PlaybackRate_1_5x:
        *speed = 1.5f;
        break;
    case movie::PlaybackRate_2x:
        *speed = 2.0f;
        break;
    default:
        ret = false;
        break;
    }
    return ret;
}

bool speedspec_to_enum(const std::string& speedspec, movie::PlaybackRate *rate, const movie::PlaybackRate current_rate, float *speed_out)
{
    bool ret = false;
    std::istringstream stream(speedspec);
    float speed;
    int peek = stream.peek();
    bool has_plus_sign = peek == '+';
    if(stream >> speed)
    {
        float offset = 0;
        if(has_plus_sign || (speed < 0))
        {
            float current_speed;
            if(playbackrate_to_float(current_rate, &current_speed))
            {
                offset = current_speed;
            }
            else
            {
                NN_SDK_LOG("Couldn't identify playback rate \"%d\" (new enumerator?)", current_rate);
            }
        }
        *speed_out = offset + speed;
        if(*speed_out < 0.5f)
        {
            *rate = movie::PlaybackRate_0_25x;
        }
        else if(*speed_out < 0.75f)
        {
            *rate = movie::PlaybackRate_0_5x;
        }
        else if(*speed_out < 1.0f)
        {
            *rate = movie::PlaybackRate_0_75x;
        }
        else if(*speed_out < 1.25f)
        {
            *rate = movie::PlaybackRate_Normal;
        }
        else if(*speed_out < 1.5f)
        {
            *rate = movie::PlaybackRate_1_25x;
        }
        else if(*speed_out < 2.0f)
        {
            *rate = movie::PlaybackRate_1_5x;
        }
        else
        {
            *rate = movie::PlaybackRate_2x;
        }
        playbackrate_to_float(*rate, speed_out);
        ret = true;
    }
    else
    {
        NN_SDK_LOG("Couldn't parse \"%s\"", speedspec.c_str());
    }
    return ret;
}

bool boolean_to_bool(const std::string& boolean, bool *result)
{
    bool ret = false;
    if(boost::iequals(boolean, "on") || boost::iequals(boolean, "1") || boost::iequals(boolean, "yes") || boost::iequals(boolean, "true"))
    {
        *result = true;
        ret = true;
    }
    else if(boost::iequals(boolean, "off") || boost::iequals(boolean, "0") || boost::iequals(boolean, "no") || boost::iequals(boolean, "false"))
    {
        *result = false;
        ret = true;
    }
    return ret;
}

bool trackspec_to_int(const std::string& trackspec, int *track)
{
    bool ret = false;
    std::istringstream stream(trackspec);
    int trackindex;
    if(stream >> trackindex)
    {
        *track = trackindex;
        ret = true;
    }
    else
    {
        NN_SDK_LOG("Couldn't parse \"%s\"", trackspec.c_str());
    }
    return ret;
}

TEST_F(ScriptedTest, RunScript)
{
    for(auto c : commands)
    {
        const char * const errorhdrstr = "[error] ";
        const char * const nullplayerstr = "CREATE must come before ";
        const char * const needsparameterstr = " must have exactly one parameter (which cannot contain whitespace)";
        const char * const noparameterstr = " must have no parameters";
        const char * const needstwoparametersstr = " must have exactly two parameters";
        const char * const maxoneparameterstr = " must have at most one parameter";
        const char * const invalidtimestr = "\" is not recognized as a valid time";
        const char * const invalidbooleanstr = "\" is not recognized as a valid boolean";
        const char * const invalidtrackstr = "\" is not recognized as a valid track index";
        if(boost::iequals(c.d, "CREATE"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            player.reset();
            log(" CREATE\n");
            player.reset(CreatePlayer());
            EXPECT_TRUE((movie::Status_Success == player->SetObserver(this)));
            movie::PlayerObserver *observer = 0;
            EXPECT_TRUE((movie::Status_Success == player->GetObserver(&observer)));
            EXPECT_EQ(observer, this);
        }
        else if(boost::iequals(c.d, "DESTROY"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            player.reset();
            log(" DESTROY\n");
        }
        else if(boost::iequals(c.d, "EXIT"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            log(" EXIT\n");
            break;
        }
        else if(boost::iequals(c.d, "OPEN"))
        {
            ASSERT_EQ(c.p.size(), 1) << errorhdrstr << c.d.c_str() << needsparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            std::string path;
            // Only prepend the base directory if there's no colon present, which would indicate a full path
            if(baseDirectory && (c.p[0].find(':') == c.p[0].npos) && c.p[0].find('/'))
            {
                path = baseDirectory;
                path += '/';
            }
            path += c.p[0];
            log(" OPEN(%s)\n", path.c_str());
            EXPECT_TRUE((movie::Status_Success == player->SetDataSource(path.c_str())));
            EXPECT_TRUE((movie::Status_Success == player->Prepare()));
        }
        else if(boost::iequals(c.d, "URI"))
        {
            ASSERT_EQ(c.p.size(), 1) << errorhdrstr << c.d.c_str() << needsparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            std::string path;
            // Only prepend the base directory if there's no colon present, which would indicate a full path
            if(baseDirectory && (c.p[0].find(':') == c.p[0].npos) && c.p[0].find('/'))
            {
                path = baseDirectory;
                path += '/';
            }
            path += c.p[0];
            log(" OPEN(%s)\n", path.c_str());
            EXPECT_TRUE((movie::Status_Success == player->SetDataSource(path.c_str())));
        }
    else if(boost::iequals(c.d, "PREPARE"))
    {
        ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
        ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
        EXPECT_TRUE((movie::Status_Success == player->Prepare()));
        log(" PREPARE\n");
    }
        else if(boost::iequals(c.d, "PLAY"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            log(" PLAY\n");
            ResetPlaybackCompletion();
            EXPECT_TRUE((movie::Status_Success == player->Start()));
        }
        else if(boost::iequals(c.d, "SEEK"))
        {
            ASSERT_EQ(c.p.size(), 1) << errorhdrstr << c.d.c_str() << needsparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            int64_t us;
            int64_t current_time = 0;
            EXPECT_TRUE((movie::Status_Success == player->GetCurrentPlaybackPosition(&current_time)));
            ASSERT_TRUE(timespec_to_us(c.p[0], &us, current_time)) << errorhdrstr << '\"' << c.p[0].c_str() << invalidtimestr;
            log(" SEEK(%" PRId64 ")\n", us);
            EXPECT_TRUE((movie::Status_Success == player->SeekTo(us)));
        }
        else if(boost::iequals(c.d, "WAIT"))
        {
            ASSERT_LE(c.p.size(), 1) << errorhdrstr << c.d.c_str() << maxoneparameterstr;
            if(c.p.size() == 1)
            {
                int64_t us;
                ASSERT_TRUE(timespec_to_us(c.p[0], &us, 0)) << errorhdrstr << '\"' << c.p[0].c_str() << invalidtimestr;
                ASSERT_GT(us, 0) << errorhdrstr << " Can't " << c.d.c_str() << " a negative interval";
                log(" WAIT(%" PRId64 ")\n", us);
                timer.StartOneShot(nn::TimeSpan::FromMicroSeconds(us));
                timer.Wait();
            }
            else if(c.p.empty())
            {
                ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str() << " (without parameters)";
                ///@todo Needs to be more generic
                int64_t duration = 0;
                EXPECT_TRUE((movie::Status_Success == player->GetPlaybackDuration(&duration)));
                int64_t current_time = 0;
                EXPECT_TRUE((movie::Status_Success == player->GetCurrentPlaybackPosition(&current_time)));
                movie::PlayerState state{};
                EXPECT_TRUE((movie::Status_Success == player->GetState(&state)));
                log(" WAIT ");
                if(state == movie::PlayerState_Seeking)
                {
                    NN_SDK_LOG("(till seek is complete)\n");
                    WaitForSeekCompletion(nn::TimeSpan::FromMilliSeconds(seekDelayMS));
                }
                else
                {
                    movie::PlaybackRate current_rate = movie::PlaybackRate_Normal;
                    EXPECT_TRUE((movie::Status_Success == player->GetPlaybackRate(&current_rate)));
                    float speed;
                    playbackrate_to_float(current_rate, &speed);
                    // Time out waiting at one second past the remaining duration, adjusted for playback rate.
                    const int64_t actualDuration = int64_t((duration - current_time) / double(speed)) + int64_t(1000000);
                    NN_SDK_LOG("(till playback is complete, max: %" PRId64 ")\n", actualDuration);
                    WaitForPlaybackCompletion(nn::TimeSpan::FromMicroSeconds(actualDuration));
                }
            }
        }
        else if(boost::iequals(c.d, "ECHO"))
        {
            if(!c.p.empty())
            {
                log("");
                for(auto p : c.p)
                {
                    NN_SDK_LOG(" %s", p.c_str());
                }
                NN_SDK_LOG("\n");
            }
            else
            {
                ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str() << " (without parameters)";
                int64_t current_time = 0;
                EXPECT_TRUE((movie::Status_Success == player->GetCurrentPlaybackPosition(&current_time)));
                int64_t duration = 0;
                EXPECT_TRUE((movie::Status_Success == player->GetPlaybackDuration(&duration)));
                log(" [state] At %d of %d\n", current_time, duration);
                int32_t width = 0;
                int32_t height = 0;
                EXPECT_TRUE((movie::Status_Success == player->GetVideoDimensions(&width, &height)));
                log(" [state] Video: %d x %d\n", width, height);
                float volume = -1.0f;
                EXPECT_TRUE((movie::Status_Success == player->GetVolume(&volume)));
                log(" [state] Volume: %f\n", volume);
                movie::PlaybackRate current_rate = movie::PlaybackRate_Normal;
                EXPECT_TRUE((movie::Status_Success == player->GetPlaybackRate(&current_rate)));
                float speed;
                playbackrate_to_float(current_rate, &speed);
                log(" [state] Speed: %f\n", speed);
            }
        }
        else if(boost::iequals(c.d, "CAPTURE"))
        {
            log(" CAPTURE unsupported\n");
            ///@todo capture implementation
        }
        else if(boost::iequals(c.d, "VOLUME"))
        {
            ASSERT_LE(c.p.size(), 1) << errorhdrstr << c.d.c_str() << maxoneparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            float current_volume = 0;
            EXPECT_TRUE((movie::Status_Success == player->GetVolume(&current_volume)));
            if(c.p.size() == 1)
            {
                float vol;
                ASSERT_TRUE(volumespec_to_float(c.p[0], &vol, current_volume));
                log(" VOLUME(%f)\n", vol);
                EXPECT_TRUE((movie::Status_Success == player->SetVolume(vol)));
            }
            else
            {
                log(" [state] Volume: %f\n", current_volume);
            }
        }
        else if(boost::iequals(c.d, "SPEED"))
        {
            ASSERT_LE(c.p.size(), 1) << errorhdrstr << c.d.c_str() << maxoneparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            movie::PlaybackRate current_rate = movie::PlaybackRate_Normal;
            EXPECT_TRUE((movie::Status_Success == player->GetPlaybackRate(&current_rate)));
            if(c.p.size() == 1)
            {
                movie::PlaybackRate rate;
                float speed;
                ASSERT_TRUE(speedspec_to_enum(c.p[0], &rate, current_rate, &speed));
                log(" SPEED(%f)\n", speed);
                EXPECT_TRUE((movie::Status_Success == player->SetPlaybackRate(rate)));
            }
            else
            {
                float speed;
                ASSERT_TRUE(playbackrate_to_float(current_rate, &speed));
                log(" [state] Playback rate: %f\n", speed);
            }
        }
        else if(boost::iequals(c.d, "RESET"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            log(" RESET\n");
            EXPECT_TRUE((movie::Status_Success == player->Reset()));
        }
        else if(boost::iequals(c.d, "STOP"))
        {
            ASSERT_EQ(c.p.size(), 0) << errorhdrstr << c.d.c_str() << noparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            log(" STOP\n");
            EXPECT_TRUE((movie::Status_Success == player->Stop()));
        }
        else if(boost::iequals(c.d, "PAUSE"))
        {
            ASSERT_LE(c.p.size(), 1) << errorhdrstr << c.d.c_str() << maxoneparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            bool pause = true;
            if(c.p.size() == 1)
            {
                ASSERT_TRUE(boolean_to_bool(c.p[0], &pause)) << errorhdrstr << '\"' << c.p[0].c_str() << invalidbooleanstr;
            }
            else if(c.p.empty())
            {
                movie::PlayerState state = movie::PlayerState_UnInitialized;
                EXPECT_TRUE((movie::Status_Success == player->GetState(&state)));
                pause = state != movie::PlayerState_Paused;
            }
            log(" PAUSE(%d)\n", pause);
            EXPECT_TRUE((movie::Status_Success == player->Pause(pause)));
        }
        else if(boost::iequals(c.d, "LOOP"))
        {
            ASSERT_LE(c.p.size(), 1) << errorhdrstr << c.d.c_str() << maxoneparameterstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            bool loop = true;
            if(c.p.size() == 1)
            {
                ASSERT_TRUE(boolean_to_bool(c.p[0], &loop)) << errorhdrstr << '\"' << c.p[0].c_str() << invalidbooleanstr;
            }
            log(" LOOP(%d)\n", loop);
            EXPECT_TRUE((movie::Status_Success == player->SetLooping(loop)));
        }
        else if(boost::iequals(c.d, "TRACK"))
        {
            ASSERT_EQ(c.p.size(), 2) << errorhdrstr << c.d.c_str() << needstwoparametersstr;
            ASSERT_TRUE(player.get()) << errorhdrstr << nullplayerstr << c.d.c_str();
            int track = 0;
            bool enable = true;
            if(c.p.size() == 2)
            {
                ASSERT_TRUE(trackspec_to_int(c.p[0], &track)) << errorhdrstr << '\"' << c.p[0].c_str() << invalidtrackstr;
                ASSERT_TRUE(boolean_to_bool(c.p[1], &enable)) << errorhdrstr << '\"' << c.p[1].c_str() << invalidbooleanstr;
            }
            log(" TRACK(%d, %d)\n", track, enable);
            if(enable)
            {
                EXPECT_TRUE((movie::Status_Success == player->SelectTrack(track)));
            }
            else
            {
                EXPECT_TRUE((movie::Status_Success == player->DeSelectTrack(track)));
            }
        }
        else if(boost::iequals(c.d, "POSITION")) // There used to be a window position API that was used by earlier versions of this program.
        {
            log(" POSITION unsupported\n");
        }
    }
}//NOLINT(impl/function_size)
