﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include "VideoRenderer.h"
#include <movie/Common.h>
#include <cstdlib>

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

static const char g_yuv2rgbVertexShader[] =
"#version 450\n"
"layout(location = 0) in vec4 a_position;"
"layout(location = 1) in vec2 a_texCoord;"
"out vec2 v_texCoord;"
"void main() {"
"   gl_Position = a_position;"
"   v_texCoord = a_texCoord;"
"   v_texCoord.y = v_texCoord.y;"
"}";

static const char g_yuv2rgbFragmentShader[] =
"#version 450\n"
"in vec2 v_texCoord;"
"layout(binding = 0) uniform sampler2D u_textureY;"
"layout(binding = 1) uniform sampler2D u_textureUV;"
"out vec4 colorOut;"
"void main() {"
"    const float wr = 0.2126f;"
"    const float wb = 0.0722f;"
"    const float wg = 1 - wr - wb;"
"    const float uMax = 0.436f;"
"    const float vMax = 0.615f;"
"    const float rvCoeff = (1 - wr) / vMax;"
"    const float guCoeff = wb * (1 - wb) / (uMax * wg);"
"    const float gvCoeff = wr * (1 - wr) / (vMax * wg);"
"    const float buCoeff = (1 - wb) / uMax;"
"    float y = texture(u_textureY, v_texCoord).r;"
"    vec2 uv = texture(u_textureUV, v_texCoord).rg;"
"    float u = uv.r - 0.5;"
"    float v = uv.g - 0.5;"
"    float r = y + rvCoeff * v;"
"    float g = y - guCoeff * u - gvCoeff * v;"
"    float b = y + buCoeff * u;"
"    colorOut = vec4(r, g, b, 1);"
"}";

VideoRenderer::VideoRenderer()
{
    m_VideoRendererThreadstacksize = 1024 * 256;
    m_VideoRendererThreadstack = NULL;
    m_YuvBufferSize = 1024 * 1024 * 6;
    m_YuvBuffer = new char [m_YuvBufferSize];
    m_InitNvn = true;
    m_frameReady = false;
    m_pTextureMemory = NULL;
    m_pBufferMemory = NULL;
    m_pVertexPoolMemory = NULL;
}

VideoRenderer::~VideoRenderer()
{
    if( m_VideoRendererThreadstack != NULL )
    {
        delete [] m_VideoRendererThreadstack;
    }

    if( m_YuvBuffer != NULL )
    {
        delete [] m_YuvBuffer;
    }
}

movie::Status VideoRenderer::Initialize()
{
    NN_SDK_LOG("VideoRenderer::Initialize() \n");
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;

    m_ThreadDone = false;
    m_Started = false;
    m_PresentationTimeUs = 0ll;
    m_RenderIntervalMs = 5;
    m_RenderThreadCreated = false;
    nn::os::InitializeMutex( &m_VideoMutex, false, 0 );

    m_VideoRendererThreadstack = new char [m_VideoRendererThreadstacksize * 2];
    void *mem = (void*) m_VideoRendererThreadstack;
    void *alignedStackPointer = (void*)(((uintptr_t)mem + 4095) & ~ (uintptr_t)0x0FFF);
    nnResult = nn::os::CreateThread(
            &m_VideoRendererThreadType,
            &VideoRendererThreadFunction,
            (void*)this,
            alignedStackPointer,
            m_VideoRendererThreadstacksize,
            nn::os::DefaultThreadPriority );

    m_RenderThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        movieStatus = movie::Status_Success;
    }

    return movieStatus;
}

movie::Status VideoRenderer::Finalize()
{
    NN_SDK_LOG("VideoRenderer::Finalize() \n");
    nn::os::FinalizeMutex( &m_VideoMutex );
    return movie::Status_Success;
}

int32_t VideoRenderer::GetRenderInterval()
{
    return m_RenderIntervalMs;
}

movie::Status VideoRenderer::Open(movie::Decoder* videoDecoder, int32_t width, int32_t height, double videoFrameRate)
{
    NN_SDK_LOG("VideoRenderer::Open() \n");

    m_Width = width;
    m_Height = height;
    m_VideoDecoder = videoDecoder;
    m_VideoFrameRate = videoFrameRate;
    if( m_VideoFrameRate > 0.0 )
    {
        m_RenderIntervalMs = (int32_t) (1000.0 / m_VideoFrameRate);
    }
    return movie::Status_Success;
}

movie::Status VideoRenderer::Start(int32_t width, int32_t height)
{
    NN_SDK_LOG("VideoRenderer::Start() \n");
    m_Width = width;
    m_Height = height;
    InitNvn(m_Width, m_Height);

    nn::os::StartThread( &m_VideoRendererThreadType );

    m_Started = true;
    return movie::Status_Success;
}

movie::Status VideoRenderer::UpdateVideoInformation(int32_t width, int32_t height)
{
    m_Width = width;
    m_Height = height;
    return movie::Status_Success;
}

movie::Status VideoRenderer::Render()
{
    VideoData* videoData;
    movie::Buffer videoBufferData;
    size_t size = 0;

    nn::os::LockMutex( &m_VideoMutex );
    int videoListSize = m_VideoBuffers.size();
    if( videoListSize > 0 )
    {
        videoData = &m_VideoBuffers.front();
    }
    else
    {
        nn::os::UnlockMutex( &m_VideoMutex );
        return movie::Status_Success;
    }

    if(!m_frameReady)
    {
        int uvOffset = m_Width * m_Height;
        char *bufferMemory = reinterpret_cast<char*>(nvnBufferMap(&m_videoBufferY.GetBuffer()));
        videoBufferData.SetDataAndCapacity(m_YuvBuffer, m_YuvBufferSize);
        videoBufferData.SetRange(0, m_YuvBufferSize);
        m_VideoDecoder->GetOutputBuffer(videoData->index, &videoBufferData);
        size  = videoBufferData.Size();
        m_VideoDecoder->ReleaseOutputBufferIndex(videoData->index);
        memcpy(bufferMemory, m_YuvBuffer, uvOffset);

        bufferMemory = reinterpret_cast<char*>(nvnBufferMap(&m_videoBufferUV.GetBuffer()));
        memcpy(bufferMemory, m_YuvBuffer + uvOffset, uvOffset / 2);

        m_VideoBuffers.erase(m_VideoBuffers.begin());

        m_frameReady = true;
    }

    nn::os::UnlockMutex( &m_VideoMutex );

    DrawVideoNvn(m_Width, m_Height);
    return movie::Status_Success;
}

void VideoRenderer::Stop()
{
    nn::os::LockMutex( &m_VideoMutex );
        NN_SDK_LOG("VideoRenderer::Stop()  \n");
    VideoData* videoData;
    int videoListSize = m_VideoBuffers.size();
    for(int i=0; i< videoListSize; i++)
    {
        videoData = &m_VideoBuffers.front();
    NN_SDK_LOG("VideoRenderer::Stop() ReleaseOutputBufferIndex: %d \n", videoData->index);
        m_VideoDecoder->ReleaseOutputBufferIndex(videoData->index);
        m_VideoBuffers.erase(m_VideoBuffers.begin());
    }
    nn::os::UnlockMutex( &m_VideoMutex );
}

bool VideoRenderer::DoneProcessing()
{
    return m_ThreadDone;
}

void VideoRenderer::Close()
{
    NN_SDK_LOG("VideoRenderer::Close() \n");
    m_ThreadDone = true;

    if( m_RenderThreadCreated == true )
    {
        nn::os::WaitThread( &m_VideoRendererThreadType );
        nn::os::DestroyThread( &m_VideoRendererThreadType );
        m_RenderThreadCreated = false;
    }

    DestroyNvn();
}

void VideoRenderer::OnOutputAvailable(int index, int64_t presentationTimeUs, uint32_t flags)
{
    VideoData videoData;
    videoData.index = index;
    videoData.presentationTimeUs = presentationTimeUs;
    videoData.flags = flags;

    if( flags != movie::BufferFlags_EndOfStream )
    {
        nn::os::LockMutex( &m_VideoMutex );
        m_VideoBuffers.push_back(videoData);
        nn::os::UnlockMutex( &m_VideoMutex );
    }
}

void VideoRendererThreadFunction(void *arg)
{
    VideoRenderer *videoRenderer = (VideoRenderer*) arg;
    for (;;)
    {
        if( videoRenderer->DoneProcessing() )
        {
            break;
        }
        //TODO: Add presentation time based rendering
        videoRenderer->Render();
        int renderIntervalMs = videoRenderer->GetRenderInterval();
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( renderIntervalMs ) );
    }
}

void VideoRenderer::InitNvn(int width, int height)
{
    const unsigned nvnMemorySize = 64 << 20;
    m_pAllocatorMemory = std::malloc(nvnMemorySize);
    m_allocator.Initialize(m_pAllocatorMemory, nvnMemorySize);

    GlslCompiler compiler;
    compiler.Init();
    compiler.CompileShader(g_yuv2rgbVertexShader, NVN_SHADER_STAGE_VERTEX);
    m_vertexShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>(compiler.GetOutput().data));
    compiler.CompileShader(g_yuv2rgbFragmentShader, NVN_SHADER_STAGE_FRAGMENT);
    m_fragmentShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>(compiler.GetOutput().data));
    compiler.Destroy();

    m_vertexDataSection = m_vertexShaderFile.GetDataSection();
    m_vertexControlSection = m_vertexShaderFile.GetControlSection();
    m_fragmentDataSection = m_fragmentShaderFile.GetDataSection();
    m_fragmentControlSection = m_fragmentShaderFile.GetControlSection();

    m_window.Init();
    m_device.Init(m_window.GetNativeWindow(), m_allocator);
    m_queue.Init(m_device.GetDevice());

    m_pCommandMemory = m_allocator.Allocate(NVN_MEMORY_POOL_STORAGE_GRANULARITY, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_commandPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_NO_ACCESS_BIT, NVN_MEMORY_POOL_STORAGE_GRANULARITY, m_pCommandMemory);
    m_drawCommands.Init(m_device, m_allocator, m_commandPool);
    m_renderTargetCommand.Init(m_device, m_allocator, m_commandPool);
    m_texturePool.Init(m_device.GetDevice(), m_allocator);
    m_videoSampler.Init(m_device.GetDevice());

    int shaderScratchGranularity = m_device.GetDeviceInfo(NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY);
    m_pShaderScratchMemory = m_allocator.Allocate(shaderScratchGranularity, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_shaderScratchPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, shaderScratchGranularity, m_pShaderScratchMemory);
    unsigned fragmentShaderOffset = Align(m_vertexDataSection.size + 1024, 256);
    unsigned shaderMemorySize = Align(fragmentShaderOffset + m_fragmentDataSection.size + 1024, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pShaderPoolMemory = m_allocator.Allocate(shaderMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    memcpy(m_pShaderPoolMemory, m_vertexDataSection.data, m_vertexDataSection.size);
    memcpy(reinterpret_cast<char*>(m_pShaderPoolMemory) + fragmentShaderOffset, m_fragmentDataSection.data, m_fragmentDataSection.size);
    m_shaderPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT, shaderMemorySize, m_pShaderPoolMemory);

    m_vertexDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(fragmentShaderOffset, 1), m_vertexDataSection.size);
    m_fragmentDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(m_fragmentDataSection.size + 1024, 256), m_fragmentDataSection.size);

    NVNshaderData stageData[2];
    stageData[0].control = m_vertexControlSection.data;
    stageData[0].data = m_vertexDataBuffer.GetAddress();
    stageData[1].control = m_fragmentControlSection.data;
    stageData[1].data = m_fragmentDataBuffer.GetAddress();
    m_program.Init(m_device.GetDevice(), stageData, sizeof(stageData) / sizeof(*stageData));

    nvnBlendStateSetDefaults(&m_blendState);
    nvnChannelMaskStateSetDefaults(&m_channelMaskState);
    nvnColorStateSetDefaults(&m_colorState);
    nvnMultisampleStateSetDefaults(&m_multisampleState);
    nvnMultisampleStateSetMultisampleEnable(&m_multisampleState, NVN_FALSE);
    nvnPolygonStateSetDefaults(&m_polygonState);
    nvnPolygonStateSetCullFace(&m_polygonState, NVN_FACE_BACK);
    nvnDepthStencilStateSetDefaults(&m_depthStencilState);

    m_frameReady = false;

    m_queue.WaitForAllCommands();
    m_device.ResizeRenderTargets(m_window.GetWidth(), m_window.GetHeight());
}//NOLINT(impl/function_size)

void VideoRenderer::DestroyNvn()
{
    m_squareMesh.Destroy();
    m_fragmentDataBuffer.Destroy();
    m_vertexDataBuffer.Destroy();
    m_videoBufferUV.Destroy();
    m_videoBufferY.Destroy();
    m_videoSampler.Destroy();
    m_videoTextureUV.Destroy();
    m_videoTextureY.Destroy();
    m_texturePool.Destroy();
    m_drawCommands.Destroy();
    m_program.Destroy();
    m_shaderScratchPool.Destroy();
    m_bufferMemory.Destroy();
    m_textureMemoryPool.Destroy();
    m_renderTargetCommand.Destroy();
    m_commandPool.Destroy();
    m_vertexPool.Destroy();
    m_shaderPool.Destroy();
    m_queue.Destroy();
    m_device.Destroy();
    m_window.Destroy();

    m_allocator.Free(m_pShaderPoolMemory);
    m_allocator.Free(m_pVertexPoolMemory);
    m_allocator.Free(m_pCommandMemory);
    m_allocator.Free(m_pTextureMemory);
    m_allocator.Free(m_pBufferMemory);
    m_allocator.Free(m_pShaderScratchMemory);
    m_allocator.Finalize();

    std::free(m_pAllocatorMemory);
}

void VideoRenderer::DrawVideoNvn(int width, int height)
{
    int bufferIndex;
    nvnQueueAcquireTexture(&m_queue.GetQueue(), &m_device.GetWindow(), &bufferIndex);

    m_queue.WaitForAllCommands();

    m_renderTargetCommand.Reset();

    NVNcommandBuffer *targetCommandBuffer = &m_renderTargetCommand.GetCommandBuffer();
    nvnCommandBufferBeginRecording(targetCommandBuffer);
    {
        NVNtexture *renderTarget = &m_device.GetRenderTarget(bufferIndex);
        nvnCommandBufferSetRenderTargets(targetCommandBuffer, 1, &renderTarget, NULL, NULL, NULL);

        if(m_frameReady)
        {
            nvnCommandBufferSetCopyRowStride(targetCommandBuffer, 0);
            NVNcopyRegion copyRegion;
            copyRegion.depth = 1;
            copyRegion.height = height;
            copyRegion.width = width;
            copyRegion.xoffset = 0;
            copyRegion.yoffset = 0;
            copyRegion.zoffset = 0;
            nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, m_videoBufferY.GetAddress(), &m_videoTextureY.GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
            copyRegion.height = height / 2;
            copyRegion.width = width / 2;
            nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, m_videoBufferUV.GetAddress(), &m_videoTextureUV.GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
            m_frameReady = false;
        }
    }
    NVNcommandHandle commandHandleTarget = nvnCommandBufferEndRecording(targetCommandBuffer);

    nvnQueueSubmitCommands(&m_queue.GetQueue(), 1, &commandHandleTarget);
    nvnQueueSubmitCommands(&m_queue.GetQueue(), 1, &m_commandHandleDraw);

    nvnQueuePresentTexture(&m_queue.GetQueue(), &m_device.GetWindow(), bufferIndex);
}

void VideoRenderer::ResizeTextures(int32_t width, int32_t height)
{
    m_videoTextureY.Destroy();
    m_videoTextureUV.Destroy();
    m_textureMemoryPool.Destroy();
    m_allocator.Free(m_pTextureMemory);
    m_videoBufferY.Destroy();
    m_videoBufferUV.Destroy();
    m_bufferMemory.Destroy();
    m_allocator.Free(m_pBufferMemory);
    m_squareMesh.Destroy();
    m_vertexPool.Destroy();
    m_allocator.Free(m_pVertexPoolMemory);

    //vertices and uv coordinates for a cube
    float aspectRatioDisplay = static_cast<float>(m_window.GetWidth()) / m_window.GetHeight();
    float aspectRatioVideo = static_cast<float>(width) / height;
    float ndcWidth = aspectRatioVideo / aspectRatioDisplay;
    const float vertices[] = {
        -ndcWidth, 1, 0, 1,
        0, 0,
        -ndcWidth, -1, 0, 1,
        0, 1,
        ndcWidth, 1, 0, 1,
        1, 0,
        ndcWidth, -1, 0, 1,
        1, 1,
    };
    unsigned vertexPoolSize = Align(sizeof(vertices), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pVertexPoolMemory = m_allocator.Allocate(vertexPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    memcpy(m_pVertexPoolMemory, vertices, sizeof(vertices));
    m_vertexPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, vertexPoolSize, m_pVertexPoolMemory);
    m_squareMesh.Init(m_device.GetDevice(), m_vertexPool.GetMemoryPool(), 0, sizeof(vertices));

    int textureStrideAlignment = m_device.GetDeviceInfo(NVN_DEVICE_INFO_LINEAR_TEXTURE_STRIDE_ALIGNMENT);
    int videoStride = Align(width, textureStrideAlignment);
    unsigned texturePoolSize = Align(videoStride * height * 3, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pTextureMemory = m_allocator.Allocate(texturePoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_textureMemoryPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, texturePoolSize, m_pTextureMemory);
    m_videoTextureY.Init(m_device.GetDevice(), m_textureMemoryPool, width, height, NVN_FORMAT_R8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
    m_videoTextureUV.Init(m_device.GetDevice(), m_textureMemoryPool, width / 2, height / 2, NVN_FORMAT_RG8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
    m_videoTextureIdY = m_texturePool.Register(m_videoTextureY.GetTexture(), m_videoSampler.GetSampler());
    m_videoTextureIdUV = m_texturePool.Register(m_videoTextureUV.GetTexture(), m_videoSampler.GetSampler());
    m_videoTextureHandleY = nvnDeviceGetTextureHandle(&m_device.GetDevice(), m_videoTextureIdY, m_videoTextureIdY);
    m_videoTextureHandleUV = nvnDeviceGetTextureHandle(&m_device.GetDevice(), m_videoTextureIdUV, m_videoTextureIdUV);

    unsigned videoBufferSize = videoStride * height;
    const unsigned bufferMemorySize = Align(videoBufferSize * 3 / 2, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pBufferMemory = m_allocator.Allocate(bufferMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_bufferMemory.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, bufferMemorySize, m_pBufferMemory);
    m_videoBufferY.Init(m_device.GetDevice(), m_bufferMemory.GetMemoryPool(), m_bufferMemory.GetNewMemoryChunkOffset(videoBufferSize, 1), videoBufferSize);
    m_videoBufferUV.Init(m_device.GetDevice(), m_bufferMemory.GetMemoryPool(), m_bufferMemory.GetNewMemoryChunkOffset(videoBufferSize / 2, 1), videoBufferSize / 2);

    NVNcommandBuffer *commandBuffer = &m_drawCommands.GetCommandBuffer();
    m_drawCommands.Reset();

    nvnCommandBufferBeginRecording(commandBuffer);
    {
        nvnCommandBufferSetScissor(commandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());
        nvnCommandBufferSetViewport(commandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());

        nvnCommandBufferBindBlendState(commandBuffer, &m_blendState);
        nvnCommandBufferBindChannelMaskState(commandBuffer, &m_channelMaskState);
        nvnCommandBufferBindColorState(commandBuffer, &m_colorState);
        nvnCommandBufferBindMultisampleState(commandBuffer, &m_multisampleState);
        nvnCommandBufferBindPolygonState(commandBuffer, &m_polygonState);
        nvnCommandBufferBindDepthStencilState(commandBuffer, &m_depthStencilState);

        nvnCommandBufferSetShaderScratchMemory(commandBuffer, &m_shaderScratchPool.GetMemoryPool(), 0, m_device.GetDeviceInfo(NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY));

        float grey[4] = { 0.5f, 0.5f, 0.5f, 1.0f };
        nvnCommandBufferClearColor(commandBuffer, 0, grey, NVN_CLEAR_COLOR_MASK_RGBA);

        nvnCommandBufferSetSampleMask(commandBuffer, static_cast<int>(~0));

        NVNvertexAttribState vertexAttrib[2];
        for(int i = 0; i < sizeof(vertexAttrib) / sizeof(*vertexAttrib); ++i)
        {
            nvnVertexAttribStateSetDefaults(vertexAttrib + i);
            nvnVertexAttribStateSetStreamIndex(vertexAttrib + i, i);
        }
        nvnVertexAttribStateSetFormat(vertexAttrib + 0, NVN_FORMAT_RGBA32F, 0);
        nvnVertexAttribStateSetFormat(vertexAttrib + 1, NVN_FORMAT_RG32F, sizeof(float) * 4);
        nvnCommandBufferBindVertexAttribState(commandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexAttrib);

        NVNvertexStreamState vertexStream[2];
        for(int i = 0; i < sizeof(vertexStream) / sizeof(*vertexStream); ++i)
        {
            nvnVertexStreamStateSetDefaults(vertexStream + i);
            nvnVertexStreamStateSetStride(vertexStream + i, sizeof(float) * 6);
        }
        nvnCommandBufferBindVertexStreamState(commandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexStream);

        nvnCommandBufferBindVertexBuffer(commandBuffer, 0, m_squareMesh.GetAddress(), sizeof(vertices));
        nvnCommandBufferBindVertexBuffer(commandBuffer, 1, m_squareMesh.GetAddress(), sizeof(vertices));

        nvnCommandBufferBindProgram(commandBuffer, &m_program.GetProgram(), NVN_SHADER_STAGE_VERTEX_BIT | NVN_SHADER_STAGE_FRAGMENT_BIT);

        nvnCommandBufferSetTexturePool(commandBuffer, &m_texturePool.GetTexturePool());
        nvnCommandBufferSetSamplerPool(commandBuffer, &m_texturePool.GetSamplerPool());
        nvnCommandBufferBindTexture(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, m_videoTextureHandleY);
        nvnCommandBufferBindTexture(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 1, m_videoTextureHandleUV);

        nvnCommandBufferDrawArrays(commandBuffer, NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP, 0, 4);
    }
    m_commandHandleDraw = nvnCommandBufferEndRecording(commandBuffer);
}//NOLINT(impl/function_size)
