﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "NvnRenderer.h"
#include <cstdlib>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nv/nv_MemoryManagement.h>

unsigned Align(unsigned value, unsigned alignment)
{
    return value + (alignment - value % alignment) % alignment;
}

char                        g_ShaderHeapBuffer[32 << 20];
nn::mem::StandardAllocator  g_ShaderAllocator(g_ShaderHeapBuffer, sizeof(g_ShaderHeapBuffer));

static void *shaderAllocate(size_t size, size_t alignment, void *userPtr)
{
    return g_ShaderAllocator.Allocate(size, alignment);
}

static void shaderDeallocate(void *block, void *userPtr)
{
    g_ShaderAllocator.Free(block);
}

static void *shaderReallocate(void *block, size_t size, void *userPtr)
{
    return g_ShaderAllocator.Reallocate(block, size);
}

GlslCompiler::GlslCompiler() : m_Init(false)
{
}

GlslCompiler::~GlslCompiler()
{
    Destroy();
}

void GlslCompiler::Init()
{
    glslcSetAllocator(shaderAllocate, shaderDeallocate, shaderReallocate, NULL);
    NN_ASSERT(glslcInitialize(&m_Compiler), "glslcInitialize");
    m_Init = true;
}

void GlslCompiler::Destroy()
{
    if(m_Init)
    {
        glslcFinalize(&m_Compiler);
        m_Init = false;
    }
}

void GlslCompiler::CompileShader(const char *src, NVNshaderStage stage)
{
    m_Compiler.input.sources = &src;
    m_Compiler.input.stages = &stage;
    m_Compiler.input.count = 1;

    if(!glslcCompile(&m_Compiler))
    {
        NN_ASSERT(0, "glslcCompile failed:\n%s\n", m_Compiler.lastCompiledResults->compilationStatus->infoLog);
    }
    GLSLCoutput *output = m_Compiler.lastCompiledResults->glslcOutput;
    m_OutputData.data = output;
    m_OutputData.size = output->size;
}

DataSection GlslCompiler::GetOutput() const
{
    return m_OutputData;
}

NvnShaderSerialize::NvnShaderSerialize() : m_ShaderData(NULL), m_ScratchSize(0)
{
}

NvnShaderSerialize::~NvnShaderSerialize()
{
    std::free(m_ShaderData);
}

void NvnShaderSerialize::LoadNvnShaderMemory(const uint8_t *shaderBinary)
{
    const GLSLCoutput *output = reinterpret_cast<const GLSLCoutput*>(shaderBinary);
    m_ShaderData = std::malloc(output->size);
    memcpy(m_ShaderData, output, output->size);
    for(int i = 0; i < output->numSections; ++i)
    {
        GLSLCsectionTypeEnum type = output->headers[i].genericHeader.common.type;
        if(type == GLSLC_SECTION_TYPE_GPU_CODE)
        {
            const GLSLCgpuCodeHeader *gpuSection = &output->headers[i].gpuCodeHeader;
            const uint8_t *data = reinterpret_cast<const uint8_t*>(m_ShaderData) + gpuSection->common.dataOffset;
            m_DataSection.size = gpuSection->dataSize;
            m_DataSection.data = data + gpuSection->dataOffset;
            m_ControlSection.size = gpuSection->controlSize;
            m_ControlSection.data = data + gpuSection->controlOffset;
            m_ScratchSize = gpuSection->scratchMemBytesRecommended;
        }
    }
}

DataSection NvnShaderSerialize::GetDataSection()
{
    return m_DataSection;
}

DataSection NvnShaderSerialize::GetControlSection()
{
    return m_ControlSection;
}

unsigned NvnShaderSerialize::GetScratchSize()
{
    return m_ScratchSize;
}

NvnMemoryPool::NvnMemoryPool() : /*m_Allocator(NULL),*/ m_Size(0), m_pMemory(NULL), m_Offset(0)
{
}

NvnMemoryPool::~NvnMemoryPool()
{
    Destroy();
}

void NvnMemoryPool::Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory)
{
    NN_ASSERT_NOT_NULL(pMemory);
    NN_ASSERT(size % NVN_MEMORY_POOL_STORAGE_GRANULARITY == 0, "NvnMemoryPool::Init: size needs to be a multiple of NVN_MEMORY_POOL_STORAGE_GRANULARITY (%i)\n", NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_Size = size;
    m_pMemory = pMemory;

    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, &device);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder, flags);
    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    NN_ASSERT(nvnMemoryPoolInitialize(&m_MemoryPool, &poolBuilder) == NVN_TRUE, "Failed to initialize buffer memory pool");
}

void NvnMemoryPool::Destroy()
{
    if(m_pMemory)
    {
        nvnMemoryPoolFinalize(&m_MemoryPool);
        m_pMemory = NULL;
        m_Size = 0;
        m_Offset = 0;
    }
}

ptrdiff_t NvnMemoryPool::GetNewMemoryChunkOffset(unsigned size, unsigned alignment)
{
    NN_ASSERT(m_Offset + size <= m_Size, "Memory pool out of memory.");
    m_Offset = Align(m_Offset, alignment);
    ptrdiff_t dataOffset = m_Offset;
    m_Offset += size;
    return dataOffset;
}

NVNmemoryPool &NvnMemoryPool::GetMemoryPool()
{
    return m_MemoryPool;
}

NvnBuffer::NvnBuffer() : m_pDevice(NULL)
{
}

NvnBuffer::~NvnBuffer()
{
    Destroy();
}

void NvnBuffer::Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size)
{
    m_pDevice = &device;

    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetDevice(&bufferBuilder, &device);
    nvnBufferBuilderSetStorage(&bufferBuilder, &pPool, offset, size);
    NN_ASSERT(nvnBufferInitialize(&m_Buffer, &bufferBuilder) == NVN_TRUE);
}

void NvnBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnBufferFinalize(&m_Buffer);
        m_pDevice = NULL;
    }
}

NVNbufferAddress NvnBuffer::GetAddress() const
{
    return nvnBufferGetAddress(&m_Buffer);
}

NVNbuffer &NvnBuffer::GetBuffer()
{
    return m_Buffer;
}

NvnCommandBuffer::NvnCommandBuffer() : m_pDevice(NULL), m_pAllocator(NULL), m_pMemoryPool(NULL), m_CommandOffset(0), m_pControlPool(NULL)
{
}

NvnCommandBuffer::~NvnCommandBuffer()
{
    Destroy();
}

void NvnCommandBuffer::Init(NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;
    m_pMemoryPool = &memoryPool;

    NN_ASSERT(nvnCommandBufferInitialize(&m_CommandBuffer, &device.GetDevice()));

    int commandBufferCommandAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT);
    int commandBufferControlAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT);

    m_CommandOffset = memoryPool.GetNewMemoryChunkOffset(m_CommandSize, commandBufferCommandAlignment);
    m_pControlPool = allocator.Allocate(m_ControlSize, commandBufferControlAlignment);

    Reset();
}

void NvnCommandBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnCommandBufferFinalize(&m_CommandBuffer);
        m_pAllocator->Free(m_pControlPool);
        m_pControlPool = NULL;
        m_CommandOffset = 0;
        m_pMemoryPool = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
    }
}

void NvnCommandBuffer::Reset()
{
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, &m_pMemoryPool->GetMemoryPool(), m_CommandOffset, m_CommandSize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, m_ControlSize);
}

size_t NvnCommandBuffer::GetCommandMemoryUsed() const
{
    return nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
}

size_t NvnCommandBuffer::GetControlMemoryUsed() const
{
    return nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
}

NVNcommandBuffer &NvnCommandBuffer::GetCommandBuffer()
{
    return m_CommandBuffer;
}

//in the Windows demos nvnBootstrapLoader needs to be loaded dynamically, but on Horizon it seems to exist in libnvn.a
extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char * name);
}

NvnDevice::NvnDevice() : m_pAllocator(NULL), m_pRenderTargetMemory(NULL), m_FirstResize(true)
{
}

NvnDevice::~NvnDevice()
{
    Destroy();
}

void NvnDevice::Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator)
{
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>(nvnBootstrapLoader("nvnDeviceInitialize"));
    NN_ASSERT(pfnc_nvnDeviceInitialize, "BootstrapLoader failed to find nvnDeviceInitialize");
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(nvnBootstrapLoader("nvnDeviceGetProcAddress"));
    NN_ASSERT(pfnc_nvnDeviceGetProcAddress, "BootstrapLoader failed to find nvnDeviceGetProcAddress");
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);

    /* If debug is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#ifdef _DEBUG
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT | NVN_DEVICE_FLAG_DEBUG_SKIP_CALLS_ON_ERROR_BIT;
#else
    deviceFlags = 0;
#endif

    NVNdeviceBuilder deviceData;
    nvnDeviceBuilderSetDefaults(&deviceData);
    nvnDeviceBuilderSetFlags(&deviceData, deviceFlags);

    NN_ASSERT(pfnc_nvnDeviceInitialize(&m_Device, &deviceData), "nvnDeviceInitialize");

    int MajorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MAJOR_VERSION);
    int MinorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MINOR_VERSION);

    NN_ASSERT(MajorVersion == NVN_API_MAJOR_VERSION && MinorVersion == NVN_API_MINOR_VERSION, "NVN SDK not supported by current driver.");

        /*
         * Debug Layer Callback
         * --------------------
         * Install the debug layer callback if the debug layer was enabled during
         * device initialization. It is possible to pass a pointer to the NVN API
         * to remember and pass back through the debug callback.
         */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

    m_pAllocator = &allocator;

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

        /* Render targets that need to be displayed to the screen need both the render target and display access bits. */
    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /* Setup the render target memory pool. (See MemoryPool.cpp/.h) */
    unsigned memorySize = Align(m_ColorTargetSize * m_NumColorBuffers, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pRenderTargetMemory = allocator.Allocate(memorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_RenderTargetMemoryPool.Init(m_Device, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, memorySize, m_pRenderTargetMemory);

        /*
         * NVN Window
         * ----------
         * The NVNwindow is used to present a render target to the screen. It manages
         * an array of render targets and associates them with an NVNnativeWindow. On
         * Horizon NVNnativeWindow defined as ANativeWindow* and on Windows it's defined
         * as HWND. Each frame, the NVNwindow needs to be queried for the active render
         * target for that frame.
         *
         * Window Builder Settings
         * -----------------------
         * SetTextures     - Sets the list of render targets to be used by the window.
         *                   Textures used by the NVNwindow must be initialized with the
         *                   display access bit. A window can be given a max of 6 targets.
         *
         * SetNativeWindow - Sets the handle to the window.
         *
         */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
}

void NvnDevice::Destroy()
{
    if(m_pAllocator)
    {
        m_RenderTargetMemoryPool.Destroy();
        m_pAllocator->Free(m_pRenderTargetMemory);
        nvnWindowFinalize(&m_pWindow);
        nvnDeviceFinalize(&m_Device);
        m_pAllocator = NULL;
        m_pRenderTargetMemory = NULL;
        m_FirstResize = true;
    }
}

void NvnDevice::ResizeRenderTargets(unsigned width, unsigned height)
{
    /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

    /*
     * Set the size of the texture to be created by the texture builder.
     * Implicitly sets depth to one.
     */
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < m_NumColorBuffers; ++i)
    {
        if(!m_FirstResize)
        {
            nvnTextureFinalize(m_RenderTargets + i);
        }

        /* Set the memory pool the texture is to be allocated from and offset to where it will be located. */
        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, &m_RenderTargetMemoryPool.GetMemoryPool(), m_ColorTargetSize * i);

        /* Create the texture using the current state of the texture builder. */
        nvnTextureInitialize(m_RenderTargets + i, &m_RenderTargetBuilder);
    }

    /*
     * Recreate the NVNwindow with the newly allocated/created render targets
     * for the new resolution.
     */
    if(!m_FirstResize)
    {
        nvnWindowFinalize(&m_pWindow);
    }

    /*
     *  Sets the number of render targets that the NVNwindow will have available and
     *  gives a pointer to an array of those targets.
     */
    NVNtexture *renderTargets[m_NumColorBuffers];
    for(int i = 0; i < m_NumColorBuffers; ++i)
        renderTargets[i] = m_RenderTargets + i;
    nvnWindowBuilderSetTextures(&m_WindowBuilder, m_NumColorBuffers, renderTargets);
    nvnWindowInitialize(&m_pWindow, &m_WindowBuilder);

    m_FirstResize = false;
}

NVNdevice &NvnDevice::GetDevice()
{
    return m_Device;
}

NVNwindow &NvnDevice::GetWindow()
{
    return m_pWindow;
}

NVNtexture &NvnDevice::GetRenderTarget(int index)
{
    NN_ASSERT(index >= 0 && index < m_NumColorBuffers, "NvnDevice::GetRenderTarget index out of range");
    return m_RenderTargets[index];
}

int NvnDevice::GetDeviceInfo(NVNdeviceInfo name)
{
    int i;
    nvnDeviceGetInteger(&m_Device, name, &i);
    return i;
}

void NvnDevice::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char * message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_SDK_LOG("NVN Debug Layer Callback:\n");
    NN_SDK_LOG("  source:       0x%08x\n", source);
    NN_SDK_LOG("  type:         0x%08x\n", type);
    NN_SDK_LOG("  id:           0x%08x\n", id);
    NN_SDK_LOG("  severity:     0x%08x\n", severity);
    NN_SDK_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "Debug layer callback hit");
}

NvnProgram::NvnProgram() : m_pDevice(NULL)
{
}

NvnProgram::~NvnProgram()
{
    Destroy();
}

void NvnProgram::Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount)
{
    m_pDevice = &device;
    NN_ASSERT(nvnProgramInitialize(&m_Program, &device) == NVN_TRUE, "nvnProgramInitialize");
    NN_ASSERT(nvnProgramSetShaders(&m_Program, stageCount, stageData) == NVN_TRUE, "nvnProgramSetShaders");
}

void NvnProgram::Destroy()
{
    if(m_pDevice)
    {
        nvnProgramFinalize(&m_Program);
        m_pDevice = NULL;
    }
}

NVNprogram &NvnProgram::GetProgram()
{
    return m_Program;
}

NvnQueue::NvnQueue() : m_pDevice(NULL)
{
}

NvnQueue::~NvnQueue()
{
    Destroy();
}

void NvnQueue::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetDevice(&queueBuilder, &device);
    NN_ASSERT(nvnQueueInitialize(&m_Queue, &queueBuilder) == NVN_TRUE, "nvnQueueInitialize");
    NN_ASSERT(nvnSyncInitialize(&m_CommandBufferSync, &device) == NVN_TRUE, "nvnSyncInitialize");
}

void NvnQueue::Destroy()
{
    if(m_pDevice)
    {
        nvnQueueFinalize(&m_Queue);
        nvnSyncFinalize(&m_CommandBufferSync);
        m_pDevice = NULL;
    }
}

void NvnQueue::WaitForAllCommands()
{
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);
}

NVNqueue &NvnQueue::GetQueue()
{
    return m_Queue;
}

NvnSampler::NvnSampler() : m_pDevice(NULL)
{
}

NvnSampler::~NvnSampler()
{
    Destroy();
}

void NvnSampler::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNsamplerBuilder builder;
    nvnSamplerBuilderSetDefaults(&builder);
    nvnSamplerBuilderSetDevice(&builder, &device);
    NN_ASSERT(nvnSamplerInitialize(&m_Sampler, &builder) == NVN_TRUE);
}

void NvnSampler::Destroy()
{
    if(m_pDevice)
    {
        nvnSamplerFinalize(&m_Sampler);
        m_pDevice = NULL;
    }
}

NVNsampler &NvnSampler::GetSampler()
{
    return m_Sampler;
}

NvnTexture::NvnTexture() : m_pDevice(NULL)
{
}

NvnTexture::~NvnTexture()
{
    Destroy();
}

void NvnTexture::Init(NVNdevice &device, NvnMemoryPool &memoryPool, unsigned width, unsigned height, NVNformat format, int32_t flags, unsigned stride)
{
    m_pDevice = &device;

    NVNtextureBuilder builder;
    nvnTextureBuilderSetDefaults(&builder);
    nvnTextureBuilderSetDevice(&builder, &device);
    nvnTextureBuilderSetSize2D(&builder, width, height);
    nvnTextureBuilderSetFlags(&builder, flags);
    nvnTextureBuilderSetFormat(&builder, format);
    nvnTextureBuilderSetTarget(&builder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetStride(&builder, stride);
    size_t size = nvnTextureBuilderGetStorageSize(&builder);
    size_t storageAlignment = nvnTextureBuilderGetStorageAlignment(&builder);
    nvnTextureBuilderSetStorage(&builder, &memoryPool.GetMemoryPool(), memoryPool.GetNewMemoryChunkOffset(size, storageAlignment));
    NN_ASSERT(nvnTextureInitialize(&m_Texture, &builder) == NVN_TRUE, "Failed to initialize texture");
}

void NvnTexture::Destroy()
{
    if(m_pDevice)
    {
        nvnTextureFinalize(&m_Texture);
        m_pDevice = NULL;
    }
}

NVNtexture &NvnTexture::GetTexture()
{
    return m_Texture;
}

NvnTextureSamplerPool::NvnTextureSamplerPool() : m_pMemory(NULL), m_pDevice(NULL), m_Id(256)
{
}

NvnTextureSamplerPool::~NvnTextureSamplerPool()
{
    Destroy();
}

void NvnTextureSamplerPool::Init(NVNdevice &device, nn::mem::StandardAllocator &allocator)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;

    int maxSamplerPoolSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_MAX_SAMPLER_POOL_SIZE, &maxSamplerPoolSize);
    int samplerDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerDescriptorSize);
    int textureDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureDescriptorSize);
    const unsigned poolSize = Align(maxSamplerPoolSize * (samplerDescriptorSize + textureDescriptorSize), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pMemory = allocator.Allocate(poolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_MemoryPool.Init(device, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, poolSize, m_pMemory);
    NN_ASSERT(nvnSamplerPoolInitialize(&m_SamplerPool, &m_MemoryPool.GetMemoryPool(), 0, maxSamplerPoolSize) == NVN_TRUE);
    int samplerPoolOffset = Align(maxSamplerPoolSize * samplerDescriptorSize, textureDescriptorSize);
    NN_ASSERT(nvnTexturePoolInitialize(&m_TexturePool, &m_MemoryPool.GetMemoryPool(), samplerPoolOffset, maxSamplerPoolSize) == NVN_TRUE);
}

void NvnTextureSamplerPool::Destroy()
{
    if(m_pDevice)
    {
        nvnTexturePoolFinalize(&m_TexturePool);
        nvnSamplerPoolFinalize(&m_SamplerPool);
        m_MemoryPool.Destroy();
        m_pAllocator->Free(m_pMemory);
        m_pMemory = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
        m_Id = 256;
    }
}

NVNtexturePool &NvnTextureSamplerPool::GetTexturePool()
{
    return m_TexturePool;
}

NVNsamplerPool &NvnTextureSamplerPool::GetSamplerPool()
{
    return m_SamplerPool;
}

unsigned NvnTextureSamplerPool::Register(NVNtexture &pTexture, NVNsampler &pSampler)
{
    NVNtextureView view;
    nvnTextureViewSetDefaults(&view);
    nvnTexturePoolRegisterTexture(&m_TexturePool, m_Id, &pTexture, &view);
    nvnSamplerPoolRegisterSampler(&m_SamplerPool, m_Id, &pSampler);
    return m_Id++;
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

NvnWindow::NvnWindow() : m_NativeWindow(NULL), m_pDisplay(NULL), m_pLayer(NULL), m_WindowWidth(1280), m_WindowHeight(720)
{
}

NvnWindow::~NvnWindow()
{

}

void NvnWindow::Init()
{
    InitMemory();
    InitWindow();
}

void NvnWindow::Destroy()
{
    DestroyLayer(m_pLayer);
    CloseDisplay(m_pDisplay);
    nn::vi::Finalize();
}

nn::vi::NativeWindowHandle NvnWindow::GetNativeWindow()
{
    return m_NativeWindow;
}

void NvnWindow::InitMemory()
{

}

void NvnWindow::InitWindow()
{
    /* Initialize Visual Interface (VI) system to display
     * to the target's screen
     */
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);

    result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
}

int NvnWindow::GetWidth() const
{
    return m_WindowWidth;
}

int NvnWindow::GetHeight() const
{
    return m_WindowHeight;
}
