﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/os.h>

#include <movie/Utils.h>
#include <movie/Decoder.h>
#include <movie/Status.h>

enum StreamFormat
{
    StreamFormat_Unknown = 0,
    StreamFormat_ivf,
    StreamFormat_h264,
};

class ByteStreamReader
{
public:
    ByteStreamReader();

    movie::Status Open(const char* fileName, StreamFormat streamFormat);

    movie::Status Prepare();

    movie::Status PrepareForLooping();

    movie::Status Close();

    movie::Status ReadNextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs);

    movie::Status GetStreamProperties(movie::DecoderType* decoderType, int32_t* videoWidth, int32_t* videoHeight, double* videoFrameRate);

    ~ByteStreamReader();

private:
    movie::Status PrepareIVFStream();

    movie::Status ReadIVFNextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs);

    movie::Status PrepareH264Stream();

    movie::Status ReadH264NextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs);

private:
    static const int32_t m_IvfHeaderSize = 32;
    static const int32_t m_IvfFrameHeaderSize = 12;
    static const int32_t m_DataBufferSize = 1024 * 1024;
    int64_t m_AvailableInputData;
    uint8_t* m_DataBufferPointer;

    nn::fs::FileHandle m_FileHandle;

    int64_t m_TrackDurationUs;
    int64_t m_FileReadOffset;

    StreamFormat m_StreamFormat;

    uint8_t m_IvfHeader[ m_IvfHeaderSize ];
    uint8_t m_IvfFrameHeader[ m_IvfFrameHeaderSize ];
    uint8_t m_dataBuffer[ m_DataBufferSize ];

public:
    uint8_t* m_H264Pps;
    size_t m_H264PpsSize;
    uint8_t* m_H264Sps;
    size_t m_H264SpsSize;

    movie::DecoderType m_DecoderType;

    int32_t m_VideoWidth;
    int32_t m_VideoHeight;
    double m_VideoFrameRate;
    int64_t m_PresentationTimeUs;
    int64_t m_PresentationTimeIncrementUs;
};

