﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
//#define LOG_NDEBUG 0

#include <cinttypes>
#ifndef RAPTOR_ENHANCEMENT
#include "HorizonAlloc.h"           // must go first due to all those redefines
#endif
#include <utils/Log.h>
#include "MedPlayer.h"
#include <binder/IServiceManager.h>
#include <binder/ProcessState.h>
#include <media/ICrypto.h>
#include <media/IMediaHTTPService.h>
#include <media/IMediaPlayerService.h>
#include <media/stagefright/foundation/ABuffer.h>
#include <media/stagefright/foundation/ADebug.h>
#include <media/stagefright/foundation/ALooper.h>
#include <media/stagefright/foundation/AMessage.h>
#include <media/stagefright/foundation/AString.h>
#include <media/stagefright/DataSource.h>
#include <media/stagefright/MediaCodec.h>
#include <media/stagefright/MediaCodecList.h>
#include <media/stagefright/MediaDefs.h>
#include <media/stagefright/NuMediaExtractor.h>
#include <gui/ISurfaceComposer.h>
#include <gui/SurfaceComposerClient.h>
#include <gui/Surface.h>
#include <ui/DisplayInfo.h>
#include <nnt/nnt_Argument.h>

#ifdef RAPTOR_ENHANCEMENT
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
long double strtold (const char *__restrict, char **__restrict);
_Noreturn void _Exit (int);
#include <ALooper.h>
#include "nvgr.h"
#include <nn/lmem/lmem_ExpHeap.h>
#else
#include "HorizonDisplay.h"
#include "HorizonPlayer.h"
#include "HorizonBase\HorizonStorage.h"
#include "HorizonBase\HorizonFile.h"

#include "HorizonStorage.h"
#define FILE_DUMP
#endif
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nnt/nntest.h>

#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <mm_MemoryManagement.h>

//#include "SfGlobal.h"
#ifdef RAPTOR_ENHANCEMENT
extern void initializeLooperRoster();
extern void initializeAtomizer();
extern void initializeDataSource();
#endif

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "MediaCodecDecoder"

static void usage(const char *me) {
    NN_LOG("usage: %s [-a] use audio\n"
                    "\t\t[-v] use video\n"
                    "\t\t[-p] playback\n"
                    "\t\t[-S] allocate buffers from a surface\n",
                    "\t\t[-W] Video stream Width\n",
                    "\t\t[-H] Video stream Height\n",
                    me);
}
namespace android {
struct CodecState {
    sp<MediaCodec> mCodec;
    Vector<sp<ABuffer> > mInBuffers;
    Vector<sp<ABuffer> > mOutBuffers;
    bool mSignalledInputEOS;
    bool mSawOutputEOS;
    int64_t mNumBuffersDecoded;
    int64_t mNumBytesDecoded;
    bool mIsAudio;
};
}  // namespace android
#ifdef RAPTOR_ENHANCEMENT
namespace{

    const int FsHeapSize = 512 * 1024;
    const int mmHeapSize = 512 << 20;
    const int mmFirmwareMemorySize = 8 << 20;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;
    char                        g_mmHeapBuffer[mmHeapSize];
    nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));

    char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }

    void* MultimediaAllocate(size_t size, size_t alignment, void *userPtr)
    {
        return g_MultimediaAllocator.Allocate(size, alignment);
    }

    void MultimediaFree(void *addr, void *userPtr)
    {
        g_MultimediaAllocator.Free(addr);
    }

    void *MultimediaReallocate(void* addr, size_t newSize, void *userPtr)
    {
        return g_MultimediaAllocator.Reallocate(addr, newSize);
    }
}
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    const size_t mallocSize = 32 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    NN_SDK_LOG("\ninit done: \n");
}

#else
// this function called before main..  Can be used to init memory for allocators used in static functions.
// Need to override - default OS one has tendency to grab all of memory not leaving us with any....
extern "C" void nninitStartup()
{
    NN_LOG("In nninitStartup()\n");
    // Set the total size of the OS memory heap
    const size_t MemoryHeapSize = 32 * 1024 * 1024;
    nn::os::SetMemoryHeapSize( MemoryHeapSize );
    // Allocated the memory space used by OS malloc from the memory heap
    uintptr_t address;
    auto result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    if(result.IsSuccess())
    {
        NN_LOG("In nninitStartup() - Allocated OS heap of size %d MB\n",MemoryHeapSize / (1024 * 1024));
    } else {
        NN_ASSERT(false && "nn::os::AllocateMemoryBlock() failed.\n");
    }

    // Set heap used by OS malloc and it's friends
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
#endif

static int decode(
        const android::sp<android::ALooper> &looper,
        const char *path,
        bool useAudio,
        bool useVideo,
        const android::sp<android::Surface> &surface,
        nn::fs::FileHandle *File) {
    using namespace android;
    static int64_t kTimeout = 500ll;
    sp<NuMediaExtractor> extractor = new NuMediaExtractor;
    if (extractor->setDataSource(NULL /* httpService */, path) != OK) {
        ALOGV("unable to instantiate extractor.\n");
        return 1;
    }
    KeyedVector<size_t, CodecState> stateByTrack;
    bool haveAudio = false;
    bool haveVideo = false;
    for (size_t i = 0; i < extractor->countTracks(); ++i) {
        sp<AMessage> format;
        status_t err = extractor->getTrackFormat(i, &format);
        CHECK_EQ(err, (status_t)OK);
        AString mime;
        CHECK(format->findString("mime", &mime));
        bool isAudio = !strncasecmp(mime.c_str(), "audio/", 6);
        bool isVideo = !strncasecmp(mime.c_str(), "video/", 6);
        if (useAudio && !haveAudio && isAudio) {
            haveAudio = true;
        } else if (useVideo && !haveVideo && isVideo) {
            haveVideo = true;
        } else {
            continue;
        }
        ALOGV("selecting track %zu", i);
        err = extractor->selectTrack(i);
        CHECK_EQ(err, (status_t)OK);
        CodecState *state =
            &stateByTrack.editValueAt(stateByTrack.add(i, CodecState()));
        state->mNumBytesDecoded = 0;
        state->mNumBuffersDecoded = 0;
        state->mIsAudio = isAudio;
        state->mCodec = MediaCodec::CreateByType(
                looper, mime.c_str(), false /* encoder */);
        CHECK(state->mCodec != NULL);
        err = state->mCodec->configure(
                //format, isVideo ? surface : NULL,
                format, NULL,
                NULL /* crypto */,
                0 /* flags */);

        CHECK_EQ(err, (status_t)OK);
        state->mSignalledInputEOS = false;
        state->mSawOutputEOS = false;
    }
    CHECK(!stateByTrack.isEmpty());
    int64_t startTimeUs = ALooper::GetNowUs();
    for (size_t i = 0; i < stateByTrack.size(); ++i) {
        CodecState *state = &stateByTrack.editValueAt(i);
        sp<MediaCodec> codec = state->mCodec;
        CHECK_EQ((status_t)OK, codec->start());
        CHECK_EQ((status_t)OK, codec->getInputBuffers(&state->mInBuffers));
        CHECK_EQ((status_t)OK, codec->getOutputBuffers(&state->mOutBuffers));
        ALOGV("got %zu input and %zu output buffers",
              state->mInBuffers.size(), state->mOutBuffers.size());
    }
    bool sawInputEOS = false;
#ifdef FILE_DUMP
    int64_t fileOffset = 0;
    int fileDataWritten = 0;
    bool flushed = false;
#endif //FILE_DUMP
    for (;;) {
        if (!sawInputEOS) {
            size_t trackIndex;
            status_t err = extractor->getSampleTrackIndex(&trackIndex);
            if (err != OK) {
                ALOGV("saw input eos");
                sawInputEOS = true;
            } else {
                CodecState *state = &stateByTrack.editValueFor(trackIndex);
                size_t index;
                err = state->mCodec->dequeueInputBuffer(&index, kTimeout);
                if (err == OK) {
                    ALOGV("filling input buffer %zu", index);
                    const sp<ABuffer> &buffer = state->mInBuffers.itemAt(index);
                    err = extractor->readSampleData(buffer);
                    CHECK_EQ(err, (status_t)OK);
                    int64_t timeUs;
                    err = extractor->getSampleTime(&timeUs);
                    CHECK_EQ(err, (status_t)OK);
                    uint32_t bufferFlags = 0;
                    err = state->mCodec->queueInputBuffer(
                            index,
                            0 /* offset */,
                            buffer->size(),
                            timeUs,
                            bufferFlags);
                    CHECK_EQ(err, (status_t)OK);
                    extractor->advance();
                } else {
                    CHECK_EQ(err, -EAGAIN);
                }
            }
        } else {
            for (size_t i = 0; i < stateByTrack.size(); ++i) {
                CodecState *state = &stateByTrack.editValueAt(i);
                if (!state->mSignalledInputEOS) {
                    size_t index;
                    status_t err =
                        state->mCodec->dequeueInputBuffer(&index, kTimeout);
                    if (err == OK) {
                        ALOGV("signalling input EOS on track %zu", i);
                        err = state->mCodec->queueInputBuffer(
                                index,
                                0 /* offset */,
                                0 /* size */,
                                0ll /* timeUs */,
                                MediaCodec::BUFFER_FLAG_EOS);
                        CHECK_EQ(err, (status_t)OK);
                        state->mSignalledInputEOS = true;
                    } else {
                        CHECK_EQ(err, -EAGAIN);
                    }
                }
            }
        }
        bool sawOutputEOSOnAllTracks = true;
        for (size_t i = 0; i < stateByTrack.size(); ++i) {
            CodecState *state = &stateByTrack.editValueAt(i);
            if (!state->mSawOutputEOS) {
                sawOutputEOSOnAllTracks = false;
                break;
            }
        }
        if (sawOutputEOSOnAllTracks) {
            break;
        }
#ifdef FILE_DUMP
        flushed = false;
#endif //FILE_DUMP
        for (size_t i = 0; i < stateByTrack.size(); ++i) {
            CodecState *state = &stateByTrack.editValueAt(i);
            if (state->mSawOutputEOS) {
                continue;
            }
            size_t index;
            size_t offset;
            size_t size;
            int64_t presentationTimeUs;
            uint32_t flags;
            status_t err = state->mCodec->dequeueOutputBuffer(
                    &index, &offset, &size, &presentationTimeUs, &flags,
                    kTimeout);
            if (err == OK) {
                ALOGV("draining output buffer %zu, time = %" PRId64 " us\n",
                      index, presentationTimeUs);
#ifdef FILE_DUMP

                if(size != 0){
                    ABuffer *aBuffer;
                    aBuffer = state->mOutBuffers[index].get();
                    aBuffer->setRange(offset,size);
                    uint8_t* buf = aBuffer->data();
                    if(File){
                        nn::fs::WriteOption writeOption = {0};
                        nn::fs::WriteFile(*File, fileOffset, buf, size, writeOption).IsSuccess();
                    }
                    fileOffset += size;
                    fileDataWritten += size;
                    if(fileDataWritten > 1024 * 1024)
                    {
                        flushed = true;
                        fileDataWritten = 0;
                        if(File){
                            nn::fs::FlushFile(*File);
                        }
                    }
                }
#endif
                ++state->mNumBuffersDecoded;
                state->mNumBytesDecoded += size;
                err = state->mCodec->releaseOutputBuffer(index);
                CHECK_EQ(err, (status_t)OK);
                if (flags & MediaCodec::BUFFER_FLAG_EOS) {
                    ALOGV("reached EOS on output.");
                    state->mSawOutputEOS = true;
                }
            } else if (err == INFO_OUTPUT_BUFFERS_CHANGED) {
                ALOGV("INFO_OUTPUT_BUFFERS_CHANGED");
                CHECK_EQ((status_t)OK,
                         state->mCodec->getOutputBuffers(&state->mOutBuffers));
                ALOGV("got %zu output buffers", state->mOutBuffers.size());
            } else if (err == INFO_FORMAT_CHANGED) {
                sp<AMessage> format;
                CHECK_EQ((status_t)OK, state->mCodec->getOutputFormat(&format));
                ALOGV("INFO_FORMAT_CHANGED: %s", format->debugString().c_str());
            } else {
                CHECK_EQ(err, -EAGAIN);
            }
        }
    }
#ifdef FILE_DUMP
    if(!flushed && File)
        nn::fs::FlushFile(*File);
#endif //FILE_DUMP
    int64_t elapsedTimeUs = ALooper::GetNowUs() - startTimeUs;
    for (size_t i = 0; i < stateByTrack.size(); ++i) {
        CodecState *state = &stateByTrack.editValueAt(i);
        CHECK_EQ((status_t)OK, state->mCodec->release());
        if (state->mIsAudio) {
            NN_LOG("track %zu: %" PRId64 " bytes received. %.2f KB/sec\n",
                   i,
                   state->mNumBytesDecoded,
                   state->mNumBytesDecoded * 1E6 / 1024 / elapsedTimeUs);
        } else {
            NN_LOG("track %zu: %" PRId64 " frames decoded, %.2f fps. %" PRId64 " bytes received. %.2f KB/sec\n",
                   i,
                   state->mNumBuffersDecoded,
                   state->mNumBuffersDecoded * 1E6 / elapsedTimeUs,
                   state->mNumBytesDecoded,
                   state->mNumBytesDecoded * 1E6 / 1024 / elapsedTimeUs);
        }
    }
    return 0;
}//NOLINT(impl/function_size)
TEST(MediaCodecDecoderTest, Decoder)
{
    NN_LOG("\n--- Testing %d bit build ---\n\n", sizeof(size_t) * 8 );

    using namespace android;
        /* Set allocator callback functions */
    nv::mm::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    const char *me = argv[0];
    nn::fs::FileHandle FileVideo;
    nn::fs::FileHandle FileAudio;

    bool useAudio = false;
    bool useVideo = false;
    bool noOutputFiles = false;
    bool playback = false;
    bool useSurface = false;
    int  VideoWidth = 0;
    int  VideoHeight = 0;
    int res;




    while ((res = getopt(argc, argv, "havpSDW:H:q")) >= 0) {
        switch (res) {
            case 'q':
            {
                noOutputFiles = true;
                break;
            }
            case 'a':
            {
                useAudio = true;
                break;
            }
            case 'v':
            {
                useVideo = true;
                break;
            }
            case 'p':
            {
                playback = true;
                break;
            }
            case 'S':
            {
                useSurface = true;
                break;
            }
            case 'W':
            {
                VideoWidth = atoi(optarg);
                break;
            }
            case 'H':
            {
                VideoHeight = atoi(optarg);
                break;
            }

            case '?':
            case 'h':
            default:
            {
                usage(me);
                FAIL();
            }
        }
    }

    argc -= optind;
    argv += optind;
    if (argc != 1) {
        usage(me);
        FAIL();
    }
    if (!useAudio && !useVideo) {
        useAudio = useVideo = true;
    }
#ifdef RAPTOR_ENHANCEMENT
    initializeLooperRoster();
    initializeAtomizer();
    initializeDataSource();
#else
    // setup stuff here so we can eventually modify parameters from command line
    const int kExtraHeapMemMB = 64;
    MV_OSMemoryHeapResize(kExtraHeapMemMB);             // allocate an extra kExtraHeapMemMB MB to use.
    MV_HeapCreate(eMVHeapNN, kExtraHeapMemMB);          // pull (N) MB from the OS memory Heap, and use it for our NN heap
    MV_UseHeap(eMVHeapNN);

    // various parameters to control malloc tracing breaks
    //MV_setTrace(eMVTraceNone,0);              // eMVTraceNone: don't trace alloc's or free's
    MV_setTrace(eMVTraceSize, 16 * 1024);       // eMVTraceSize: Trace allocs/frees with sizes (N)K or larger
    //MV_setTrace(eMVTraceChange,4*1024);       // eMVTraceChange: when sum of memory allocs changes by (N)K or more
    //MV_setAllocBreak(1025);       // break on this particular alloc...
    SfGlobal::Global();


    //HorizonStorage gStorage;
    //gStorage.Initialize("c:/");                         // map mmfs: to this directory [CHANGE PER YOUR MACHINE]

    ProcessState::self()->startThreadPool();
#endif
    DataSource::RegisterDefaultSniffers();
    sp<ALooper> looper = new ALooper;
    looper->start();
    sp<SurfaceComposerClient> composerClient;
    sp<SurfaceControl> control;
    sp<Surface> surface;

#ifdef FILE_DUMP
    nn::Result resultHostMount = nn::fs::MountHostRoot();
    if (resultHostMount.IsFailure())
    {
        NN_SDK_LOG("nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                resultHostMount.GetModule(),
                resultHostMount.GetDescription());
        FAIL();
    }
    if(useVideo && !noOutputFiles){
        const char *videoFile = "C:/temp/VideoDecoder_out.yuv";
        nn::fs::CreateFile(videoFile, 0);
        ASSERT_EQ(true, nn::fs::OpenFile(&FileVideo, videoFile, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend).IsSuccess());
    }
    if(useAudio && !noOutputFiles){
        const char *audioFile = "C:/temp/AudioDecoder_out.wav";
        nn::fs::CreateFile(audioFile, 0);
        ASSERT_EQ(true, nn::fs::OpenFile(&FileAudio, audioFile, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend).IsSuccess());
    }

#endif

    if (playback || (useSurface && useVideo)) {
        if (VideoWidth == 0 || VideoHeight == 0) {
            usage(me);
            FAIL();
        }
#ifdef RAPTOR_ENHANCEMENT
        composerClient = new SurfaceComposerClient;
        CHECK_EQ(composerClient->initCheck(), (status_t)OK);
        sp<IBinder> display(SurfaceComposerClient::getBuiltInDisplay(
                ISurfaceComposer::eDisplayIdMain)); //HACK
        DisplayInfo info;
        SurfaceComposerClient::getDisplayInfo(display, &info);
        ssize_t displayWidth = info.w;
        ssize_t displayHeight = info.h;
        NN_LOG("display is %ld x %ld\n", displayWidth, displayHeight);
        control = composerClient->createSurface(
                String8("A Surface"),
                VideoWidth,
                VideoHeight,
                NVGR_PIXEL_FORMAT_NV12,
                0);
        CHECK(control != NULL);
        CHECK(control->isValid());
        SurfaceComposerClient::openGlobalTransaction();
        CHECK_EQ(control->setLayer(INT_MAX), (status_t)OK);
        CHECK_EQ(control->show(), (status_t)OK);
        SurfaceComposerClient::closeGlobalTransaction();//
        surface = control->getSurface();
        CHECK(surface != NULL);
#endif
        NN_LOG("Inside playback || (useSurface && useVideo case\n");
    }
    if (playback) {
        sp<MedPlayer> player = new MedPlayer;
        looper->registerHandler(player);
        player->setDataSource(argv[0]);
#ifdef RAPTOR_ENHANCEMENT
        player->setSurface(surface->getIGraphicBufferProducer());
#endif
        player->start();
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(60) ); // run for a second
        player->stop();
        player->reset();
    } else {
        if(useVideo)
            decode(looper, argv[0], false, useVideo, surface, noOutputFiles ? 0 : &FileVideo);
        if(useAudio)
            decode(looper, argv[0], useAudio, false, surface, noOutputFiles ? 0 : &FileAudio);
    }
#ifdef RAPTOR_ENHANCEMENT
    if (playback || (useSurface && useVideo)) {
        composerClient->dispose();
    }
#endif
    looper->stop();
#ifdef FILE_DUMP
    if(useVideo && !noOutputFiles) {
        //nn::fs::FlushFile(FileVideo).IsSuccess();
        nn::fs::CloseFile(FileVideo);
    }
    if(useAudio && !noOutputFiles) {
        //nn::fs::FlushFile(FileAudio).IsSuccess();
        nn::fs::CloseFile(FileAudio);
    }
    nn::fs::UnmountHostRoot();
#endif
#ifndef RAPTOR_ENHANCEMENT
    SfGlobal::FreeGlobal();
#endif
    //MV_check("At End - Any Leaked Memory?", 3,0,-1);

    //MV_UseHeap(eMVHeapStandard);          // returning you back to our regular programming..

    // skip these for now due to MemoryPermisson bug (caused by not deleting threads?) so we exit cleanly
    // MV_HeapDestroy(eMVHeapNN);
    // MV_OSMemoryHeapResize(-kExtraHeapMemMB);          // allocate 64 MB of for total heap space


    SUCCEED();
}//NOLINT(impl/function_size)
