﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// #define VERBOSE_LOG 1

#include "MediaDecoder.h"
#include <nn/os.h>
#include <nn/nn_Log.h>
using namespace nn;
using namespace movie;

//static
MediaDecoder* MediaDecoder::Create()
{
    return new MediaDecoder();
}

MediaDecoder::MediaDecoder() :
    m_pCodec(nullptr),
    m_pExtractor(nullptr)
{
}

MediaDecoder::~MediaDecoder()
{
}

int MediaDecoder::DecodeAudioData(const void* in_data, int in_len, void* out_buf, int in_buflen, int* out_samplerate, int* out_bitspersample, int* out_channels, int* out_endian)
{
    NN_LOG("Creating extractor\n");
    movie::PlayerExtractor::Create(&m_pExtractor);

    NN_LOG("Setting extractor datasource\n");
    m_pExtractor->SetDataSource(in_data, in_len);

    MediaData format;
    const char *mime;
    int tracks = 0;
    m_pExtractor->CountTracks(&tracks);

    NN_LOG("Found %d tracks.\n", tracks);

    if(tracks <= 0)
    {
        return 0;
    }

    for (size_t i = 0; i < tracks; ++i) {
        m_pExtractor->GetTrackFormat(i, &format);
        format.FindString("mime", &mime);
        bool isAudio = !strncasecmp(mime, "audio/", 6);
        if (isAudio) {
            NN_LOG("Selecting track %d.\n", i);
            m_pExtractor->SelectTrack(i);
            break;
        }
        else {
            continue;
        }
    }
    NN_LOG("Creating Codec\n");
    movie::Codec::Create(&m_pCodec, mime, false, false);
    MediaDecoderCallback* pCallback = new MediaDecoderCallback(m_pExtractor, m_pCodec, out_buf, in_buflen);

    if (!format.FindInt32("sample-rate", out_samplerate))
        *out_samplerate = 480000;
    if (!format.FindInt32("channel-count", out_channels))
        *out_channels = 1;
    *out_bitspersample = 16;
    *out_endian = 0;

    m_pCodec->SetCallback(pCallback);
    m_pCodec->Configure(&format, NULL, NULL);
    m_pCodec->Start();
    NN_LOG("Started decode\n");
    while(!pCallback->DecodeComplete()){
        nn::os::YieldThread();
    }
    NN_LOG("Decode done\n");
    m_pCodec->Flush();
    m_pCodec->Stop();
    m_pCodec->Release();

    size_t NumBytesDecoded = pCallback->BytesDecoded();

    movie::Codec::Destroy(m_pCodec);
    m_pCodec = nullptr;

    delete(pCallback);

    return NumBytesDecoded;
}

MediaDecoderCallback::MediaDecoderCallback(movie::PlayerExtractor* pExtractor, movie::Codec* pCodec, void* buf, size_t bufLen) :
    m_ReachedEos(false),
    m_pExtractor(pExtractor),
    m_pCodec(pCodec),
    m_decodeComplete(false),
    m_pWritePos(static_cast<uint8_t*>(buf)),
    m_outBufLen(bufLen),
    m_numBytesDecoded(0)
{
}

void MediaDecoderCallback::OnError(Codec* codec, movie::Status result)
{
    NN_LOG("Ran into error.\n");
}

void MediaDecoderCallback::OnInputBufferAvailable(Codec* codec, int index)
{
    int64_t presentationTimeUs = 0;
    uint32_t flags = 0;
    movie::Status movieStatus = movie::Status_Success;
    int32_t offset = 0;
    int32_t size = 0;

#ifdef VERBOSE_LOG
    NN_LOG("Input Buffer Available %d\n", index);
#endif

    if( ( codec != NULL ) && ( codec == m_pCodec) )
    {
        // NN_LOG("Processing input\n");
        Buffer buffer;
        codec->GetInputBuffer(index, &buffer);
        movieStatus = m_pExtractor->ReadSampleData(&buffer);
        if ( movieStatus != movie::Status_Success ||  movieStatus == movie::Status_EndOfStream || m_ReachedEos == true )
        {
            NN_LOG("End of input stream\n");
            m_ReachedEos = true;
            flags |= movie::BufferFlags_EndOfStream;
        }
        else {
            movieStatus = m_pExtractor->GetSampleTime(&presentationTimeUs);
            offset = buffer.Offset();
            size = buffer.Size();
        }

        codec->QueueInputBuffer(
            index,
            offset,
            size,
            presentationTimeUs,
            flags);

        m_pExtractor->Advance();
    }
}

void MediaDecoderCallback::OnOutputBufferAvailable(Codec* codec,
        int index,
        int64_t presentationTimeUs,
        uint32_t flags)
{
#ifdef VERBOSE_LOG
    NN_LOG("Output Buffer Available %d Presentation time = %lld\n", index, presentationTimeUs);
#endif

    if( codec != NULL && ( codec == m_pCodec ) )
    {
#ifdef VERBOSE_LOG
        NN_LOG("Processing output\n");
#endif
        if( flags &= movie::BufferFlags_EndOfStream )
        {
            NN_LOG("End of output stream\n");
            m_decodeComplete = true;
            return;
        }

        int32_t size = 0;
        size_t remains = m_outBufLen;

        Buffer pcmData;
#ifdef VERBOSE_LOG
        NN_LOG("Getting output buffer\n");
#endif
        m_pCodec->GetOutputBuffer(index, &pcmData);
#ifdef VERBOSE_LOG
        NN_LOG("Checking buffer size\n");
#endif
        size = pcmData.Size();

        if (m_outBufLen > m_numBytesDecoded)
            remains = m_outBufLen - m_numBytesDecoded;

        size_t copyLength = remains < size ? remains : size;

        if(pcmData.Base() != nullptr) {

#ifdef VERBOSE_LOG
            NN_LOG("Copy length = %zu, Size = %zu\n", copyLength, size);
#endif
            if (copyLength > 0 && size != 0){
#ifdef VERBOSE_LOG
                NN_LOG("Copying output at %x\n", m_pWritePos);
#endif
                const void* buf = pcmData.Base();
                ::memcpy(m_pWritePos, buf, copyLength);
                m_pWritePos += copyLength;
            }
        }
#ifdef VERBOSE_LOG
        NN_LOG("Releasing output buffer\n");
#endif
        m_pCodec->ReleaseOutputBuffer(index);
        m_numBytesDecoded += size;
#ifdef VERBOSE_LOG
        NN_LOG("Decoded %d bytes\n", m_numBytesDecoded);
#endif
    }
}

void MediaDecoderCallback::OnOutputFormatChanged(Codec* codec, MediaData* format)
{
    //OnFormatChanged(); //TODO
}

bool MediaDecoderCallback::DecodeComplete()
{
    return m_decodeComplete;
}

size_t MediaDecoderCallback::BytesDecoded()
{
    return m_numBytesDecoded;
}
