﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <sys/types.h>

#ifndef off64_t
#define off64_t off_t
#endif

#include <nn/fs.h>
#include <movie/Extractor.h>
#include <movie/PlayerExtractor.h>
#include <movie/DataSource.h>
#include <movie/MSESource.h>
#include <nn/os/os_Mutex.h>
#include <vector>



/****
* Implements a MSE DataSource using NX SDK
* You must initialize fs library (nn::fs::Initialize) before using this class
*/

enum MSEDataType
{
    INITIALIZATION_SEGMENT = 0,
    DATA_SEGMENT,
    END_OF_FILE_SEGMENT
};

enum MSESegmentStatus{
    MSE_INITIALIZATION_SEGMENT = 0,
    MSE_INITIALIZATION_SEGMENT_INCOMPLETE,
    MEDIA_SEGMENT,
    MEDIA_SEGMENT_INCOMPLETE_HEADER,
    MEDIA_SEGMENT_INCOMPLETE,
    INVALID_SEGMENT
};

/*
DataSegment has a similiar structure to a Skip List Node except it doesn't have
a key-value pair.
*/
struct DataSegment
{
    off64_t mOffset;
    off64_t mLength;
    uint8_t* mData;
    bool mPending;
    off64_t mTimeOffsetUs;
    off64_t mCachedDurationUs;
    off64_t mStartTimeUs;
    int32_t mSegmentIndex;
    off64_t mDataBegin;
    off64_t mDataEnd;

    // If DATA_SEGMENT
    //next = next data segment
    //nextInitialization = NULL
    // Else if INITIALIZATION_SEGMENT
    //next = next data segment
    //nextInitialization = next initialization segment
    // Else if END_OF_FILE_SEGMENT
    //next = NULL
    //nextInitialization = NULL
    DataSegment* next;                  // next data segment
    DataSegment* nextInitialization;    // next initialization segment
    MSEDataType type;

    explicit DataSegment(size_t size)
    {
        mOffset = 0;
        mLength = 0;
        mData = nullptr;
        mPending = true;
        next = nullptr;
        nextInitialization = nullptr;
        if (size > 0)
        {
            mData = new uint8_t[size];
        }
        else
        {
            mData = nullptr;
        }

    }

    ~DataSegment()
    {
        delete[] mData;
        mData = nullptr;
        next = nullptr;
        nextInitialization = nullptr;
    }
};

class NXMSESource : public movie::MSESource
{
public:
    //NXMSESource();
    //virtual ~NXMSESource();

    virtual movie::Status AddSourceBuffer(movie::DataSource* dataSource);

    virtual ssize_t GetSourceBufferCount();

    virtual movie::DataSource* GetSourceBuffer(size_t bufferIndex);

    virtual movie::Status RemoveSourceBuffer(size_t bufferIndex);

    virtual movie::Status AdvanceSourceBuffer(size_t bufferIndex);

    virtual bool IsEndOfStreamReached();

private:
    std::vector<movie::DataSource*> m_dataSourceList;
};

class NXMSESourceBuffer : public movie::DataSource
{
public:
    NXMSESourceBuffer();
    virtual ~NXMSESourceBuffer();

    virtual bool initCheck();

    virtual ssize_t readAt(off64_t offset, void *data, size_t size);

    virtual off64_t getSize();

    // List Management
    int64_t appendData(void* data, size_t size, MSEDataType type);
    int64_t removeDataSegment(int64_t dataIndex);
    movie::Status advanceDataSegment();

    // Timing Management
    int64_t getSegmentDuration(int64_t dataIndex);
    int64_t setSegmentStartTime(int64_t dataIndex, int64_t startTime, bool addTimeToFollowingSegments = false);
    int64_t getSegmentStartTime(int64_t dataIndex);
    int64_t getSegmentEndTime(int64_t dataIndex);
    //int mCurrentSegmentCount;

    // Segment Changes
    DataSegment getNextInitializationSegment(void);
    DataSegment getCurrentInitializationSegment(void);
    DataSegment getCurrentDataSegment(void);
    int cancelCurrentSegment(void);

    // Query Management
    bool getEndOfData();
    int getTotalDataBuffered();
    static bool isTypeSupported(const char* in_mimetype, const char* in_codec, const char* in_system, const char* in_url);



    // Unappended Segment Property Querying Methods
    static int64_t getSegmentDuration(void* data, size_t size);
    static MSESegmentStatus getSegmentStatus(void* data, size_t size);

private:
    nn::fs::FileHandle mFile;
    int64_t mOffset;
    int64_t mLength;
    int64_t mTimeOffsetUs;

    int64_t mSegmentIndex;
    bool mEOFReached;
    movie::PlayerExtractor* mExtractor;

    NXMSESourceBuffer(const NXMSESourceBuffer &);
    NXMSESourceBuffer &operator=(const NXMSESourceBuffer &);

    nn::os::Mutex* mMutex;

    // Helper functions
    char* newBuffer(void* data, size_t size);
    char* allocateBuffer(size_t size);

    DataSegment* mSegmentBuffers;                   // head of list
    DataSegment* mCurrentInitializationSegment;     // Current initialization segment that is readat()
    DataSegment* mCurrentDataSegment;               // Current Data Segment that is readat()
    DataSegment* mLastAppendedInitSegment;          // Last Initialization segment appended to buffer
    DataSegment* mLastAppendedSegment;              // Last Data segment appended to buffer
};


