﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <movie/Player.h>
#include <movie/BrowserPlayer.h>
#include <vector>
#include <nn/ssl.h>
#include <nns/mm.h>

#include <memory>

class MediaPlayerObserver : public movie::BrowserPlayerObserver
{
public:
    explicit MediaPlayerObserver(movie::PlayerConfig const&);
    ~MediaPlayerObserver();

    virtual void OnError(movie::Status movieStatus) override;
    virtual void OnStateChange(movie::PlayerState state) override;
    virtual void OnAudioOutputFrameAvailable(movie::AudioFrameInfo* frameInfo) override;
    virtual void OnVideoOutputFrameAvailable(movie::VideoFrameInfo* frameInfo) override;

    virtual void OnBufferingUpdate(float startTime, float endTime) override;
    virtual int32_t OnCurlInit(CURL *easyRequest, const char* uri) override;
    virtual int32_t OnHttpRequest(CURL *easyRequest, const char* uri) override;
    virtual int32_t OnMultiConfig(CURLM *multiRequest, const char* uri) override;
    virtual int32_t OnHttpResponse(CURLM *multiRequest, const char* uri) override;

    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType) override;
    virtual void OnOutputBufferAvailable(int trackNumber, movie::TrackType eTrackType, int64_t presentationTimeUs, int32_t index) override;
    virtual void OnFormatChanged(movie::TrackType eTrackType) override;
    virtual void OnVideoFrameDropped(int64_t presentationTimeUs) override;

    void SetMoviePlayer(movie::BrowserPlayer& player);

    void GetMoviPlayer(movie::BrowserPlayer** player)  { *player = m_Player; }

    void SetAudioTrackNumber(int32_t audioTrackNumber)  { m_AudioTrackNumber = audioTrackNumber; }

    void GetAudioTrackNumber(int32_t *audioTrackNumber)  { *audioTrackNumber = m_AudioTrackNumber; }

    void SetVideoTrackNumber(int32_t videoTrackNumber)  { m_VideoTrackNumber = videoTrackNumber; }

    void GetVideoTrackNumber(int32_t *videoTrackNumber)  { *videoTrackNumber = m_VideoTrackNumber; }

    void ResizeTextures(int width, int height);

    void DrawVideoNvn(int videoBufferIndex, int cropWidth, int cropheight, int yOffset, int uvOffset, int yStride, int colorSpace);

    void WaitForPrepare() { nn::os::WaitEvent(&m_PreparedEvent); }
    void WaitForStart() { nn::os::WaitEvent(&m_StartPlaybackEvent); }
    void WaitForStop() { nn::os::WaitEvent(&m_StopPlaybackEvent); }
    void WaitForComplete() { nn::os::WaitEvent(&m_PlaybackCompleteEvent); }

    template <typename ...Args>
    void GetOutputBufferProperties(Args&&...args) {
        if (m_Player) {
            m_Player->GetOutputBufferProperties(std::forward<Args>(args)...);
        }
    }

    bool IsPlaybackComplete() const { return playback_complete_.load(); }

public:
    struct OutputBufferInfo
    {
        int32_t trackNumber;
        int32_t bufferIndex;
    };

    nn::os::MutexType m_VideoOutputBuffersListMutex;
    std::vector<OutputBufferInfo> m_VideoOutputBuffersList;
    int32_t m_VideoOutputBufferSize {0};
    int32_t m_VideoTrackNumber {0};
    std::unique_ptr<char[]> m_PcmBuffer;
    int32_t m_PcmBufferSize {0};

    static const int m_videoBufferCount = 1;
    NvnBuffer m_videoBufferYUV[m_videoBufferCount];
    movie::Buffer m_videobufferArray[m_videoBufferCount];
    int32_t m_videoBufferIndexArray[m_videoBufferCount];
    int32_t m_PresentationIndex {0};

    nn::os::EventType m_PlaybackCompleteEvent;
    nn::os::EventType m_StartPlaybackEvent;
    nn::os::EventType m_StopPlaybackEvent;
    nn::os::EventType m_PreparedEvent;

    movie::PlayerConfig m_PlayerConfig;

    int m_CropWidth {0};
    int m_CropHeight {0};
    movie::MediaData m_BufferProperty;

private:
    std::unique_ptr<NvnVideoRenderer> m_NvnVideoRenderer;
    std::atomic_flag format_changed_;
    std::atomic_bool playback_complete_{false};

    movie::BrowserPlayer* m_Player {nullptr};
    int32_t m_AudioTrackNumber {0};

    nn::ssl::Context* m_ctxPtr {nullptr};
};
