﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/mem/mem_StandardAllocator.h>

#if __has_include(<memory_resource>)
    #include <memory_resource>
    namespace pmr = std::pmr;
#else
    #include <experimental/memory_resource>
    namespace pmr = ::std::experimental::pmr;
#endif

namespace movie { namespace sample {

struct UntrackedAllocator : pmr::memory_resource {
    void* memory_{};
    size_t sz_{};
    nn::mem::StandardAllocator allocator_;

    UntrackedAllocator(UntrackedAllocator const& other)
        : UntrackedAllocator(other.sz_ / nn::os::MemoryBlockUnitSize)
    {}

    explicit UntrackedAllocator(size_t pages = 1)
        : sz_{(pages > 0 ? pages : 1) * nn::os::MemoryBlockUnitSize}
    {
        uintptr_t address_{};
        nn::os::AllocateMemoryBlock(&address_, sz_);
        memory_ = reinterpret_cast<void*>(address_);
        allocator_.Initialize(memory_, sz_);
    }

    ~UntrackedAllocator() {
        allocator_.Finalize();
        nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(memory_), sz_);
    }

    void *do_allocate(size_t bytes, [[maybe_unused]] size_t alignment) override {
        void* result_ = allocator_.Allocate(bytes, alignment);
        NN_ABORT_UNLESS(result_ != nullptr, "out of memory..");
        return result_;
    }

    void do_deallocate(void *p, size_t bytes, [[maybe_unused]] size_t alignment) override {
        allocator_.Free(p);
    }

    bool do_is_equal(const pmr::memory_resource& other) const noexcept override {
        return &other == this;
    }
};

}} // namespaces
