﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{BlitVic.cpp,PageSampleBlitVic}
 *
 * @brief
 *  Vic Sample Program
 */

/**
 * @page PageSampleBlitVic BlitVic
 * @tableofcontents
 *
 * @brief
 *  It is a commentary of VIC sample program .
 *
 * @section PageSampleBlitVic_SectionBrief Overview
 *  Here , then a description of the sample program using the VIC function
 *
 * @section PageSampleBlitVic_SectionFileStructure File Structure
 *  This sample program are located in the following link:
 *  ../../../Samples/Sources/Applications/BlitVic Samples / Sources / Applications / BlitVic
 *
 * @section PageSampleBlitVic_SectionNecessaryEnvironment Necessary environment
 * Libnvn_src.a, internal developers binary files such as libnvn_bin.a is required .
 * It is not included in the SDK to be provided to the general of the app developers .
 *
 * @section PageSampleBlitVic_SectionHowToOperate Method of operation
 *  Nothing in particular
 *
 * @section PageSampleBlitVic_SectionPrecaution Notes
 * In this demo will be displayed on the LCD using the VIC a monochromatic background and color bar to order .
 * The execution of the program when all of the pattern is displayed to exit .
 *
 * @section PageSampleBlitVic_SectionHowToExecute How to Execute
 *  Build the sample program , please run .。
 *
 * @section PageSampleBlitVic_SectionDetail Detail
 *
 * @subsection PageSampleBlitVic_SectionSampleProgram Sample Program
 *  To quote the source code that nnMain is placed of this sample program below
 *
 *  BlitVic.cpp
 *  @includelineno BlitVic.cpp
 *
 * @subsection PageSampleBlitVic_SectionSampleDetail sample program commentary
 *  Overall picture of the previous sample program is as follows .
 *
 * - To display the red background 5 seconds .
 * - To display the green background 5 seconds .
 * - To display the RGB color bar of 5 seconds .
 * - The end .
 *
 */

#include <memory>
#include <algorithm>
#include <nn/nn_Assert.h>
#include "BlitVicSession.h"
#include "BlitVicSurface.h"
#include "BlitVicController.h"
#include "BlitVicTest.h"

#include "nvblit.h"

#include "Small_Time.h"

#include <nn/init.h>
#include "nn/pcv/pcv.h"


// VIC does not depend on any particular video interface,
// but output is useful for debugging
#define USED_NVDC_VIDEO_INTERFACE   NVDC_DSI

const nn::TimeSpan SleepTime(nn::TimeSpan::FromSeconds(1));


void BlitQuery(nvdcHandle dcHandle)
{
    NvBlitContext *pCtx;
    NvError err = NvBlitOpen( &pCtx );
    NN_ASSERT_EQUAL(NvSuccess, err);


    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    const NvBlitCapability *pBCap = NvBlitQuery( pCtx );


    NN_LOG("\n aaaaa  ----- NvBlitQuery Output ------------------------\n");
    NN_LOG("  Features  : 0x%08x\n",pBCap->Features);
    NN_LOG("  MaxWidth  : %8d\n",pBCap->MaxWidth);
    NN_LOG("  MaxHeight : %8d\n",pBCap->MaxHeight);
    NN_LOG("  MinScale  : %8.3f\n",pBCap->MinScale);
    NN_LOG("  MaxScale  : %8.3f\n",pBCap->MaxScale);



    NvBlitClose ( pCtx );
}




void BackgroundFill(nvdcHandle dcHandle)
{
    NvRmDeviceHandle device;
    NN_ASSERT_EQUAL(NvSuccess, NvRmOpenNew(&device));

    nvdcDisplayInfo info;
    NN_ASSERT(BindDisplay(&info, dcHandle, USED_NVDC_VIDEO_INTERFACE));



    nvdcMode mode;
    NN_ASSERT_EQUAL(NvSuccess, nvdcGetMode(dcHandle, info.boundHead, &mode));

    VicSession session(device, mode.hActive, mode.vActive);
    //                                                                     R    G    B    A (red)
    NN_ASSERT_EQUAL(NvSuccess, session.Execute(nullptr, 0, NvDdkVicFloatColorRec{1.f, 0.f, 0.f, 1.f}));

    NvRmSurface* pTarget = session.GetOutput();
    NN_ASSERT(Flip(dcHandle, pTarget, info));

    std::uint32_t size = NvRmSurfaceComputeSize(pTarget);
    std::unique_ptr<std::uint32_t[]> pPixels(new std::uint32_t[size / sizeof(std::uint32_t)]);
    NN_ASSERT_NOT_EQUAL(nullptr, pPixels.get());

    NvRmSurfaceRead(pTarget, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    // with no input surfaces, everything should be the background color (red)
    for ( std::uint32_t i = 0; i < mode.hActive * mode.vActive; ++i )
    {
        //          AARRGGBB color format (red)
        NN_ASSERT_EQUAL(0xFFFF0000, pPixels[i]);
    }

    nn::os::SleepThread(SleepTime);

    NvRmClose(device);
}

void BackgroundOverride(nvdcHandle dcHandle)
{
    NvRmDeviceHandle device;
    NN_ASSERT_EQUAL(NvSuccess, NvRmOpenNew(&device));

    nvdcDisplayInfo info;
    NN_ASSERT(BindDisplay(&info, dcHandle, USED_NVDC_VIDEO_INTERFACE));

    nvdcMode mode;
    NN_ASSERT_EQUAL(NvSuccess, nvdcGetMode(dcHandle, info.boundHead, &mode));

    // creating a green surface
    NvRmSurface surface;
    NN_ASSERT_EQUAL(NvSuccess, InitializeSurface(&surface, device, mode.hActive, mode.vActive));
    // fill with green ->                AARRGGBB color format
    NN_ASSERT(FillSurface(&surface, 0xFF00FF00));

    VicSession session(device, mode.hActive, mode.vActive);
    // background is red, but shouldn't appear
    NN_ASSERT_EQUAL(NvSuccess, session.Execute(&surface, 1, NvDdkVicFloatColorRec{1.f, 0.f, 0.f, 1.f}));

    NvRmSurface* pTarget = session.GetOutput();
    NN_ASSERT(Flip(dcHandle, pTarget, info));

    std::uint32_t size = NvRmSurfaceComputeSize(pTarget);
    std::unique_ptr<std::uint32_t[]> pPixels(new std::uint32_t[size / sizeof(std::uint32_t)]);
    NN_ASSERT_NOT_EQUAL(nullptr, pPixels.get());

    NvRmSurfaceRead(pTarget, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    // input surface should override background, so testing for green everywhere
    for ( std::uint32_t i = 0; i < mode.hActive * mode.vActive; ++i )
    {
        //          AARRGGBB color format (green)
        NN_ASSERT_EQUAL(0xFF00FF00, pPixels[i]);
    }

    nn::os::SleepThread(SleepTime);

    NvRmClose(device);
}

void AlphaComposition(nvdcHandle dcHandle)
{
    NvRmDeviceHandle device;
    NN_ASSERT_EQUAL(NvSuccess, NvRmOpenNew(&device));

    nvdcDisplayInfo info;
    NN_ASSERT(BindDisplay(&info, dcHandle, USED_NVDC_VIDEO_INTERFACE));

    nvdcMode mode;
    NN_ASSERT_EQUAL(NvSuccess, nvdcGetMode(dcHandle, info.boundHead, &mode));

    // constructing color bars in three different surfaces
    NvRmSurface red;
    NN_ASSERT_EQUAL(NvSuccess, InitializeSurface(&red, device, mode.hActive, mode.vActive));

    // size is consistent across all surfaces
    std::uint32_t size = NvRmSurfaceComputeSize(&red);
    size_t pixelCount = size / sizeof(std::uint32_t);
    std::unique_ptr<std::uint32_t[]> pPixels(new std::uint32_t[pixelCount]);
    NN_ASSERT_NOT_EQUAL(nullptr, pPixels.get());

    std::fill_n(pPixels.get(), pixelCount, 0);

    // number of pixels in a color bar
    const std::uint32_t segment = mode.vActive * mode.hActive / 3;
    // filling with segment with red ->   AARRGGBB color format
    std::fill_n(pPixels.get(), segment, 0xFFFF0000);
    // surface should start as segment red pixels, then black for the rest
    NvRmSurfaceWrite(&red, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    NvRmSurface green;
    NN_ASSERT_EQUAL(NvSuccess, InitializeSurface(&green, device, mode.hActive, mode.vActive));
    // clear buffer
    std::fill_n(pPixels.get(), pixelCount, 0);
    // filling segment with green ->                AARRGGBB color format
    std::fill_n(pPixels.get() + segment, segment, 0xFF00FF00);
    // surface should be black for segment pixels, then green for segment pixels,
    // and finally black for the rest
    NvRmSurfaceWrite(&green, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    NvRmSurface blue;
    NN_ASSERT_EQUAL(NvSuccess, InitializeSurface(&blue, device, mode.hActive, mode.vActive));
    // clear buffer
    std::fill_n(pPixels.get(), pixelCount, 0);
    // filling segment with blue ->                                                       AARRGGBB color format
    std::fill_n(pPixels.get() + segment * 2, mode.vActive * mode.hActive - 2 * segment, 0xFF0000FF);
    // surface should be black for segment * 2 pixels, then blue for the rest
    NvRmSurfaceWrite(&blue, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    VicSession session(device, mode.hActive, mode.vActive);
    NvRmSurface surfaces[] = { red, green, blue };
    NN_ASSERT_EQUAL(NvSuccess, session.Execute(surfaces, 3, NvDdkVicFloatColorRec{0.f, 0.f, 0.f, 0.f}));

    NvRmSurface* pTarget = session.GetOutput();
    NN_ASSERT(Flip(dcHandle, pTarget, info));

    // since alpha was set to zero for the black segment(s) of the surfaces,
    // should have continuous color throughout the final surface

    // final output should be red section, green section, blue section
    NvRmSurfaceRead(pTarget, 0, 0, mode.hActive, mode.vActive, pPixels.get());

    // first section should be red
    for ( std::uint32_t i = 0; i < segment; ++i )
    {
        //          AARRGGBB color format (red)
        NN_ASSERT_EQUAL(0xFFFF0000, pPixels[i]);
    }

    // second section should be green
    for ( std::uint32_t i = 0; i < segment; ++i )
    {
        //          AARRGGBB color format (green)
        NN_ASSERT_EQUAL(0xFF00FF00, pPixels[i + segment]);
    }

    // final section should be blue
    for ( std::uint32_t i = segment * 2; i < mode.hActive * mode.vActive; ++i )
    {
        //          AARRGGBB color format (blue)
        NN_ASSERT_EQUAL(0xFF0000FF, pPixels[i]);
    }

    nn::os::SleepThread(SleepTime);

    NvRmClose(device);
}


void BackgroundFill()
{
    VicTest::SetUpTestCase();
    BackgroundFill(VicTest::dcHandle);
    VicTest::TearDownTestCase();
}

void BackgroundOverride()
{
    VicTest::SetUpTestCase();
    BackgroundOverride(VicTest::dcHandle);
    VicTest::TearDownTestCase();
}

void AlphaComposition()
{
    VicTest::SetUpTestCase();
    AlphaComposition(VicTest::dcHandle);
//    VicTest::TearDownTestCase();
}

const size_t HeapSize  = 256 * 1024 * 1024;
const size_t BlockSize = 16 * 1024 * 1024;

extern "C" void nninitStartup()
{
    NN_ASSERT(nn::os::SetMemoryHeapSize(HeapSize).IsSuccess(), "SetMemoryHeapSize failed.\n");

    uintptr_t address;
    nn::Result result = nn::os::AllocateMemoryBlock(&address, BlockSize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), BlockSize);
}


void TestClockRates()
{
    //nn::pcv::Initialize();
    nn::pcv::Module mdls[] = {nn::pcv::Module_Vic, nn::pcv::Module_Nvenc,
                              nn::pcv::Module_Nvdec, nn::pcv::Module_Nvjpg,
                              nn::pcv::Module_Cpu,   nn::pcv::Module_Gpu};
    const char* mdls_names[]     = {"Module_Vic", "Module_Nvenc",
                              "Module_Nvdec", "Module_Nvjpg",
                              "Module_Cpu",   "Module_Gpu"};

    for(int i = 0; i < sizeof(mdls) / sizeof(mdls[0]); i++)
    {
        nn::pcv::Module mdl = mdls[i];
        const char *name = mdls_names[i];
        nn::pcv::ModuleState state = {0};
        nn::Result result = nn::pcv::GetState(&state, mdl);
        NN_LOG("getState %s result %s \n", name, result.IsSuccess() ? "OK" : "Fail");

        // Print state TODO:: Move printing to Client sample test process
        NN_LOG("Module: %s\n",  name );
        NN_LOG("\tReset:         %s\n", state.resetAsserted ? "TRUE" : "FALSE");
        NN_LOG("\tPower enabled: %s\n", state.powerEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock enabled: %s\n", state.clockEnabled ? "TRUE" : "FALSE");
        NN_LOG("\tClock frequency: %.2f MHz\n\n", state.clockFrequency / (1000.0 * 1000.0));

    }
    //nn::pcv::Finalize();
}


extern "C" void nnMain()
{
    NN_LOG("nninitStartup:%p\n", (void *)nninitStartup);
/*
    nn::pcv::Initialize();
    NN_LOG("pcv::Initialize()\n");


    if(!nn::pcv::IsInitialized())
        NN_LOG("Warning!  : pcv is not initialized!\n");
    else
        NN_LOG("Yeah!  pcv successfully initialized\n");

    // by default, some module's clocks aren't enabled. Start them now
    nn::pcv::Module startclk_mdls[] = {nn::pcv::Module_Vic, nn::pcv::Module_Nvdec,
                                      // nn::pcv::Module_Gpu  // badness happens if use this
                                        };
    nn::Result result;
    for(int i = 0; i < sizeof(startclk_mdls) / sizeof(startclk_mdls[0]); i++)
    {
        nn::pcv::Module mdl = startclk_mdls[i];
        result = SetClockEnabled( mdl, true);
        NN_LOG("SetClockEnabled Module_%d result %s \n", mdl, result.IsSuccess() ? "OK" : "Fail");
    }

    TestClockRates();
*/

    VicTest::SetUpTestCase();
    BlitQuery(VicTest::dcHandle);
    VicTest::TearDownTestCase();


    BackgroundFill();
    BackgroundOverride();
    AlphaComposition();

//    nn::pcv::Finalize();

}
