﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
 /*
 * Copyright 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cassert>
#include <cctype>
#include <fcntl.h>
#include <getopt.h>
#include <csignal>
#include <cstring>
#include <sys/wait.h>
#include <termios.h>

#define LOG_TAG "MediaCodecAacEncoder"
#define ATRACE_TAG ATRACE_TAG_GRAPHICS

const int LOG_NDEBUG = 0;

#include <utils/Log.h>
#include <binder/IPCThreadState.h>
#include <utils/Errors.h>
#include <utils/Timers.h>
#include <utils/Trace.h>
#include <gui/Surface.h>
#include <gui/SurfaceComposerClient.h>
#include <gui/ISurfaceComposer.h>
#include <ui/DisplayInfo.h>
#include <media/openmax/OMX_IVCommon.h>
#include <media/stagefright/foundation/ABuffer.h>
#include <media/stagefright/foundation/AMessage.h>
#include <media/stagefright/MediaCodec.h>
#include <media/stagefright/MediaSource.h>
#include <media/stagefright/MediaErrors.h>
#include <media/stagefright/MediaMuxer.h>
#include <media/ICrypto.h>

#include "fAudioSource.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nn_TimeSpan.h>
#include <nv/nv_MemoryManagement.h>
#include <mm_MemoryManagement.h>
#include <mm_Threads.h>

#define MOUNT_SDCARD
#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#include <string>
#include <cinttypes>

#include <nnt/nnt_Argument.h>

static void usage(const char *me) {
    NN_LOG("usage: %s [-p] Run audio encoder with wav audio file\n"
                 "\t\t[-f] Run audio encoder with fake audio source\n", me);
}

/*
   Expanded this for AAC Encoder test.
   Main steps:
     - to configure the AAC audio encoder;
     - to use realAudioIn/PCM/WAVsamples/fakeAudioSource API
     - feed the encoder with 20 seconds of audio data
     - audio format is 48kHz sample rate, 2-channel
     - output buffer size: 2048 bytes
     - create ADTS header so that it can be played, instead of raw AAC;
     - write bitstream to output via MP4 writer
*/

long double strtold (const char *__restrict, char **__restrict);
_Noreturn void _Exit (int);

#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <ALooper.h>

#include <nn/init.h>
#include <nnt/nntest.h>

#include "WebmFrame.h"

extern void initializeLooperRoster();
extern void initializeAtomizer();
extern void initializeDataSource();

extern void initializeWebFrame();
extern void shutdownWebFrame();

using namespace android;

// aac encoder format
static const char* kMimeTypeAac = "audio/mp4a-latm";
static const char* kBitRateMode = "CBR";       // CBR/CQ

static const uint32_t kBitRate = 1536000;
static const uint32_t kSampleRate = 48000;
static const uint32_t kChannelCount = 2;
static const uint32_t kAacProfile = 2;         // ProfileLevel: ObjectLC = 2, ObjectHE = 5, ObjectELD = 39;
static const uint32_t kMaxTimeLimitSec = 180;  // 3 minutes

// Command-line parameters.
static enum {
    FORMAT_MP4, FORMAT_H264, FORMAT_FRAMES, FORMAT_RAW_FRAMES
} gOutputFormat = FORMAT_MP4;                  // data format for output

static bool aEncoder = false;
static bool pcmSrc = false;
static bool fakeAudioSrc = false;

static uint32_t gTimeLimitSec = kMaxTimeLimitSec;
static uint32_t gFrameBufSize = 4096;
static uint32_t gFrameRate = 40;
static uint8_t  gWavHeaderSize = 0x68;

static int      NUM_FRAMES = 800;

void* wavData;
//nns::audio::WavFormat format;
std::size_t dataSize = 0;
std::size_t gDataOffset = 0;

sp<MediaSource> audioSource;

static volatile bool gStopRequested;

/**
 * create fakeAudioSource from Android MediaSource
 *     struct fAudioSource: public android::MediaSource
 *     fAudioSource(sampleRate, numChannels);
 *     start(params = NULL);
 *     stop();
 *     read(MediaBuffer, readOptions);
 */

namespace{
    const int FsHeapSize = 512 * 1024;
    const int mmHeapSize = 512 << 20;
    const int mmFirmwareMemorySize = 8 << 20;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;
    char                        g_mmHeapBuffer[mmHeapSize];
    nn::mem::StandardAllocator  g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));

    char                        g_mmFirmwareMemory[mmFirmwareMemorySize] __attribute__((aligned(4096)));

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }
    void* MultimediaAllocate(size_t size, size_t alignment, void *userPtr)
    {
        return g_MultimediaAllocator.Allocate(size, alignment);
    }

    void MultimediaFree(void *addr, void *userPtr)
    {
        g_MultimediaAllocator.Free(addr);
    }

    void *MultimediaReallocate(void* addr, size_t newSize, void *userPtr)
    {
        return g_MultimediaAllocator.Reallocate(addr, newSize);
    }
}
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    const size_t mallocSize = 32 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    //ASSERT(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    //ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    NN_SDK_LOG("\ninit done: \n");
}

namespace{
    void *mmAllocate(size_t size, size_t align)
    {
        uintptr_t  address = NULL;
        static size_t totalAllocatedSize = 0;

        nn::Result result = nn::os::AllocateMemoryBlock(&address, size);
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed to allocate memory\n");
        }

        totalAllocatedSize += size;
        //NN_SDK_LOG("Allocated memory address   : 0x%08x\n", address);
        //NN_SDK_LOG("Allocated memory size      : 0x%08x\n", size);
        //NN_SDK_LOG("Total allocated memory size: 0x%08x\n", totalAllocatedSize);
        NN_SDK_LOG("\n");

        return (void *)(address);
    }

    void mmDeallocate(void *address, size_t size)
    {
        //NN_SDK_LOG("Deallocate memory address   : 0x%08x\n", address);
        //NN_SDK_LOG("Deallocate memory size      : 0x%08x\n", size);
        nn::os::FreeMemoryBlock((uintptr_t)address, size);
    }
}


/**
 * Release encoder
 */
static status_t releaseAudioEncoder() {
    status_t err = NO_ERROR;

    return err;
    // Release encoder and cleanup
}


/**
 * Configures and starts the MediaCodec encoder.
 *
 */
static status_t prepareAudioEncoder(sp<MediaCodec>* pCodec){
    status_t err = 0;
    sp<AMessage> format = new AMessage;
    format->setString("mime", kMimeTypeAac);
    format->setInt32("aac-profile", kAacProfile);
    format->setInt32("encoder", true);
    format->setInt32("bitrate", kBitRate);
    format->setInt32("sample-rate", kSampleRate);
    format->setInt32("channel-count", kChannelCount);
    format->setString("bitrate-mode", kBitRateMode);

    NN_LOG("New Looper\n");
    sp<ALooper> looper = new ALooper;
    looper->setName("pcmtoaacconverter_looper");
    looper->start();
    NN_LOG("Creating aac codec\n");
    sp<MediaCodec> codec = MediaCodec::CreateByType(looper, kMimeTypeAac, true);
    if (codec == NULL) {
        NN_LOG( "ERROR: unable to create %s codec instance\n", kMimeTypeAac);
        return UNKNOWN_ERROR;
    }

    err = codec->configure(format,
                           NULL,     // surface
                           NULL,     // crypto
                           MediaCodec::CONFIGURE_FLAG_ENCODE);
    if (err != NO_ERROR) {
        NN_LOG( "ERROR: unable to configure %s codec at %dx%d (err=0x%x)\n",
                kMimeTypeAac, kSampleRate, kBitRate, err);
        codec->release();
        return err;
    }

    NN_LOG("Starting aac codec");
    err = codec->start();
    if (err != NO_ERROR) {
        NN_LOG( "ERROR: unable to start codec (err=0x%x)\n", err);
        codec->release();
        return err;
    }

    NN_LOG("Codec prepared OK!\n");
    *pCodec = codec;
    return err;
}


/**
 * Generates the presentation time for frame N, in microseconds.
 */
static long computePresentationTime(int frameIndex) {
    return frameIndex * 1000000 / gFrameRate;
}


/**
 *  create audio frame date via different options
 */
void generateAudioFrame(int frameIndex, uint8_t* frameData)
{
    //const std::size_t WavHeaderDataSize = 0x68;

    //NN_LOG("\nGenerate Audio Frame data...\n");
    if (fakeAudioSrc) {
        // from fAudioSource read()
        MediaBuffer *Mbuffer;
        audioSource->read(&Mbuffer, NULL);
        memcpy(frameData, Mbuffer->data(), gFrameBufSize);
        Mbuffer->release();
        Mbuffer = NULL;
    }
    else if (pcmSrc) {
        // read from pcm data to frameData (dataSize/wavData)
        if ((dataSize - gWavHeaderSize - gDataOffset) >= gFrameBufSize) {
            memcpy(frameData, ((uint8_t*)wavData + gWavHeaderSize + gDataOffset), gFrameBufSize);
            gDataOffset += gFrameBufSize;
        }
        else {   // last chunk of the audio data in wav file data
            memcpy(frameData, ((uint8_t*)wavData + gWavHeaderSize + gDataOffset), (dataSize - gWavHeaderSize - gDataOffset));
        }
    }
    else {
    // or read audio source input
        ;
    }

    return;
}


/**
 * Since MediaCodec encoder generates the raw AAC stream which needs
 *  to be converted to a playable format, for example ADTS stream
 */
void addADTSHeader(char* buf, int packetLen)
{

    int profile = 2;     // AAC LC
    int freqIndex = 5;   // 48kHz
    int chanConfig = 2;  // CPE

    buf[0] = 0xFF;
    buf[1] = 0xF9;
    buf[2] = ( ((profile - 1)<<6) + (freqIndex<<2) + (chanConfig>>2) );
    buf[3] = ( ((chanConfig&3)<<6) + (packetLen>>11) );
    buf[4] = ( (packetLen&0x7FF) >> 3 );
    buf[5] = ( ((packetLen&7) << 5) + 0x1F );
    buf[6] = 0xFC;
}


/*
 * Runs the MediaCodec encoder, sending the output to the MediaMuxer.
 *
 * Exactly one of muxer
 *
 * The muxer must *not* have been started before calling.
 */
/**
 * to run AAC encoder with AACWriter() or just using muxer
 */
static status_t runAudioEncoder(const sp<MediaCodec>& encoder,
                                const sp<MediaMuxer>& muxer) {
    Vector<sp<ABuffer> > mInBuffers;
    Vector<sp<ABuffer> > mOutBuffers;

    static int kTimeout = 250000;
    ssize_t trackIdx = -1;
    uint32_t debugNumFrames = 0;
    status_t err;
    size_t index;

    bool inputDone = true;
    bool outputDone = false;
    int generateIndex = 0;
    // need to get the audio source via generateAudioFrame()
    uint8_t *frameData = new uint8_t[gFrameBufSize];
    int64_t totalTicks = 0;
    int64_t startTicks;
    int64_t endTicks;

    err = encoder->getInputBuffers(&mInBuffers);
    if (err != NO_ERROR) {
        NN_LOG("\nUnable to get input buffers (err=%d)\n", err);
        return err;
    }
    err = encoder->getOutputBuffers(&mOutBuffers);
    if (err != NO_ERROR) {
        NN_LOG("\nUnable to get output buffers (err=%d)\n", err);
        return err;
    }
    NN_LOG("\nGot %d input and %d output buffers",mInBuffers.size(), mOutBuffers.size());

    while (!outputDone) {
        if(!inputDone) {
            err = encoder->dequeueInputBuffer(&index,kTimeout);
            if (err == OK) {
                long ptsUsec = computePresentationTime(generateIndex);
                if (generateIndex == NUM_FRAMES) {
                    // Send an empty frame with the end-of-stream flag set.  If we set EOS
                    // on a frame with data, that frame data will be ignored, and the
                    // output will be short one frame.
                    err = encoder->queueInputBuffer(
                                index,
                                0 /* offset */,
                                0 /* size */,
                                ptsUsec,
                                MediaCodec::BUFFER_FLAG_EOS);
                    inputDone = true;
                    if (pcmSrc)
                        delete(static_cast<char*>(wavData));
                }
                else{
                    //NN_LOG("\nFilling input buffer %d\n", index);
                    // generate audio frame data from source
                    generateAudioFrame(generateIndex, frameData);
                    const sp<ABuffer> &buffer = mInBuffers.itemAt(index);
                    //NN_LOG("\nBuffer capacity: %d while Frame buffer size: %d\n", buffer->capacity(), gFrameBufSize);
                    if(buffer->capacity() < gFrameBufSize){
                        NN_ASSERT(false &&"Buffer capacity is less than required\n");
                    }
                    // write in buffer frameData
                    // Need to set frame data size
                    memset(buffer->data(), 0, gFrameBufSize);
                    memcpy((uint8_t *)buffer->data(),frameData, gFrameBufSize);
                    uint32_t bufferFlags = 0;
                    err = encoder->queueInputBuffer(
                            index,
                            0 /* offset */,
                            buffer->size(),
                            ptsUsec,
                            bufferFlags);
                }
                generateIndex++;
                //NN_LOG("\nGenerated Frame Index:%d\n", generateIndex);
            } else {
                 NN_LOG("**filling input buffer error %d\n**", index);
            }
        }
        gStopRequested = false;
        if(!gStopRequested) {
            size_t bufIndex, offset, size;
            int64_t ptsUsec;
            uint32_t flags;
            // NN_LOG("\nCalling dequeueOutputBuffer");
            startTicks = nn::os::GetSystemTick().GetInt64Value();
            err = encoder->dequeueOutputBuffer(&bufIndex,
                                               &offset,
                                               &size,
                                               &ptsUsec,
                                               &flags,
                                               kTimeout);
            endTicks = nn::os::GetSystemTick().GetInt64Value();
            totalTicks += endTicks - startTicks;
            // NN_LOG("\ndequeueOutputBuffer returned %d", err);
            inputDone = false;
            switch (err) {
            case NO_ERROR:
                // NN_LOG("\nGot codec config buffer (%zu bytes)", size);
                if ((flags & MediaCodec::BUFFER_FLAG_CODECCONFIG) != 0) {
                    NN_LOG("\nBUFFER_FLAG_CODECCONFIG (%zu bytes)", size);
                    if (muxer != NULL) {
                        // ignore this -- we passed the CSD into MediaMuxer when
                        // we got the format change notification
                        size = 0;
                    }
                }
                if (size != 0) {
                    if(trackIdx == -1){
                        NN_ASSERT(false && "******TrackIdx != -1 error\n");
                    }
                    // NN_LOG("\nSet System time :%d\n",ptsUsec);
                    if (muxer != NULL) {
                        err = muxer->writeSampleData(mOutBuffers[bufIndex], trackIdx,
                                                ptsUsec, flags);
                    }
                    if (err != NO_ERROR) {
                        NN_LOG("\nFailed writing data to muxer (err=%d)\n", err);
                        return err;
                    }
                }
                err = encoder->releaseOutputBuffer(bufIndex);
                if (err != NO_ERROR) {
                    NN_LOG("\nUnable to release output buffer (err=%d)\n",err);
                    return err;
                }
                if ((flags & MediaCodec::BUFFER_FLAG_EOS) != 0) {
                    // Not expecting EOS from SurfaceFlinger.  Go with it.
                    NN_LOG("\nReceived end-of-stream");
                    gStopRequested = true;
                    outputDone = true;
                }
                break;
            case -EAGAIN:                       // INFO_TRY_AGAIN_LATER
                NN_LOG("Got -EAGAIN, looping ..Try again later");//*
                break;
            case INFO_FORMAT_CHANGED:           // INFO_OUTPUT_FORMAT_CHANGED//*
                {
                    // Format includes CSD, which we must provide to muxer.
                    NN_LOG("\nEncoder format changed");
                    sp<AMessage> newFormat;
                    encoder->getOutputFormat(&newFormat);
                    if (muxer != NULL) {
                        trackIdx = muxer->addTrack(newFormat);
                        NN_LOG("Starting muxer with trackIdx : %d\n",trackIdx );
                        err = muxer->start();
                        if (err != NO_ERROR) {
                            NN_LOG("\nUnable to start muxer (err=%d)\n", err);
                            return err;
                        }
                    }
                    else
                    {
                        ++trackIdx;
                    }
                }
                break;
            case INFO_OUTPUT_BUFFERS_CHANGED:   // INFO_OUTPUT_BUFFERS_CHANGED //*
                // Not expected for an encoder; handle it anyway.
                NN_LOG("\nEncoder buffers changed");
                err = encoder->getOutputBuffers(&mOutBuffers);
                if (err != NO_ERROR) {
                    NN_LOG("\nUnable to get new output buffers (err=%d)\n", err);
                    return err;
                }
                break;
            case INVALID_OPERATION:
                NN_LOG("\ndequeueOutputBuffer returned INVALID_OPERATION");
                return err;
            default:
                NN_LOG("\nGot weird result %d from dequeueOutputBuffer\n", err);
                return err;
            }
        }
    }
    NN_LOG("AAC Encoder stopping (req=%d)", gStopRequested);
    NN_LOG("\ndequeing output buffers took %d ms\n", nn::os::ConvertToTimeSpan(nn::os::Tick(totalTicks)).GetMilliSeconds());
    return NO_ERROR;
}//NOLINT(impl/function_size)

// Modified from AudioMemoryPool sample
std::size_t aacReadWavFile(void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("\nWav file size: %d", size);
    *data = new char[size];
    NN_ABORT_UNLESS_NOT_NULL(*data);

    result = nn::fs::ReadFile(handle, 0, *data, size);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    NN_LOG("\nWav file size: %d", size);
    return static_cast<std::size_t>(size);
}

/*
 * Parses args and kicks things off.
 */
TEST(MediaCodecEncoderTest, AacEncoder)
{

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    const char *me = argv[0];
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

    NN_SDK_LOG("Entering Main\n");
    /* Set allocator callback functions */
    nv::mm::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::mm::SetThreadStackAllocator(MultimediaAllocate, MultimediaFree, NULL);
    nv::SetGraphicsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, NULL);
    nv::InitializeGraphics(g_mmFirmwareMemory, sizeof(g_mmFirmwareMemory));

    const int kExtraHeapMemMB = 64;
    status_t err = NO_ERROR;
    sp<MediaCodec> encoder;
    sp<MediaCodec> audioEncoder;

    bool aEncoder = false;
    bool vEncoder = false;
    bool fakeAudioSrc = false;

    int res;

    NN_SDK_LOG("Command line parameters: %d program: %s\n", argc, argv[0] );
#if 0
    if (argc <= 1) {
        usage(me);
        FAIL();
    }
    while ((res = getopt(argc, argv, "afp:")) >= 0) {
        switch (res) {
            case 'f':
            {
                fakeAudioSrc = true;
                break;
            }
            case 'p':
            {
                pcmSrc = true;
                break;
            }

            case '?':
            case 'h':
            default:
            {
                usage(me);
                FAIL();
            }
        }
    }

    argc -= optind;
    argv += optind;

    NN_SDK_LOG("Now argc: %d argv: %s\n", argc, argv[0]);
#endif

    aEncoder = true;
    fakeAudioSrc = false;
    pcmSrc = true;

    initializeLooperRoster();
    initializeAtomizer();
    initializeDataSource();

    const char* inputFileName = "c:/temp/pcm_s16le_48000_2ch.wav";
    const char* outFilePath = "c:/temp/MediaOutAudioOnly.mp4";
    for (int i = 1; i < argc; i++)
    {
        if(!strcmp(argv[i], "--no-output-files")) {
            outFilePath = 0;
            continue;
        }
        if(argc - i < 2)
            break;
        if(!strcmp(argv[i], "--in"))
            inputFileName = argv[++i];
        else if(!strcmp(argv[i], "--out")) {
            outFilePath = argv[++i];
        }
    }
    NN_SDK_LOG( "\n inputFileName = %s\n" , inputFileName);
    std::string path = inputFileName;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    bool sdcardMounted = false;
    if( token == "sdcard" ) {
        NN_LOG("\nUse SDCARD file system, input(pcm): %s", inputFileName);
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if( resultSdcardMount.IsFailure() )
        {
            NN_SDK_LOG( "\n nn::fs::SD card mount failure. Module:%d, Description:%d\n" ,
                    resultSdcardMount.GetModule(),
                    resultSdcardMount.GetDescription());
            FAIL();
        }
        sdcardMounted = true;
    }
    else {   // Host file system
        NN_LOG("\nUse HOST file system, input(pcm): %s", inputFileName);
        nn::Result resultHostMount = nn::fs::MountHostRoot();
        if( resultHostMount.IsFailure() ) {
            NN_SDK_LOG( "\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                resultHostMount.GetModule(),
                resultHostMount.GetDescription() );
            FAIL();
        }
    }

    if (fakeAudioSrc) {
        audioSource = new FAudioSource(kSampleRate, kChannelCount);
        audioSource->start();
    }
    else {
        // read wav/pcm file
        dataSize = aacReadWavFile(&wavData, inputFileName);
    }
    // start audio encoder
    if (aEncoder) {
        // check the input media file
        nn::fs::FileHandle inFileHandle;
        int64_t startTick;
        int64_t endTick;
        inFileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&inFileHandle, inputFileName, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_SDK_LOG( "\n Failed to open %s\n Exiting...", inputFileName);
            FAIL();
        }
        else
        {
             NN_SDK_LOG("\nAudio Input File: %s\n", inputFileName);
            //nn::fs::CloseFile(inFileHandle);
        }

        NN_LOG("Inside Main, Calling PrepareAudioEncoder for AAC encoder\n");
        err = prepareAudioEncoder( &audioEncoder);
        NN_LOG("Prepare AAC encoder returned with 0x%x\n", err);
        if (err != NO_ERROR){
            if (audioEncoder != NULL)
            audioEncoder->release();
            //nn::fs::CloseFile(inFileHandle);
            FAIL();
        }

        // consider using Muxer format mpeg4 with aac tracks only
        sp<MediaMuxer> muxer = NULL;

        NN_LOG("\nMuxer Instance with MPEG4 FORMAT (output: %s)...\n", outFilePath ? outFilePath : "(none)");
        if(outFilePath)
        {
            muxer = new MediaMuxer(outFilePath, MediaMuxer::OUTPUT_FORMAT_MPEG_4);
        }

        NN_LOG("\nRun Audio Encoder...\n");
        startTick = nn::os::GetSystemTick().GetInt64Value();
        err = runAudioEncoder(audioEncoder, muxer);
        endTick = nn::os::GetSystemTick().GetInt64Value();
        NN_LOG("runAudioEncoder took %d ms\n", nn::os::ConvertToTimeSpan(nn::os::Tick(endTick - startTick)).GetMilliSeconds());
        // check error
        if (err != NO_ERROR) {
            NN_LOG("\nrunAudioEncoder failed (err=%d)\n", err);
            if (audioEncoder != NULL)
            audioEncoder->release();
            //nn::fs::CloseFile(inFileHandle);
            //FAIL();
        }

        // release AAC encoder
        NN_LOG("\nRelease AAC Encoder...\n");
        if (audioEncoder != NULL)
            audioEncoder->release();
        if (muxer != NULL) {
            muxer->stop();
        }
        if (audioSource != NULL)
            audioSource->stop();
        //nn::fs::CloseFile(inFileHandle);
    }

    //shutdownWebFrame();
    SUCCEED();
}//NOLINT(impl/function_size)

