﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef WIN32
#define USE_WIN32_MALLOC_IMPLEMENTATION               // #define if want to use Win32 code rather than horizons..
#endif

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#ifdef USE_WIN32_MALLOC_IMPLEMENTATION            // if horizon, we want this header file and link to this library
    #include <Win\platform\WinAlloc.h>            // but if want to use these Win32 malloc implementation, use this instead.
    #include <Win\platform\WinBase\WinStorage.h>
    #pragma comment(lib, "libmm_windows.lib")     // not exactly sure this is going to work in Horizon - need to test
#else
    #include <Horizon\platform\HorizonAlloc.h>
    #include <Horizon\platform\HorizonBase\HorizonStorage.h>
    #include <Horizon\platform\HorizonBase\HorizonFile.h>
#endif

static void TestCreateFile(const char *fileName);
static void TestWriteFile(const char *fileName, const char *str);
static void TestReadFile(const char *fileName);

extern "C" void nnMain()
{
    MV_HeapCreate(32); // specify how much of total heap space used by our malloc…

    TestCreateFile("mine.txt");
    TestWriteFile("mine.txt", "you asked me some questions?");
    TestReadFile("mine.txt");

    MV_HeapDestroy();
}

void TestCreateFile(const char *fileName)
{
    NN_LOG("%s\n", __FUNCTION__);
    NN_LOG("Creating file %s\n", fileName);
#ifdef USE_WIN32_MALLOC_IMPLEMENTATION
    WinStorage storage;
#else
    HorizonStorage storage;
#endif
    MoFile *file = storage.OpenFile(fileName, 'c');
    NN_ASSERT(file != NULL, "Failed to create file %s\n", fileName);
    delete file;
}

void TestWriteFile(const char *fileName, const char *str)
{
    NN_LOG("%s\n", __FUNCTION__);
    NN_LOG("Writing %s to %s\n", str, fileName);
#ifdef USE_WIN32_MALLOC_IMPLEMENTATION
    WinStorage storage;
#else
    HorizonStorage storage;
#endif
    MoFile *file = storage.OpenFile(fileName, 'w');
    NN_ASSERT(file != NULL, "Failed to open file %s\n", fileName);
    file->Write(str, strlen(str));
    delete file;
}

void TestReadFile(const char *fileName)
{
    NN_LOG("%s\n", __FUNCTION__);
    NN_LOG("Reading from %s\n", fileName);
#ifdef USE_WIN32_MALLOC_IMPLEMENTATION
    WinStorage storage;
#else
    HorizonStorage storage;
#endif
    MoFile *file = storage.OpenFile(fileName, 'r');
    NN_ASSERT(file != NULL, "Failed to open file %s\n", fileName);
    int fileSize = file->GetSize();
    char *buffer = new char[fileSize + 1];
    file->Read(buffer, fileSize);
    buffer[fileSize] = 0;
    NN_LOG("File contains: %s\n", buffer);
    delete [] buffer;
    delete file;
}
