﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <memory>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/crypto.h>
#include <nn/migration/idc/migration_Client.h>
#include <nn/migration/idc/migration_ClientContext.h>
#include <nn/migration/idc/migration_CommandApi.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/migration_CommunicationApi.h>
#include <nn/migration/idc/migration_KeyExchangeCommandEncryptor.h>
#include <nn/migration/idc/migration_MessageEncryptor.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/migration_Server.h>
#include <nn/migration/idc/migration_ServerContext.h>
#include <nn/migration/idc/migration_SharedBufferConnection.h>
#include <nn/migration/idc/migration_SharedBufferConnectionManager.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/detail/migration_EncryptionPolicy.h>
#include <nn/migration/idc/detail/migration_Result.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/socket.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nntest.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nn/spl/spl_Api.h>
#endif

using namespace nn;
namespace midc = nn::migration::idc;

const size_t TooLargeCommandSize = std::numeric_limits<size_t>::max() / 2;

const size_t                    ThreadStackSize = 8 * 1024;
NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[ThreadStackSize];
os::ThreadType                  g_Thread;

class TestCancellable : public migration::detail::Cancellable
{
public:
    TestCancellable() {};
};

void RunOnThread(void(*f)(void*), void* arg)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&g_Thread, f, arg, g_ThreadStack, ThreadStackSize, os::DefaultThreadPriority));
    os::StartThread(&g_Thread);
}

void DestroyThread()
{
    os::DestroyThread(&g_Thread);
}

// #define TEST_USER_CONTEXT_LOG(...) NN_LOG(__VA_ARGS__)
#define TEST_USER_CONTEXT_LOG(...) //

class TestUserContext
{
public:
    TestUserContext(size_t consumeCommandSize = 1024, size_t produceCommandSize = 1024) NN_NOEXCEPT
        : m_ConsumeCommandSize(consumeCommandSize)
        , m_ProduceCommandSize(produceCommandSize)
        , m_ConsumedSize(0u)
        , m_ProducedSize(0u)
    {
    }

    Result Consume(void* stream, size_t size) NN_NOEXCEPT
    {
        for( size_t i = 0; i < size; i++ )
        {
            // TEST_USER_CONTEXT_LOG("%02x", reinterpret_cast<Bit8*>(stream)[i]);
            auto val = reinterpret_cast<Bit8*>(stream)[i];
            EXPECT_EQ(val, static_cast<Bit8>(m_ConsumedSize));
            m_ConsumedSize++;
            if( m_ConsumedSize == m_ConsumeCommandSize )
            {
                TEST_USER_CONTEXT_LOG("TestUserContext::Consume: NN_RESULT_SUCCESS\n");
                m_ConsumedSize = 0u;
                NN_RESULT_SUCCESS;
            }
        }
        TEST_USER_CONTEXT_LOG("TestUserContext::Consume: ResultConsumeCommandContinue (consumed %zu/%zu)\n", m_ConsumedSize, m_ConsumeCommandSize);
        return midc::ResultConsumeCommandContinue();
    }
    void SetByteSizeToConsume(size_t) NN_NOEXCEPT
    {
    }
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
    {
        *pOutProducedSize = 0u;
        for( size_t i = 0; i < outStreamSize; i++ )
        {
            reinterpret_cast<Bit8*>(outStream)[i] = static_cast<Bit8>(m_ProducedSize);
            (*pOutProducedSize)++;
            m_ProducedSize++;
            if( m_ProducedSize == m_ProduceCommandSize )
            {
                TEST_USER_CONTEXT_LOG("TestUserContext::Produce: NN_RESULT_SUCCESS\n");
                m_ProducedSize = 0u;
                NN_RESULT_SUCCESS;
            }
        }
        TEST_USER_CONTEXT_LOG("TestUserContext::Produce: ResultProduceCommandContinue (produced %zu/%zu)\n", m_ProducedSize, m_ProduceCommandSize);
        return midc::ResultProduceCommandContinue();
    }
    size_t GetProducableByteSize()
    {
        return m_ProduceCommandSize;
    }
private:
    const size_t m_ConsumeCommandSize;
    const size_t m_ProduceCommandSize;
    size_t m_ConsumedSize;
    size_t m_ProducedSize;
};

template <typename ServerContextType, typename ClientContextType, typename ConnectionType>
void MakeConnectedServerClient(ServerContextType* pServerContext, ClientContextType* pClientContext,
    ConnectionType& serverConnection, ConnectionType& clientConnection)
{
    struct ServerThreadArg
    {
        ServerContextType* pContext;
        ConnectionType* pConnection;
    } arg{ pServerContext, &serverConnection };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ServerThreadArg*>(arg);
        static Bit8 serverBuffer[256];
        NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitConnection(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
    }, &arg);

    static Bit8 clientBuffer[256];
    NNT_ASSERT_RESULT_SUCCESS(midc::Client::Initiate(clientConnection, *pClientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));

    DestroyThread();
}

typedef midc::detail::DefaultEncryptionPolicy EncryptionPolicy;
typedef midc::ServerContext<EncryptionPolicy> ServerContextType;
typedef midc::ClientContext<EncryptionPolicy> ClientContextType;

void MakeConnectedServerClient(ServerContextType* pServerContext, ClientContextType* pClientContext)
{
    static Bit8 buffer[32 * 1024];
    midc::SharedBufferConnectionManager connectionManager(buffer, sizeof(buffer));
    midc::SharedBufferConnection serverConnection, clientConnection;
    ASSERT_TRUE(connectionManager.CreateConnection(&serverConnection, &clientConnection, 2048));

    MakeConnectedServerClient(pServerContext, pClientContext, serverConnection, clientConnection);
}

void MakeConnectedServerContext(ServerContextType* pServerContext)
{
    ClientContextType clientContext;
    MakeConnectedServerClient(pServerContext, &clientContext);
}

void MakeConnectedClientContext(ClientContextType* pClientContext)
{
    ServerContextType serverContext;
    MakeConnectedServerClient(&serverContext, pClientContext);
}

void MakeSuspendedServerClient(ServerContextType* pServerContext, ClientContextType* pClientContext)
{
    static Bit8 buffer[32 * 1024];
    midc::SharedBufferConnectionManager connectionManager(buffer, sizeof(buffer));
    midc::SharedBufferConnection serverConnection, clientConnection;
    ASSERT_TRUE(connectionManager.CreateConnection(&serverConnection, &clientConnection, 2048));

    MakeConnectedServerClient(pServerContext, pClientContext, serverConnection, clientConnection);

    struct ServerThreadArg
    {
        ServerContextType* pContext;
        midc::SharedBufferConnection* pConnection;
    } arg{ pServerContext, &serverConnection };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ServerThreadArg*>(arg);
        Bit8 serverBuffer[256];
        NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitTermination(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
    }, &arg);

    Bit8 clientBuffer[256];
    NNT_ASSERT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, *pClientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));

    DestroyThread();
}

void MakeSuspendedServerContext(ServerContextType* pServerContext)
{
    ClientContextType clientContext;
    MakeSuspendedServerClient(pServerContext, &clientContext);
}

void MakeSuspendedClientContext(ClientContextType* pClientContext)
{
    ServerContextType serverContext;
    MakeSuspendedServerClient(&serverContext, pClientContext);
}

void MakeWaitInitiationServerClient(ServerContextType* pOutServerContext, ClientContextType* pOutClientContextType)
{
    pOutClientContextType->SetCommandKind(midc::CommandKind::Initiate0);
    size_t requestSize;
    midc::Initiate0Request initiate0Request;
    NNT_ASSERT_RESULT_SUCCESS(pOutClientContextType->GetProducer().Produce(&requestSize, &initiate0Request, sizeof(initiate0Request)));
    ASSERT_EQ(requestSize, sizeof(initiate0Request));
    NNT_ASSERT_RESULT_SUCCESS(pOutServerContext->Consume(&initiate0Request, sizeof(initiate0Request)));

    size_t responseSize;
    midc::Initiate0Response initiate0Response;
    NNT_ASSERT_RESULT_SUCCESS(pOutServerContext->Produce(&responseSize, &initiate0Response, sizeof(initiate0Response)));
    ASSERT_EQ(responseSize, sizeof(initiate0Response));
    NNT_ASSERT_RESULT_SUCCESS(pOutClientContextType->GetConsumer().Consume(&initiate0Response, sizeof(initiate0Response)));
}

void MakeWaitInitiationServerContext(ServerContextType* pOutServerContext)
{
    ClientContextType clientContext;
    MakeWaitInitiationServerClient(pOutServerContext, &clientContext);
}

void MakeWaitInitiationClientContext(ClientContextType* pClientContext)
{
    ServerContextType serverContext;
    MakeWaitInitiationServerClient(&serverContext, pClientContext);
}

void MakeWaitResumeServerClient(ServerContextType* pOutServerContext, ClientContextType* pOutClientContextType)
{
    MakeSuspendedServerClient(pOutServerContext, pOutClientContextType);

    pOutClientContextType->SetCommandKind(midc::CommandKind::Resume0);
    size_t requestSize;
    midc::Resume0Request resume0Request;
    NNT_ASSERT_RESULT_SUCCESS(pOutClientContextType->GetProducer().Produce(&requestSize, &resume0Request, sizeof(resume0Request)));
    ASSERT_EQ(requestSize, sizeof(resume0Request));
    NNT_ASSERT_RESULT_SUCCESS(pOutServerContext->Consume(&resume0Request, sizeof(resume0Request)));

    size_t responseSize;
    midc::Resume0Response resume0Response;
    NNT_ASSERT_RESULT_SUCCESS(pOutServerContext->Produce(&responseSize, &resume0Response, sizeof(resume0Response)));
    ASSERT_EQ(responseSize, sizeof(resume0Response));
    NNT_ASSERT_RESULT_SUCCESS(pOutClientContextType->GetConsumer().Consume(&resume0Response, sizeof(resume0Response)));
}

void MakeWaitResumeServerContext(ServerContextType* pOutServerContext)
{
    ClientContextType clientContext;
    MakeWaitResumeServerClient(pOutServerContext, &clientContext);
}

void MakeWaitResumeClientContext(ClientContextType* pOutClientContext)
{
    ServerContextType serverContext;
    MakeWaitResumeServerClient(&serverContext, pOutClientContext);
}

class MigrationIdcServerClientTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::InitializeForCrypto();
#endif
    }
    static void TearDownTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::Finalize();
#endif
    }
};

class MigrationIdcSharedBufferConnectionServerClientTest : public MigrationIdcServerClientTest
{
public:
    typedef midc::SharedBufferConnection ConnectionType;

private:
    midc::SharedBufferConnectionManager m_SharedBufferConnectionManager;
    Bit8 m_ConnectionBuffer[32 * 1024];

public:

    MigrationIdcSharedBufferConnectionServerClientTest() :
        m_SharedBufferConnectionManager(m_ConnectionBuffer, sizeof(m_ConnectionBuffer))
    {
    }
    void GetConnection(ConnectionType* pServerConnection, ConnectionType* pClientConnection)
    {
        ASSERT_TRUE(m_SharedBufferConnectionManager.CreateConnection(pServerConnection, pClientConnection, 2048));
    }
protected:
    virtual void SetUp()
    {
    }
    virtual void TearDown()
    {
    }
    static void SetUpTestCase()
    {
        MigrationIdcServerClientTest::SetUpTestCase();
    }

    static void TearDownTestCase()
    {
        MigrationIdcServerClientTest::TearDownTestCase();
    }
};

class MigrationIdcSocketConnectionServerClientTest : public MigrationIdcServerClientTest
{
public:
    typedef midc::SocketConnection ConnectionType;

private:
    nn::socket::SockAddrIn  m_ServerSocketAddress;          // Client から接続するための Server 側のアドレスとポートの情報。
    int                     m_ServerAcceptSocketDescriptor; // Server が Listen, Accept に使うソケット。
    int                     m_ServerSocketDescriptor;       // Server が Send と Receive に使うソケット（Acceptで取得）。
    int                     m_ClientSocketDescriptor = -1;  // Client が Send と Receive に使うソケット。
    os::Event               m_ConnectionReadyEvent;

public:
    MigrationIdcSocketConnectionServerClientTest() :
        m_ServerAcceptSocketDescriptor(-1),
        m_ServerSocketDescriptor(-1),
        m_ClientSocketDescriptor(-1),
        m_ConnectionReadyEvent(os::EventClearMode_ManualClear)
    {
    }

    void GetConnection(ConnectionType* pServerConnection, ConnectionType* pClientConnection)
    {
        new (pServerConnection) ConnectionType(m_ServerSocketDescriptor);
        new (pClientConnection) ConnectionType(m_ClientSocketDescriptor);
    }

private:
    void SetupServer()
    {
        NN_LOG("-- Server startup --\n");

        if( (m_ServerAcceptSocketDescriptor = socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)) < 0 )
        {
            NN_ABORT("server: Socket failed (error %d)\n", socket::GetLastError());
        }

        nn::socket::SockAddrIn serverSocketAddress = { 0 };
        serverSocketAddress.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        serverSocketAddress.sin_port = nn::socket::InetHtons(0);
        serverSocketAddress.sin_family = nn::socket::Family::Af_Inet;

        if( socket::Bind(m_ServerAcceptSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&serverSocketAddress), sizeof(serverSocketAddress)) < 0 )
        {
            NN_ABORT("server: Bind failed (error %d)\n", nn::socket::GetLastError());
        }

        nn::socket::SockLenT saLen = sizeof(m_ServerSocketAddress);
        if( socket::GetSockName(m_ServerAcceptSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&m_ServerSocketAddress), &saLen) < 0 )
        {
            NN_ABORT("server: GetSockName failed (error %d)\n", nn::socket::GetLastError());
        }

        auto result = nifm::GetCurrentPrimaryIpAddress(&m_ServerSocketAddress.sin_addr);
        if( result.IsSuccess() )
        {
            NN_LOG("server: listening for incoming messages at: %s:%d\n",
                socket::InetNtoa(m_ServerSocketAddress.sin_addr),
                static_cast<int>(socket::InetNtohs(m_ServerSocketAddress.sin_port)));
        }
        else
        {
            NN_ABORT("server: GetCurrentIpAddress failed (error 0x%08x)\n", result.GetInnerValueForDebug());
        }

        if( socket::Listen(m_ServerAcceptSocketDescriptor, 4) < 0 )
        {
            NN_ABORT("server: listen failed (error %d)\n", nn::socket::GetLastError());
        }
    }

    void CreateConnection()
    {
        RunOnThread([](void* arg)
        {
            auto testInstance = reinterpret_cast<MigrationIdcSocketConnectionServerClientTest*>(arg);
            nn::socket::SockAddrIn saClientAddress = { 0 };
            nn::socket::SockLenT clientAddressSize = sizeof(saClientAddress);
            // メモ : Accept はタイムアウトしない。
            if( (testInstance->m_ServerSocketDescriptor = nn::socket::Accept(testInstance->m_ServerAcceptSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saClientAddress), &clientAddressSize)) < 0 )
            {
                NN_LOG("server: Accept failed (error %d)\n", nn::socket::GetLastError());
                return;
            }
            NN_LOG("server: connection accepted from %s\n", nn::socket::InetNtoa(saClientAddress.sin_addr));
        }, this);

        NN_LOG("-- Client startup --\n");
        if( (m_ClientSocketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)) < 0 )
        {
            NN_LOG("client: Socket failed (error %d)\n", nn::socket::GetLastError());
            return;
        }

        NN_LOG("client: connecting to %s:%d\n", socket::InetNtoa(m_ServerSocketAddress.sin_addr), static_cast<int>(socket::InetNtohs(m_ServerSocketAddress.sin_port)));

        if( socket::Connect(m_ClientSocketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&m_ServerSocketAddress), sizeof(m_ServerSocketAddress)) < 0 )
        {
            NN_LOG("client: Connect failed (error %d)\n", nn::socket::GetLastError());
            return;
        }
        NN_LOG("client: established connection to server\n");
        m_ConnectionReadyEvent.Signal();

        m_ConnectionReadyEvent.Wait();

        DestroyThread();
    }

    void CleanupConnection()
    {
        m_ConnectionReadyEvent.Clear();
        socket::Close(m_ServerSocketDescriptor);
        socket::Close(m_ClientSocketDescriptor);
        socket::Close(m_ServerAcceptSocketDescriptor);
    }

protected:
    virtual void SetUp()
    {
        SetupServer();
        CreateConnection();
    }
    virtual void TearDown()
    {
        CleanupConnection();
    }

    static void SetUpTestCase()
    {
        MigrationIdcServerClientTest::SetUpTestCase();
        static socket::ConfigDefaultWithMemory s_SocketConfigWithMemory;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(s_SocketConfigWithMemory));
        nifm::SubmitNetworkRequestAndWait();
        NN_ABORT_UNLESS(nifm::IsNetworkAvailable());
    }

    static void TearDownTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Finalize());
        MigrationIdcServerClientTest::TearDownTestCase();
    }
};

template <typename EncryptionPolicy, size_t UserCommandSize>
void ServerClientTest()
{
    static Bit8 connectionBuffer[32 * 1024];
    midc::SharedBufferConnectionManager connectionManager(connectionBuffer, sizeof(connectionBuffer));
    midc::SharedBufferConnection clientConnection, serverConnection;
    ASSERT_TRUE(connectionManager.CreateConnection(&clientConnection, &serverConnection, 8 * 1024));

    RunOnThread([](void* arg) {

        midc::ServerContext<EncryptionPolicy> serverContext;

        static Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBuffer<ServerContextType>)];
        TestCancellable serverCancellable;

        auto pConnection = reinterpret_cast<midc::SharedBufferConnection*>(arg);

        // (1) Initiate -> Terminate で1回接続して終了。
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitConnection(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitTermination(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));

        // (2) Resume ->再開後に UserCommand を受信して Terminate で終了。
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitConnection(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));
        {
            TestUserContext serverUserContext(UserCommandSize, UserCommandSize);
            auto result = midc::Server::WaitUserCommand<midc::SharedBufferConnection, ServerContextType, TestUserContext>(*pConnection, serverContext, serverUserContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable);
            NNT_EXPECT_RESULT_SUCCESS(result);
        }
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitTermination(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));

        // (3) Resume ->再開後に Terminate で終了。
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitConnection(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitTermination(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));
    }, &serverConnection);

    midc::ClientContext<EncryptionPolicy> clientContext;

    static Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
    TestCancellable clientCancellable;

    // (1) Initiate -> Terminate で1回接続して終了。
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Initiate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    // (2) Resume -> UserCommand -> Terminate で1回接続して終了。
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Resume(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));
    {
        TestUserContext clientUserContext(UserCommandSize, UserCommandSize);
        auto result = midc::Client::InvokeUserCommand<midc::SharedBufferConnection, ClientContextType, TestUserContext>(clientConnection, clientContext, clientUserContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable);
        NNT_EXPECT_RESULT_SUCCESS(result);
    }
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    // (3) Resume -> Terminate で1回接続して終了。
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Resume(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));
    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcServerClientTest, ServerClient)
{
    ServerClientTest<midc::detail::DefaultEncryptionPolicy, 1>();
    ServerClientTest<midc::detail::DefaultEncryptionPolicy, 8 * 1024 * 1024>();
}

template <typename TestClass>
void ClientInitiateCancelTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);

    ClientContextType clientContext;

    Bit8 clientBuffer[256];
    TestCancellable clientCancellable;

    // 1秒後に Cancel。
    RunOnThread([](void* arg) {
        auto pCancellable = reinterpret_cast<TestCancellable*>(arg);
        os::SleepThread(TimeSpan::FromSeconds(1));
        pCancellable->Cancel();
    }, &clientCancellable);

    NNT_EXPECT_RESULT_FAILURE(midc::ResultCanceled, midc::Client::Initiate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientInitiateCancel)
{
    ClientInitiateCancelTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientInitiateCancel)
{
    ClientInitiateCancelTest(this);
}

template <typename TestClass>
void ClientInitiateTimeoutTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);

    ClientContextType clientContext;
    Bit8 clientBuffer[256];

    NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::Initiate(clientConnection, clientContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientInitiateTimeout)
{
    ClientInitiateTimeoutTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientInitiateTimeout)
{
    ClientInitiateTimeoutTest(this);
}

template <typename TestClass>
void ClientInitiatePeerCloseTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    serverConnection.Close();

    ClientContextType clientContext;
    Bit8 clientBuffer[256];

    NNT_EXPECT_RESULT_FAILURE(midc::ResultPeerClosed, midc::Client::Initiate(clientConnection, clientContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientInitiatePeerCloseTest)
{
    ClientInitiatePeerCloseTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientInitiatePeerCloseTest)
{
    ClientInitiatePeerCloseTest(this);
}

template <typename TestClass>
void ClientUserCommandCancelTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
    TestCancellable clientCancellable;

    // 1秒後に Cancel。
    RunOnThread([](void* arg) {
        auto pCancellable = reinterpret_cast<TestCancellable*>(arg);
        os::SleepThread(TimeSpan::FromSeconds(1));
        pCancellable->Cancel();
    }, &clientCancellable);

    TestUserContext clientUserContext;
    NNT_EXPECT_RESULT_FAILURE(midc::ResultCanceled, midc::Client::InvokeUserCommand(clientConnection, clientContext, clientUserContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientUserCommandCancel)
{
    ClientUserCommandCancelTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientUserCommandCancel)
{
    ClientUserCommandCancelTest(this);
}

template <typename TestClass>
void ClientUserCommandTimeoutTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
    TestUserContext clientUserContext;

    NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(clientConnection, clientContext, clientUserContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientUserCommandTimeout)
{
    ClientUserCommandTimeoutTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientUserCommandTimeout)
{
    ClientUserCommandTimeoutTest(this);
}

template <typename TestClass>
void ClientUserCommandPeerCloseTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    serverConnection.Close();

    Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
    TestUserContext clientUserContext;

    NNT_EXPECT_RESULT_FAILURE(midc::ResultPeerClosed, midc::Client::InvokeUserCommand(clientConnection, clientContext, clientUserContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ClientUserCommandPeerClose)
{
    ClientUserCommandPeerCloseTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ClientUserCommandPeerClose)
{
    ClientUserCommandPeerCloseTest(this);
}

template <typename TestClass>
void ServerWaitConnectionCancelTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;

    Bit8 serverBuffer[256];
    TestCancellable serverCancellable;

    // 1秒後に Cancel。
    RunOnThread([](void* arg) {
        auto pCancellable = reinterpret_cast<TestCancellable*>(arg);
        os::SleepThread(TimeSpan::FromSeconds(1));
        pCancellable->Cancel();
    }, &serverCancellable);

    NNT_EXPECT_RESULT_FAILURE(midc::ResultCanceled, midc::Server::WaitConnection(serverConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitConnectionCancel)
{
    ServerWaitConnectionCancelTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitConnectionCancel)
{
    ServerWaitConnectionCancelTest(this);
}

template <typename TestClass>
void ServerWaitConnectionTimeoutTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;

    Bit8 serverBuffer[256];

    NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Server::WaitConnection(serverConnection, serverContext, 1, serverBuffer, sizeof(serverBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitConnectionTimeout)
{
    ServerWaitConnectionTimeoutTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitConnectionTimeout)
{
    ServerWaitConnectionTimeoutTest(this);
}

template <typename TestClass>
void ServerWaitConnectionPeerCloseTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;

    Bit8 serverBuffer[256];
    clientConnection.Close();

    NNT_EXPECT_RESULT_FAILURE(midc::ResultPeerClosed, midc::Server::WaitConnection(serverConnection, serverContext, 1, serverBuffer, sizeof(serverBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitConnectionPeerClose)
{
    ServerWaitConnectionPeerCloseTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitConnectionPeerClose)
{
    ServerWaitConnectionPeerCloseTest(this);
}

template <typename TestClass>
void ServerWaitUserCommandCancelTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBuffer<ServerContextType>)];
    TestCancellable serverCancellable;

    // 1秒後に Cancel。
    RunOnThread([](void* arg) {
        auto pCancellable = reinterpret_cast<TestCancellable*>(arg);
        os::SleepThread(TimeSpan::FromSeconds(1));
        pCancellable->Cancel();
    }, &serverCancellable);

    TestUserContext serverUserContext;
    NNT_EXPECT_RESULT_FAILURE(midc::ResultCanceled, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandCancel)
{
    ServerWaitUserCommandCancelTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandCancel)
{
    ServerWaitUserCommandCancelTest(this);
}

template <typename TestClass>
void ServerWaitUserCommandTimeoutTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBuffer<ServerContextType>)];

    TestUserContext serverUserContext;
    NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 1, serverBuffer, sizeof(serverBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandTimeout)
{
    ServerWaitUserCommandTimeoutTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandTimeout)
{
    ServerWaitUserCommandTimeoutTest(this);
}

template <typename TestClass>
void ServerWaitUserCommandPeerCloseTest(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBuffer<ServerContextType>)];
    clientConnection.Close();

    TestUserContext serverUserContext;
    NNT_EXPECT_RESULT_FAILURE(midc::ResultPeerClosed, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 1, serverBuffer, sizeof(serverBuffer), nullptr));
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandPeerClose)
{
    ServerWaitUserCommandPeerCloseTest(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandPeerClose)
{
    ServerWaitUserCommandPeerCloseTest(this);
}

// 以下、受信消費並列版用のテスト。

migration::detail::ThreadResourceType GetServerThreadResource()
{
    NN_OS_ALIGNAS_THREAD_STACK static char s_ServerThreadResourceStack[8 * 1024];
    migration::detail::ThreadResourceType threadResource;
    threadResource.name = "ServerThread";
    threadResource.priority = os::DefaultThreadPriority;
    threadResource.stack = s_ServerThreadResourceStack;
    threadResource.stackSize = sizeof(s_ServerThreadResourceStack);
    return threadResource;
}

migration::detail::ThreadResourceType GetClientThreadResource()
{
    NN_OS_ALIGNAS_THREAD_STACK static char s_ClientThreadResourceStack[8 * 1024];
    migration::detail::ThreadResourceType threadResource;
    threadResource.name = "ClientThread";
    threadResource.priority = os::DefaultThreadPriority;
    threadResource.stack = s_ClientThreadResourceStack;
    threadResource.stackSize = sizeof(s_ClientThreadResourceStack);
    return threadResource;
}

template <typename EncryptionPolicy, size_t UserCommandSize>
void ServerClientUserCommandTest_Threaded()
{
    static Bit8 connectionBuffer[32 * 1024];
    midc::SharedBufferConnectionManager connectionManager(connectionBuffer, sizeof(connectionBuffer));
    midc::SharedBufferConnection clientConnection, serverConnection;
    ASSERT_TRUE(connectionManager.CreateConnection(&clientConnection, &serverConnection, 8 * 1024));

    RunOnThread([](void* arg) {

        auto pConnection = reinterpret_cast<midc::SharedBufferConnection*>(arg);

        midc::ServerContext<midc::detail::DebugEncryptionPolicy> serverContext;
        static Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];
        TestCancellable serverCancellable;

        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitConnection(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));

        auto threadResource = GetServerThreadResource();
        TestUserContext serverUserContext(UserCommandSize, UserCommandSize);
        auto result = midc::Server::WaitUserCommand<decltype(*pConnection), decltype(serverContext)>(*pConnection, serverContext, serverUserContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr);
        NNT_EXPECT_RESULT_SUCCESS(result);

        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitTermination(*pConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), &serverCancellable));
    }, &serverConnection);

    midc::ClientContext<midc::detail::DebugEncryptionPolicy> clientContext;
    static Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>)];
    TestCancellable clientCancellable;

    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Initiate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    auto threadResource = GetClientThreadResource();
    TestUserContext clientUserContext(UserCommandSize, UserCommandSize);
    auto result = midc::Client::InvokeUserCommand<decltype(clientConnection), decltype(clientContext)>(clientConnection, clientContext, clientUserContext, 0, &threadResource, clientBuffer, sizeof(clientBuffer), nullptr);
    NNT_EXPECT_RESULT_SUCCESS(result);

    NNT_EXPECT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), &clientCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcServerClientTest, ServerClientUserCommand_Threaded)
{
    ServerClientUserCommandTest_Threaded<midc::detail::DefaultEncryptionPolicy, 1>();
    ServerClientUserCommandTest_Threaded<midc::detail::DefaultEncryptionPolicy, 8 * 1024 * 1024>();

}

template <typename TestClass>
void ServerWaitUserCommandCancelTest_Threaded(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    TestCancellable serverCancellable;

    struct ThreadArg
    {
        typename TestClass::ConnectionType* pConnection;
        ClientContextType* pContext;
        TestCancellable* pServerCancellable;
    } threadArg{ &clientConnection, &clientContext, &serverCancellable };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ThreadArg*>(arg);
        Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>)];
        // スレッド処理が始まってからのキャンセルを確認するため、タイムアウト時間（1秒）以内には受信しきれないデータ量を設定して、ある程度データ送信されたから Server 側のキャンセルを行う。
        TestUserContext clientUserContext(TooLargeCommandSize, TooLargeCommandSize);
        NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
        pArg->pServerCancellable->Cancel();
    }, &threadArg);

    auto threadResource = GetServerThreadResource();
    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];

    TestUserContext serverUserContext(TooLargeCommandSize, TooLargeCommandSize);
    NNT_EXPECT_RESULT_FAILURE(midc::ResultCanceled, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), &serverCancellable));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandCancel_Threaded)
{
    ServerWaitUserCommandCancelTest_Threaded(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandCancel_Threaded)
{
    ServerWaitUserCommandCancelTest_Threaded(this);
}

template <typename TestClass>
void ServerWaitUserCommandTimeoutTest_Threaded(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    struct ThreadArg
    {
        typename TestClass::ConnectionType* pConnection;
        ClientContextType* pContext;
    } threadArg{ &clientConnection, &clientContext };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ThreadArg*>(arg);

        Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
        // タイムアウト時間（1秒）以内には受信しきれないデータ量を設定する。
        TestUserContext clientUserContext(TooLargeCommandSize, TooLargeCommandSize);
        NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));

    }, &threadArg);

    auto threadResource = GetServerThreadResource();
    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];

    // タイムアウト時間（1秒）以内には受信しきれないデータ量を設定する。
    TestUserContext serverUserContext(TooLargeCommandSize, TooLargeCommandSize);
    NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 1, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandTimeout_Threaded)
{
    ServerWaitUserCommandTimeoutTest_Threaded(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandTimeout_Threaded)
{
    ServerWaitUserCommandTimeoutTest_Threaded(this);
}

template <typename TestClass>
void ServerWaitUserCommandPeerCloseTest_Threaded(TestClass* pTestClass)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    struct ThreadArg
    {
        typename TestClass::ConnectionType* pConnection;
        ClientContextType* pContext;
    } threadArg{ &clientConnection, &clientContext };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ThreadArg*>(arg);

        Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>)];
        // タイムアウト時間（1秒）以内には受信しきれないデータ量を設定する。
        TestUserContext clientUserContext(TooLargeCommandSize, TooLargeCommandSize);
        NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 1, clientBuffer, sizeof(clientBuffer), nullptr));
        pArg->pConnection->Close();
    }, &threadArg);

    auto threadResource = GetServerThreadResource();
    Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];

    TestUserContext serverUserContext(TooLargeCommandSize, TooLargeCommandSize);
    NNT_EXPECT_RESULT_FAILURE(midc::ResultPeerClosed, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr));

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ServerWaitUserCommandPeerClose_Threaded)
{
    ServerWaitUserCommandPeerCloseTest_Threaded(this);
}

TEST_F(MigrationIdcSocketConnectionServerClientTest, ServerWaitUserCommandPeerClose_Threaded)
{
    ServerWaitUserCommandPeerCloseTest_Threaded(this);
}

// 1度に送受信されるバッファのサイズを絞って送受信のテスト（コマンドが細切れに送受信されても問題ないことの確認）
void CommunicationBufferSizeTest(bool useThreadedCommandMediator)
{
    for( int communicationBufferSize = 1; communicationBufferSize < 256; communicationBufferSize++ )
    {
        NN_LOG("Communication Buffer Size : %d\n", communicationBufferSize);
        ServerContextType serverContext;
        ClientContextType clientContext;

        static Bit8 buffer[32 * 1024];
        midc::SharedBufferConnectionManager connectionManager(buffer, sizeof(buffer));
        midc::SharedBufferConnection serverConnection, clientConnection;
        connectionManager.CreateConnection(&serverConnection, &clientConnection, communicationBufferSize);

        struct ServerThreadArg
        {
            ServerContextType* pContext;
            midc::SharedBufferConnection* pConnection;
            bool useThreadedCommandMediator;
        } arg{ &serverContext, &serverConnection, useThreadedCommandMediator };

        // Initiate -> UserCmmand -> Terminate -> Resume -> UserCommand -> Terminate の順でテスト。

        RunOnThread([](void* arg) {
            auto pArg = reinterpret_cast<ServerThreadArg*>(arg);
            NN_STATIC_ASSERT(sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>) >= sizeof(midc::Server::WaitUserCommandBuffer<ServerContextType>));
            static Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];
            NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitConnection(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
            {
                TestUserContext userContext;
                if( pArg->useThreadedCommandMediator )
                {
                    auto threadResource = GetServerThreadResource();
                    NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitUserCommand(*pArg->pConnection, *pArg->pContext, userContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr));
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitUserCommand(*pArg->pConnection, *pArg->pContext, userContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
                }
            }
            NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitTermination(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));

            NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitConnection(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
            {
                TestUserContext userContext;
                if( pArg->useThreadedCommandMediator )
                {
                    auto threadResource = GetServerThreadResource();
                    NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitUserCommand(*pArg->pConnection, *pArg->pContext, userContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr));
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitUserCommand(*pArg->pConnection, *pArg->pContext, userContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
                }
            }
            NNT_ASSERT_RESULT_SUCCESS(midc::Server::WaitTermination(*pArg->pConnection, *pArg->pContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
        }, &arg);

        static Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>)];
        NN_STATIC_ASSERT(sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>) >= sizeof(midc::Client::InvokeUserCommandBuffer<ClientContextType>));
        NNT_ASSERT_RESULT_SUCCESS(midc::Client::Initiate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
        {
            TestUserContext userContext;
            if( useThreadedCommandMediator )
            {
                auto threadResource = GetClientThreadResource();
                NNT_ASSERT_RESULT_SUCCESS(midc::Client::InvokeUserCommand(clientConnection, clientContext, userContext, 0, &threadResource, clientBuffer, sizeof(clientBuffer), nullptr));
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(midc::Client::InvokeUserCommand(clientConnection, clientContext, userContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
        NNT_ASSERT_RESULT_SUCCESS(midc::Client::Resume(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
        {
            TestUserContext userContext;
            if( useThreadedCommandMediator )
            {
                auto threadResource = GetClientThreadResource();
                NNT_ASSERT_RESULT_SUCCESS(midc::Client::InvokeUserCommand(clientConnection, clientContext, userContext, 0, &threadResource, clientBuffer, sizeof(clientBuffer), nullptr));
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(midc::Client::InvokeUserCommand(clientConnection, clientContext, userContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
            }
        }
        NNT_ASSERT_RESULT_SUCCESS(midc::Client::Terminate(clientConnection, clientContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));

        DestroyThread();
    }
}

TEST_F(MigrationIdcServerClientTest, CommunicationBufferSize)
{
    CommunicationBufferSizeTest(false);
    CommunicationBufferSizeTest(true);
}

// ClientContext 異常系

void TestResultInvalidResponseKind(midc::CommandKind requestKind, midc::CommandKind responseKind)
{
    if( requestKind == responseKind )
    {
        return;
    }
    if( responseKind == midc::CommandKind::Error &&
        (requestKind == midc::CommandKind::User ||
         requestKind == midc::CommandKind::Terminate))
    {
        return;
    }

    ClientContextType clientContext;
    TestUserContext userContext;
    midc::UserCommandMediatorWorkBuffer<ClientContextType::EncryptionPolicy> workBuffer;
    auto consumerHolder = clientContext.AcquireConsumerHolder(&userContext, &workBuffer);
    auto& consumer = consumerHolder.GetProcessorRef();
    clientContext.SetCommandKind(requestKind);

    const auto commandSize = midc::GetResponseCommandSize(responseKind);
    std::unique_ptr<Bit8[]> response(new Bit8[commandSize]);
    response[0] = responseKind;

    switch( requestKind )
    {
    case midc::CommandKind::Initiate0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForInitiate0Request, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::Initiate1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForInitiate1Request, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::Resume0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForResume0Request, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::Resume1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForResume1Request, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::User:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForUserRequest, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::Terminate:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForTerminateRequest, consumer.Consume(response.get(), commandSize));
        break;
    case midc::CommandKind::Error:
        // Error コマンドを Client から送信することはないのでここには到達しない。
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseKindForErrorRequest, consumer.Consume(response.get(), commandSize));
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void TestResultInvalidResponseKind(midc::CommandKind requestKind)
{
    midc::CommandKind responseKinds[] = {
        midc::CommandKind::Initiate0,
        midc::CommandKind::Initiate1,
        midc::CommandKind::Resume0,
        midc::CommandKind::Resume1,
        midc::CommandKind::User,
        midc::CommandKind::Terminate,
        midc::CommandKind::Error,
    };
    for( auto response : responseKinds )
    {
        TestResultInvalidResponseKind(requestKind, response);
    }
}

void TestResultInvalidResponseSize(midc::CommandKind requestKind)
{
    ClientContextType clientContext;
    if( requestKind != midc::CommandKind::Error )
    {
        // クライアントから Error リクエストを送信することはない。
        // Error レスポンスはどのリクエストを送信した状態からでも受け付ける。
        clientContext.SetCommandKind(requestKind);
    }

    // どのコマンドよりも大きい適当なサイズ。
    Bit8 response[256];
    response[0] = requestKind;

    auto consumer = clientContext.GetConsumer();

    switch( requestKind )
    {
    case midc::CommandKind::Initiate0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForInitiate0, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::Initiate1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForInitiate1, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::Resume0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForResume0, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::Resume1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForResume1, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::User:
        // User コマンドのハンドリングは ClientContext ではないのでここには到達しない。
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForUser, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::Terminate:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForTerminate, consumer.Consume(response, sizeof(response)));
        break;
    case midc::CommandKind::Error:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseSizeForError, consumer.Consume(response, sizeof(response)));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void MakeInvalidDataResponse(ClientContextType& clientContext, midc::CommandKind responseKind, void* pOut, size_t outSize)
{
    if( responseKind != midc::CommandKind::Error)
    {
        // 指定したレスポンスを受け入れ可能にしておく。
        clientContext.SetCommandKind(responseKind);
        size_t requestSize;
        Bit8 request[256];
        NNT_ASSERT_RESULT_SUCCESS(clientContext.GetProducer().Produce(&requestSize, request, sizeof(request)));
    }
    std::memset(pOut, 0xAA, midc::GetResponseCommandSize(responseKind));
    reinterpret_cast<Bit8*>(pOut)[0] = responseKind;
}

class ClientContextInvalidResponseTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::InitializeForCrypto();
#endif
    }
    static void TearDownTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::Finalize();
#endif
    }
};

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForInitiate0)
{
    TestResultInvalidResponseKind(midc::CommandKind::Initiate0);
    TestResultInvalidResponseSize(midc::CommandKind::Initiate0);

    {
        ClientContextType clientContext;

        midc::Initiate0Response initiate0Response;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Initiate0, &initiate0Response, sizeof(initiate0Response));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForInitiate0, clientContext.GetConsumer().Consume(&initiate0Response, sizeof(initiate0Response)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForInitiate1)
{
    TestResultInvalidResponseKind(midc::CommandKind::Initiate1);
    TestResultInvalidResponseSize(midc::CommandKind::Initiate1);

    {
        ClientContextType clientContext;
        MakeWaitInitiationClientContext(&clientContext);

        midc::Initiate1Response initiate1Response;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Initiate1, &initiate1Response, sizeof(initiate1Response));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForInitiate1, clientContext.GetConsumer().Consume(&initiate1Response, sizeof(initiate1Response)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForResume0)
{
    TestResultInvalidResponseKind(midc::CommandKind::Resume0);
    TestResultInvalidResponseSize(midc::CommandKind::Resume0);

    {
        ClientContextType clientContext;
        MakeSuspendedClientContext(&clientContext);

        midc::Resume0Response resume0Response;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Resume0, &resume0Response, sizeof(resume0Response));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForResume0, clientContext.GetConsumer().Consume(&resume0Response, sizeof(resume0Response)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForResume1)
{
    TestResultInvalidResponseKind(midc::CommandKind::Resume1);
    TestResultInvalidResponseSize(midc::CommandKind::Resume1);

    {
        ClientContextType clientContext;
        MakeWaitResumeClientContext(&clientContext);

        midc::Resume1Response resume1Response;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Resume1, &resume1Response, sizeof(resume1Response));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForResume1, clientContext.GetConsumer().Consume(&resume1Response, sizeof(resume1Response)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForTerminate)
{
    TestResultInvalidResponseKind(midc::CommandKind::Terminate);
    TestResultInvalidResponseSize(midc::CommandKind::Terminate);

    {
        ClientContextType clientContext;
        MakeSuspendedClientContext(&clientContext);

        midc::TerminateResponse terminateResponse;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Terminate, &terminateResponse, sizeof(terminateResponse));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForTerminate, clientContext.GetConsumer().Consume(&terminateResponse, sizeof(terminateResponse)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForError)
{
    // Error レスポンスは Error リクエストに対応するものではないので TestResultInvalidResponseKind(request) テストは不要。
    TestResultInvalidResponseSize(midc::CommandKind::Error);
    {
        ClientContextType clientContext;
        MakeConnectedClientContext(&clientContext);
        clientContext.SetCommandKind(midc::CommandKind::User);
        midc::ErrorResponse errorResponse;
        MakeInvalidDataResponse(clientContext, midc::CommandKind::Error, &errorResponse, sizeof(errorResponse));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidResponseDataForError, clientContext.GetConsumer().Consume(&errorResponse, sizeof(errorResponse)));
    }
}

TEST_F(ClientContextInvalidResponseTest, InvalidResponseForUser)
{
    TestResultInvalidResponseKind(midc::CommandKind::User);
}

// ServerContext 異常系

template <typename ExpectedResult, int N>
void TestResultInvalidRequestKind(ServerContextType& serverContext, midc::CommandKind(&acceptableKinds)[N])
{
    midc::CommandKind requestKinds[] = {
        midc::CommandKind::Initiate0,
        midc::CommandKind::Initiate1,
        midc::CommandKind::Resume0,
        midc::CommandKind::Resume1,
        midc::CommandKind::User,
        midc::CommandKind::Terminate,
        midc::CommandKind::Error,
    };
    for( auto request : requestKinds )
    {
        if( std::find(std::begin(acceptableKinds), std::end(acceptableKinds), request) == std::end(acceptableKinds) )
        {
            NNT_EXPECT_RESULT_FAILURE(ExpectedResult, serverContext.Consume(&request, sizeof(request)));
        }
    }
}

// 呼び出し前に引数に渡したリクエストを受け入れられる State にしておく必要がある。
void TestResultInvalidRequestSize(ServerContextType& serverContext, midc::CommandKind requestKind)
{
    Bit8 request[256];
    request[0] = requestKind;

    switch( requestKind )
    {
    case midc::CommandKind::Initiate0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForInitiate0, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::Initiate1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForInitiate1, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::Resume0:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForResume0, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::Resume1:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForResume1, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::User:
        // User コマンドのハンドリングは serverContext ではないのでここには到達しない。
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForUser, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::Terminate:
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestSizeForTerminate, serverContext.Consume(request, sizeof(request)));
        break;
    case midc::CommandKind::Error: // Error リクエストは存在しない。
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void MakeInvalidDataRequest(midc::CommandKind requestKind, void* pOut, size_t outSize)
{
    std::memset(pOut, 0xAA, midc::GetRequestCommandSize(requestKind));
    reinterpret_cast<Bit8*>(pOut)[0] = requestKind;
}

class ServerContextInvalidRequestTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::InitializeForCrypto();
#endif
    }
    static void TearDownTestCase()
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        spl::Finalize();
#endif
    }
};

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForStateWaitConnection)
{
    ServerContextType serverContext;

    midc::CommandKind acceptableKinds[] = { midc::CommandKind::Initiate0, midc::CommandKind::Resume0 };
    TestResultInvalidRequestKind<midc::ResultInvalidRequestKindForStateWaitConnection>(serverContext, acceptableKinds);
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForStateWaitInitiation)
{
    ServerContextType serverContext;
    MakeWaitInitiationServerContext(&serverContext);

    midc::CommandKind acceptableKinds[] = { midc::CommandKind::Initiate1 };
    TestResultInvalidRequestKind<midc::ResultInvalidRequestKindForStateWaitInitiation>(serverContext, acceptableKinds);
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForStateWaitResume)
{
    ServerContextType serverContext;
    MakeWaitResumeServerContext(&serverContext);

    midc::CommandKind acceptableKinds[] = { midc::CommandKind::Resume1 };
    TestResultInvalidRequestKind<midc::ResultInvalidRequestKindForStateWaitResume>(serverContext, acceptableKinds);
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForStateConnected)
{
    ServerContextType serverContext;
    MakeConnectedServerContext(&serverContext);

    midc::CommandKind acceptableKinds[] = { midc::CommandKind::User, midc::CommandKind::Terminate };
    TestResultInvalidRequestKind<midc::ResultInvalidRequestKindForStateConnected>(serverContext, acceptableKinds);
}

TEST_F(ServerContextInvalidRequestTest, InvalidReqeustForInitiate0)
{
    {
        ServerContextType serverContext;
        TestResultInvalidRequestSize(serverContext, midc::CommandKind::Initiate0);
    }

    {
        ServerContextType serverContext;
        midc::Initiate0Request initiate0Request;
        MakeInvalidDataRequest(midc::CommandKind::Initiate0, &initiate0Request, sizeof(initiate0Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestDataForInitiate0, serverContext.Consume(&initiate0Request, sizeof(initiate0Request)));
    }
    {
        ServerContextType serverContext;
        MakeSuspendedServerContext(&serverContext);

        // データ検証よりも鍵交換状態が先に判定されるのでリクエストの中身は適当で問題ない。
        midc::Initiate0Request initiate0Request;
        initiate0Request.commandId = midc::CommandKind::Initiate0;
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestKeyAlreadyExchangedForInitiate0, serverContext.Consume(&initiate0Request, sizeof(initiate0Request)));
    }
    {
        // クライアント検証用チャレンジ保存済み：未テスト
        // クライアント検証用チャレンジは Initiate0/Resume0リクエストを受け付けると保存される。
        // 一方で Initiate0/Resume0 リクエストを受け付けると State が変化して Initiat0 リクエストを受け付けられなくなるので、
        // 実際のシーケンス上このエラーは発生しない想定（ResultInvalidRequestKindForState...が先に返る）。
    }
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForInitiate1)
{
    {
        ServerContextType serverContext;
        MakeWaitInitiationServerContext(&serverContext);

        TestResultInvalidRequestSize(serverContext, midc::CommandKind::Initiate1);
    }
    {
        ServerContextType serverContext;
        MakeWaitInitiationServerContext(&serverContext);

        midc::Resume1Request initiate1Request;
        MakeInvalidDataRequest(midc::CommandKind::Initiate1, &initiate1Request, sizeof(initiate1Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestDataForInitiate1, serverContext.Consume(&initiate1Request, sizeof(initiate1Request)));
    }
    {
        ServerContextType serverContext;
        ClientContextType clientContext;

        clientContext.SetCommandKind(midc::CommandKind::Initiate0);
        midc::Initiate0Request initiate0Request;
        size_t requestSize;
        NNT_ASSERT_RESULT_SUCCESS(clientContext.GetProducer().Produce(&requestSize, &initiate0Request, sizeof(initiate0Request)));
        ASSERT_EQ(requestSize, sizeof(initiate0Request));
        NNT_ASSERT_RESULT_SUCCESS(serverContext.Consume(&initiate0Request, sizeof(initiate0Request)));

        // サーバー検証用チャレンジは Initiate0 レスポンス生成時に同時に保存されるので、
        // Initiate0 リクエスト受け付け後、Initiate0 レスポンスを生成せずに Initiate1 リクエストを受け付ける。
        // 実際のシーケンス上このエラーは発生しない想定。
        midc::Resume1Request initiate1Request;
        MakeInvalidDataRequest(midc::CommandKind::Initiate1, &initiate1Request, sizeof(initiate1Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestServerChallengeNotStoredForInitiate1, serverContext.Consume(&initiate1Request, sizeof(initiate1Request)));
    }
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForResume0)
{
    {
        ServerContextType serverContext;
        MakeSuspendedServerContext(&serverContext);

        TestResultInvalidRequestSize(serverContext, midc::CommandKind::Resume0);
    }
    {
        ServerContextType serverContext;
        MakeSuspendedServerContext(&serverContext);

        midc::Resume0Request resume0Request;
        MakeInvalidDataRequest(midc::CommandKind::Resume0, &resume0Request, sizeof(resume0Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestDataForResume0, serverContext.Consume(&resume0Request, sizeof(resume0Request)));
    }
    {
        ServerContextType serverContext;

        // データ検証よりも鍵交換状態が先に判定されるのでリクエストの中身は適当で問題ない。
        midc::Resume0Request resume0Request;
        resume0Request.commandId = midc::CommandKind::Resume0;
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestKeyNotExchangedForResume0, serverContext.Consume(&resume0Request, sizeof(resume0Request)));
    }
    {
        // クライアント検証用チャレンジ保存済み：未テスト
        // クライアント検証用チャレンジは Initiate0/Resume0リクエストを受け付けると保存される。
        // 一方で Initiate0/Resume0 リクエストを受け付けると State が変化して Initiat0/Resume0 リクエストを受け付けられなくなるので、
        // 実際のシーケンス上このエラーは発生しない想定（ResultInvalidRequestKindForState...が先に返る）。
    }
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForResume1)
{
    {
        ServerContextType serverContext;
        MakeWaitResumeServerContext(&serverContext);

        TestResultInvalidRequestSize(serverContext, midc::CommandKind::Resume1);
    }
    {
        ServerContextType serverContext;
        MakeWaitResumeServerContext(&serverContext);

        midc::Resume1Request resume1Request;
        MakeInvalidDataRequest(midc::CommandKind::Resume1, &resume1Request, sizeof(resume1Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestDataForResume1, serverContext.Consume(&resume1Request, sizeof(resume1Request)));
    }
    {
        ServerContextType serverContext;
        ClientContextType clientContext;
        MakeSuspendedServerClient(&serverContext, &clientContext);

        clientContext.SetCommandKind(midc::CommandKind::Resume0);
        midc::Resume0Request resume0Request;
        size_t requestSize;
        NNT_ASSERT_RESULT_SUCCESS(clientContext.GetProducer().Produce(&requestSize, &resume0Request, sizeof(resume0Request)));
        ASSERT_EQ(requestSize, sizeof(resume0Request));
        NNT_ASSERT_RESULT_SUCCESS(serverContext.Consume(&resume0Request, sizeof(resume0Request)));

        // サーバー検証用チャレンジは Resume0 レスポンス生成時に同時に保存されるので、
        // Resume0 リクエスト受け付け後、Resume0 レスポンスを生成せずに Resume1 リクエストを受け付ける。
        // 実際のシーケンス上このエラーは発生しない想定。
        midc::Resume1Request resume1Request;
        MakeInvalidDataRequest(midc::CommandKind::Resume1, &resume1Request, sizeof(resume1Request));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestServerChallengeNotStoredForResume1, serverContext.Consume(&resume1Request, sizeof(resume1Request)));
    }
}

TEST_F(ServerContextInvalidRequestTest, InvalidRequestForTerminate)
{
    {
        ServerContextType serverContext;
        MakeConnectedServerContext(&serverContext);

        TestResultInvalidRequestSize(serverContext, midc::CommandKind::Terminate);
    }
    {
        ServerContextType serverContext;
        MakeConnectedServerContext(&serverContext);

        midc::TerminateRequest terminateRequest;
        MakeInvalidDataRequest(midc::CommandKind::Terminate, &terminateRequest, sizeof(terminateRequest));
        NNT_EXPECT_RESULT_FAILURE(midc::ResultInvalidRequestDataForTerminate, serverContext.Consume(&terminateRequest, sizeof(terminateRequest)));
    }
    {
        // 鍵交換済みでない：未テスト
        // 鍵交換済みでない限り Terminate リクエストを受け付けられないので、
        // 実際のシーケンス上このエラーは発生しない想定（ResultInvalidRequestKindForState...が先に返る）。
    }
}

template <typename TestClass>
void LowSpeedTest(TestClass* pTestClass, bool useThreaded)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    struct ThreadArg
    {
        typename TestClass::ConnectionType* pConnection;
        ClientContextType* pContext;
        bool useThreaded;
    } threadArg{ &clientConnection, &clientContext, useThreaded };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ThreadArg*>(arg);

        auto start = os::GetSystemTick();

        // フル稼働しても到達できない速度を設定。
        midc::detail::TransferSpeedMonitor clientSpeedMonitor(std::numeric_limits<size_t>::max(), nn::TimeSpan::FromSeconds(5));
        TestUserContext clientUserContext(TooLargeCommandSize, TooLargeCommandSize);
        static Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>)];
        if( pArg->useThreaded )
        {
            auto threadResource = GetClientThreadResource();
            NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 0, &threadResource, clientBuffer, sizeof(clientBuffer), nullptr, &clientSpeedMonitor));
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 0, clientBuffer, sizeof(clientBuffer), nullptr, &clientSpeedMonitor));
        }

        NN_LOG("Client timeout took %lld ms\n", (os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds());

    }, &threadArg);

    auto start = os::GetSystemTick();

    // フル稼働しても到達できない速度を設定。
    midc::detail::TransferSpeedMonitor serverSpeedMonitor(std::numeric_limits<size_t>::max(), nn::TimeSpan::FromSeconds(5));
    TestUserContext serverUserContext(TooLargeCommandSize, TooLargeCommandSize);
    static Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];
    if( useThreaded )
    {
        auto threadResource = GetServerThreadResource();
        NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr, &serverSpeedMonitor));

    }
    else
    {
        NNT_EXPECT_RESULT_FAILURE(midc::ResultTimeout, midc::Server::WaitUserCommand(serverConnection, serverContext, serverUserContext, 0, serverBuffer, sizeof(serverBuffer), nullptr, &serverSpeedMonitor));
    }

    NN_LOG("Server timeout took %lld ms\n", (os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds());

    DestroyThread();
}

/* SIGLO-79528
TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, LowSpeedTest)
{
    LowSpeedTest(this, false);
}
*/

TEST_F(MigrationIdcSocketConnectionServerClientTest, LowSpeedTest)
{
    LowSpeedTest(this, false);
}

/* SIGLO-79528
TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, LowSpeedTest_Threaded)
{
    LowSpeedTest(this, true);
}
*/

TEST_F(MigrationIdcSocketConnectionServerClientTest, LowSpeedTest_Threaded)
{
    LowSpeedTest(this, true);
}

template <typename TestClass>
void ErrorResponseUserCommandTest(TestClass* pTestClass, bool useThreaded)
{
    typename TestClass::ConnectionType serverConnection, clientConnection;
    pTestClass->GetConnection(&serverConnection, &clientConnection);
    ServerContextType serverContext;
    ClientContextType clientContext;

    MakeConnectedServerClient(&serverContext, &clientContext, serverConnection, clientConnection);

    struct ThreadArg
    {
        typename TestClass::ConnectionType* pConnection;
        ClientContextType* pContext;
        bool useThreaded;
    } threadArg{ &clientConnection, &clientContext, useThreaded };

    RunOnThread([](void* arg) {
        auto pArg = reinterpret_cast<ThreadArg*>(arg);

        auto start = os::GetSystemTick();

        TestUserContext clientUserContext(1, 32 * 1024);
        static Bit8 clientBuffer[sizeof(midc::Client::InvokeUserCommandBufferForThreaded<ClientContextType>)];
        if( pArg->useThreaded )
        {
            auto threadResource = GetClientThreadResource();
            NNT_EXPECT_RESULT_FAILURE(midc::ResultServiceStopped, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 0, &threadResource, clientBuffer, sizeof(clientBuffer), nullptr));
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(midc::ResultServiceStopped, midc::Client::InvokeUserCommand(*pArg->pConnection, *pArg->pContext, clientUserContext, 0, clientBuffer, sizeof(clientBuffer), nullptr));
        }

        NN_LOG("Client operation took %lld ms\n", (os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds());

    }, &threadArg);

    auto start = os::GetSystemTick();

    static Bit8 serverBuffer[sizeof(midc::Server::WaitUserCommandBufferForThreaded<ServerContextType>)];
    if( useThreaded )
    {
        auto threadResource = GetServerThreadResource();
        serverContext.SetErrorResponse(midc::ErrorKind::ServiceStopped);
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitUserCommandAndRespondWithError(serverConnection, serverContext, 0, &threadResource, serverBuffer, sizeof(serverBuffer), nullptr));
    }
    else
    {
        serverContext.SetErrorResponse(midc::ErrorKind::ServiceStopped);
        NNT_EXPECT_RESULT_SUCCESS(midc::Server::WaitUserCommandAndRespondWithError(serverConnection, serverContext, 0, serverBuffer, sizeof(serverBuffer), nullptr));
    }

    NN_LOG("Server operation took %lld ms\n", (os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds());

    DestroyThread();
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ErrorResponse)
{
    ErrorResponseUserCommandTest(this, false);
}

TEST_F(MigrationIdcSharedBufferConnectionServerClientTest, ErrorResponse_Threaded)
{
    ErrorResponseUserCommandTest(this, true);
}
