﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/migration_SharedBufferConnection.h>
#include <nn/util/util_StringUtil.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include "testMigrationIdc_Common.h"

using namespace nn;

const size_t BufferSize = 8 * 1024 * 1024;
Bit8 g_SendBuffer[BufferSize];
Bit8 g_RecvBuffer[BufferSize];

template <typename C>
void FillSendBuffer(const C& connection)
{
    size_t sentCount = 0;
    TestCancellable cancellable;
    connection.Send(&sentCount, g_SendBuffer, BufferSize, 1, &cancellable);
    // Windows + Socket は 2回 Send が必要。他でもタイムアウトするだけで害はないので常に 2回送信。
    connection.Send(&sentCount, g_SendBuffer, BufferSize, 1, &cancellable);
}

template <typename C>
void CheckSendResultSuccess(const C& connection, TestCancellable* pCancellable, size_t sendSize = 32, bool doSentSizeCheck = false, size_t expectedSentSize = 32)
{
    NN_ABORT_UNLESS_LESS_EQUAL(sendSize, BufferSize);
    NN_LOG("CheckSendResultSuccess\n");
    size_t actualSentCount = 0;
    auto result = connection.Send(&actualSentCount, g_SendBuffer, sendSize, 3, pCancellable);
    NNT_ASSERT_RESULT_SUCCESS(result);
    if( doSentSizeCheck )
    {
        ASSERT_EQ(expectedSentSize, actualSentCount);
    }
    else
    {
        ASSERT_GT(actualSentCount, 0u);
    }
}

template <typename C, typename R>
void CheckSendResultFailure(const C& connection, TestCancellable* pCancellable, size_t sendSize = 32, migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr)
{
    NN_ABORT_UNLESS_LESS_EQUAL(sendSize, BufferSize);
    NN_LOG("CheckSendResultFailure\n");
    size_t actualSentCount = 0;
    auto result = connection.Send(&actualSentCount, g_SendBuffer, sendSize, 3, pCancellable, pSpeedMonitor);
    NNT_ASSERT_RESULT_FAILURE(R, result);
    ASSERT_EQ(0u, actualSentCount);
}

template <typename C>
void CheckReceiveResultSuccess(const C& connection, TestCancellable* pCancellable, size_t receiveSize = 32, bool doReceivedSizeCheck = false, size_t expectedReceivedSize = 32)
{
    NN_ABORT_UNLESS_LESS_EQUAL(receiveSize, BufferSize);
    NN_LOG("CheckReceiveResultSuccess\n");
    size_t actualRecvCount = 0;
    auto result = connection.Receive(&actualRecvCount, g_RecvBuffer, receiveSize, 3, pCancellable);
    NNT_ASSERT_RESULT_SUCCESS(result);
    if( doReceivedSizeCheck )
    {
        ASSERT_EQ(expectedReceivedSize, actualRecvCount);
    }
    else
    {
        ASSERT_GT(actualRecvCount, 0u);
    }
}

template <typename C, typename R>
void CheckReceiveResultFailure(const C& connection, TestCancellable* pCancellable, size_t recvSize = 32, migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr)
{
    NN_ABORT_UNLESS_LESS_EQUAL(recvSize, BufferSize);
    NN_LOG("CheckReceiveResultFailure\n");
    size_t actualRecvCount = 0;
    auto result = connection.Receive(&actualRecvCount, g_RecvBuffer, recvSize, 3, pCancellable, pSpeedMonitor);
    NNT_ASSERT_RESULT_FAILURE(R, result);
}

template <typename C>
void TestBasicSendReceive(const C& c0, const C& c1)
{
    RunOnThreadA([](void* arg)
        {
            auto pC = reinterpret_cast<const C*>(arg);
            TestCancellable cancellable;
            Bit8 buffer[128] = {};
            size_t recvCount = 0;
            NNT_EXPECT_RESULT_SUCCESS(pC->Receive(&recvCount, buffer, sizeof(buffer), 0, &cancellable));
            NN_LOG("server received : %s (%d)\n", buffer, recvCount);
            EXPECT_EQ(0, std::strncmp("message", reinterpret_cast<char*>(buffer), sizeof(buffer)));
            // Echo.
            size_t sentCount = 0;
            NNT_EXPECT_RESULT_SUCCESS(pC->Send(&sentCount, buffer, recvCount, 0, &cancellable));
            EXPECT_EQ(recvCount, sentCount);
        },
        const_cast<C*>(&c0));

    RunOnThreadB([](void* arg)
        {
            auto pC = reinterpret_cast<const C*>(arg);
            TestCancellable cancellable;
            Bit8 sendBuffer[128] = {};
            size_t sentCount = 0;
            std::strncpy(reinterpret_cast<char*>(sendBuffer), "message", sizeof(sendBuffer));
            NNT_EXPECT_RESULT_SUCCESS(pC->Send(&sentCount, sendBuffer, util::Strnlen(sendBuffer, sizeof(sendBuffer)) + 1, 0, &cancellable)); // null終端分 + 1
            NN_LOG("client send : %s (%d)\n", sendBuffer, sentCount);

            Bit8 recvBuffer[128] = {};
            size_t recvCount = 0;
            NNT_EXPECT_RESULT_SUCCESS(pC->Receive(&recvCount, recvBuffer, sizeof(recvBuffer), 0, &cancellable));
            NN_LOG("client received : %s (%d)\n", recvBuffer, recvCount);
            EXPECT_EQ(sentCount, recvCount);
            EXPECT_EQ(0, std::strncmp("message", reinterpret_cast<char*>(recvBuffer), sizeof(recvBuffer)));
        },
        const_cast<C*>(&c1));

    DestroyThreadA();
    DestroyThreadB();
}

template <typename C>
void TestBasicSendReceiveLarge(const C& c0, const C& c1)
{
    TestCancellable cancellable;
    for( size_t i = 0; i < BufferSize; i++ )
    {
        g_SendBuffer[i] = static_cast<Bit8>(i % 256);
    }
    for( size_t totalSentCount = 0; totalSentCount < BufferSize;)
    {
        size_t sentCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(c0.Send(&sentCount, g_SendBuffer + totalSentCount, BufferSize - totalSentCount, 0, &cancellable));
        NN_LOG("sent %d bytes (total %d bytes)\n", sentCount, sentCount + totalSentCount);

        for( size_t totalRecvCount = 0; totalRecvCount < sentCount;)
        {
            size_t recvCount = 0;
            NNT_EXPECT_RESULT_SUCCESS(c1.Receive(&recvCount, g_RecvBuffer, BufferSize, 0, &cancellable));
            NN_LOG("received %7d bytes (total %7d bytes)\n", recvCount, recvCount + totalRecvCount);
            ASSERT_EQ(0, std::memcmp(&g_SendBuffer[totalSentCount + totalRecvCount], &g_RecvBuffer, recvCount));
            totalRecvCount += recvCount;
        }
        totalSentCount += sentCount;
    }
}

/// 送信系のテスト

void CheckSendPeerClose(const migration::idc::SocketConnection& connection)
{
    os::TimerEvent timer(os::EventClearMode_ManualClear);
    timer.StartOneShot(nn::TimeSpan::FromSeconds(10));

    while( !timer.TryWait() )
    {
        size_t sendCount = 0;
        auto result = connection.Send(&sendCount, g_SendBuffer, 1024, 1, nullptr);
        if( nn::migration::idc::ResultPeerClosed::Includes(result) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(result);
        os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    ADD_FAILURE();
}

void CheckSendPeerClose(const migration::idc::SharedBufferConnection& connection)
{
    CheckSendResultFailure<migration::idc::SharedBufferConnection, migration::idc::ResultPeerClosed>(connection, nullptr, 1024);
}

template <typename C>
void TestSendPeerClose(const C& c0, C&& c1)
{
    c1.Close();
    CheckSendPeerClose(c0);
}

template <typename C>
void TestSendCancel(const C& c)
{
    TestCancellable cancellable;
    cancellable.Cancel();

    CheckSendResultFailure<C, migration::idc::ResultCanceled>(c, &cancellable);
}

template <typename C>
void TestSendCancelWhileSendBufferFull(const C& c)
{
    TestCancellable cancellable;

    FillSendBuffer(c);

    struct ConnectionData
    {
        const C*            pConnection;
        TestCancellable*    pCancellable;
    }
    data{ &c, &cancellable };

    RunOnThreadA(
        [](void* arg)
        {
            auto pConnectionData = reinterpret_cast<ConnectionData*>(arg);
            CheckSendResultFailure<C, migration::idc::ResultCanceled>(*(pConnectionData->pConnection), pConnectionData->pCancellable);
        }, &data);

    os::SleepThread(nn::TimeSpan::FromSeconds(1));
    cancellable.Cancel();
    DestroyThreadA();
}

template <typename C>
void TestSendCancelAndClose(const C& c0, C&& c1)
{
    TestCancellable cancellable;
    cancellable.Cancel();
    c1.Close();

    CheckSendResultFailure<C, migration::idc::ResultCanceled>(c0, &cancellable);
}

template <typename C>
void TestSendTimeout(const C& c)
{
    TestCancellable cancellable;
    FillSendBuffer(c);
    CheckSendResultFailure<C, migration::idc::ResultTimeout>(c, &cancellable);
}

template <typename C>
void TestSendLowSpeedTimeout(C& c)
{
    FillSendBuffer(c);
    // CheckSendResultFailure は 3 秒でタイムアウトする。LowSpeedLimit の期間を 1秒にして、3秒以内にタイムアウトしていることを確認。
    migration::idc::detail::TransferSpeedMonitor speedMonitor(std::numeric_limits<size_t>::max(), nn::TimeSpan::FromSeconds(1));
    auto start = os::GetSystemTick();
    CheckSendResultFailure<C, migration::idc::ResultTimeout>(c, nullptr, BufferSize, &speedMonitor);
    auto elapsed = os::GetSystemTick() - start;
    EXPECT_LE(elapsed.ToTimeSpan().GetMilliSeconds(), 3000);
}

/// 受信系のテスト

template <typename C>
void TestReceivePeerClose(const C& c0, C&& c1)
{
    TestCancellable cancellable;
    c1.Close();

    CheckReceiveResultFailure<C, migration::idc::ResultPeerClosed>(c0, &cancellable);
}

template <typename C>
void TestReceivePeerCloseWhileReceiveBufferEmpty(const C& c0, C&& c1)
{
    TestCancellable cancellable;

    struct ConnectionData
    {
        const C*         pConnection;
        TestCancellable* pCancellable;
    }
    data{ &c0, &cancellable };

    RunOnThreadA(
        [](void* arg)
        {
            auto pConnectionData = reinterpret_cast<ConnectionData*>(arg);
            CheckReceiveResultFailure<C, migration::idc::ResultPeerClosed>(*(pConnectionData->pConnection), pConnectionData->pCancellable);
        }, &data);

    os::SleepThread(nn::TimeSpan::FromSeconds(1));
    c1.Close();
    DestroyThreadA();
}

template <typename C>
void TestReceivePeerCloseWithReceiveBuffer(const C& c0, C&& c1)
{
    TestCancellable cancellable;

    CheckSendResultSuccess<>(c1, &cancellable, 32, true, 32);
    c1.Close();
    // 送信されている分を受け取るまでは成功する。
    CheckReceiveResultSuccess<>(c0, &cancellable, 16, true, 16);
    CheckReceiveResultSuccess<>(c0, &cancellable, 16, true, 16);
    CheckReceiveResultFailure<C, migration::idc::ResultPeerClosed>(c0, &cancellable);
}

template <typename C>
void TestReceiveCancel(const C& c)
{
    TestCancellable cancellable;
    cancellable.Cancel();

    CheckReceiveResultFailure<C, migration::idc::ResultCanceled>(c, &cancellable);
}

template <typename C>
void TestReceiveCancelWhileReceiveBufferEmpty(const C& c)
{
    TestCancellable cancellable;

    struct ConnectionData
    {
        const C*            pConnection;
        TestCancellable*    pCancellable;
    }
    data{ &c, &cancellable };

    RunOnThreadA(
        [](void* arg)
        {
            auto pConnectionData = reinterpret_cast<ConnectionData*>(arg);
            CheckReceiveResultFailure<C, migration::idc::ResultCanceled>(*(pConnectionData->pConnection), pConnectionData->pCancellable);
        }, &data);

    os::SleepThread(nn::TimeSpan::FromSeconds(1));
    cancellable.Cancel();

    DestroyThreadA();
}

template <typename C>
void TestReceiveCancelWithReceiveBuffer(const C& c0, const C& c1)
{
    TestCancellable cancellable;

    CheckSendResultSuccess(c1, &cancellable);

    cancellable.Cancel();
    CheckReceiveResultFailure<C, migration::idc::ResultCanceled>(c0, &cancellable);
}

template <typename C>
void TestReceiveCancelAndClose(const C& c0, C&& c1)
{
    c1.Close();
    TestCancellable cancellable;
    cancellable.Cancel();
    CheckReceiveResultFailure<C, migration::idc::ResultCanceled>(c0, &cancellable);
}

template <typename C>
void TestReceiveTimeout(const C& c)
{
    TestCancellable cancellable;
    CheckReceiveResultFailure<C, migration::idc::ResultTimeout>(c, &cancellable);
}

template <typename C>
void TestReceiveLowSpeedTimeout(C& c)
{
    // CheckSendResultFailure は 3 秒でタイムアウトする。LowSpeedLimit の期間を 1秒にして、3秒以内にタイムアウトしていることを確認。
    migration::idc::detail::TransferSpeedMonitor speedMonitor(std::numeric_limits<size_t>::max(), nn::TimeSpan::FromSeconds(1));
    auto start = os::GetSystemTick();
    CheckReceiveResultFailure<C, migration::idc::ResultTimeout>(c, nullptr, BufferSize, &speedMonitor);
    auto elapsed = os::GetSystemTick() - start;
    EXPECT_LE(elapsed.ToTimeSpan().GetMilliSeconds(), 3000);
}
