﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nntMigration_Http.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Version.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nnt { namespace migration {
namespace {
#define USERMIGRATION_SDK_VERSION_STRING(major, minor, micro, relstep) \
    # major "." # minor "." # micro "." # relstep

const char* GetUserAgent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::os::MutexType, g_Lock, = NN_OS_MUTEX_INITIALIZER(false));
    nn::os::LockMutex(&g_Lock);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&g_Lock);
    };

#define USERMIGRATION_AGENT     "libcurl"
#define USERMIGRATION_MODULE    "UserMigration"
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define USERMIGRATION_PLATFORM  "cac82318-df2c-4e18-8502-3fd69b37e060"
#else
#define USERMIGRATION_PLATFORM  "f1d2eb8d-744c-4352-a190-8749c9f391db"
#endif

    typedef char UserAgentStringType[256];
    NN_FUNCTION_LOCAL_STATIC(UserAgentStringType, g_UserAgent, = {});

    NN_FUNCTION_LOCAL_STATIC(bool, g_Initialized, = false);
    if (!g_Initialized)
    {

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        auto l = nn::util::SNPrintf(
            g_UserAgent, sizeof(g_UserAgent),
            "%s (%s; %s; SDK %d.%d.%d.%d; Add-on %d.%d.%d.%d)",
            USERMIGRATION_AGENT, USERMIGRATION_MODULE, USERMIGRATION_PLATFORM,
            NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP,
            NN_NX_ADDON_VERSION_MAJOR, NN_NX_ADDON_VERSION_MINOR, NN_NX_ADDON_VERSION_MICRO, NN_NX_ADDON_VERSION_RELSTEP);
#else
        auto l = nn::util::SNPrintf(
            g_UserAgent, sizeof(g_UserAgent),
            "%s (%s; %s; SDK %d.%d.%d.%d)",
            USERMIGRATION_AGENT, USERMIGRATION_MODULE, USERMIGRATION_PLATFORM,
            NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
#endif
        NN_ABORT_UNLESS(l < sizeof(g_UserAgent), "[Migration] Error: Insufficient buffer for user agent: required %d bytes\n", l + 1);

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        NN_LOG("[Migration] User agent is initialized as \"%s\"\n", g_UserAgent);
#endif
        g_Initialized = true;
    }
    return g_UserAgent;
}

inline bool IsReservedCharacter(char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    static const char Reserved[] = {
        0x20, // スペース : 未定義
        0x21, // !
        0x22, // " : 未定義
        0x23, // #
        0x24, // $
        0x25, // % : 未定義
        0x26, // &
        0x27, // '
        0x28, // (
        0x29, // )
        0x2A, // *
        0x2B, // +
        0x2C, // ,
        // 0x2D, // -
        // 0x2E, // .
        0x2F, // /
        0x3A, // :
        0x3B, // ;
        0x3C, // < : 未定義
        0x3D, // =
        0x3E, // > : 未定義
        0x3F, // ?
        0x40, // @
        0x5B, // [
        0x5C, // \ : 未定義
        0x5D, // ]
        0x5E, // ^ : 未定義
        // 0x5F, // _
        0x60, // ` : 未定義
        0x7B, // { : 未定義
        0x7C, // | : 未定義
        0x7D, // } : 未定義
        // 0x7E, // ~
    };
    size_t i = 0u;
    while (i < sizeof(Reserved) && Reserved[i] != c)
    {
        ++ i;
    }
    return i != sizeof(Reserved);
}

inline bool IsUnreservedCharacter(char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    return false
        || ('A' <= c && c <= 'Z')
        || ('a' <= c && c <= 'z')
        || ('0' <= c && c <= '9')
        || c == '-'
        || c == '_'
        || c == '.'
        || c == '~';
}

inline size_t PutCharacterWithUrlEncoding(char* str, size_t strSize, char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    NN_SDK_ASSERT(strSize >= 3);
    NN_UNUSED(strSize);
    auto p = str;
    if (c == ' ')
    {
        // +
        *(p ++) = '+';
    }
    else if (IsUnreservedCharacter(c))
    {
        // c
        *(p ++) = c;
    }
    else
    {
        NN_ABORT_UNLESS(IsReservedCharacter(c));
        auto u = (c >> 4);
        NN_ABORT_UNLESS(u <= 7);
        auto l = (c & 0x0F);
        NN_SDK_ASSERT(l <= 15);

        // %xx
        *(p ++) = '%';
        *(p ++) = static_cast<char>('0' + u);
        *(p ++) = static_cast<char>((l >= 10 ? 'A' + l - 10 : '0' + l));
    }
    return static_cast<int>(p - str);
}
} // ~namespace nnt::migration::<anonymous>


int UrlEncode(char *dst, size_t dstSize, const char* src, size_t srcSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(strnlen(src, srcSize) < srcSize); // 終端
    NN_ABORT_UNLESS(dstSize >= 3 * (srcSize - 1) + 1); // 上限
    size_t offset = 0;
    for (size_t i = 0u; i < srcSize; ++ i)
    {
        auto c = src[i];
        if (c == '\0')
        {
            *(dst + offset) = '\0';
            return static_cast<int>(offset);
        }

        auto l = PutCharacterWithUrlEncoding(dst + offset, dstSize - offset, c);
        NN_SDK_ASSERT(l < dstSize - offset);
        offset += l;
    }
    NN_ABORT("[Migration] ABORT: Unreachable\n");
}

const SimpleDownloader::SslCtxFunctionContext SimpleDownloader::SslCtxFunctionContext::Default = {false, nn::ResultSuccess(), nullptr};

SimpleDownloader::SimpleDownloader(CURL* curlHandle) NN_NOEXCEPT
    : m_CurlHandle(curlHandle)
{
    Reset();
}

size_t SimpleDownloader::HeaderFunction(char*, size_t unitBytes, size_t count, void*) NN_NOEXCEPT
{
    return unitBytes * count;
}
size_t SimpleDownloader::WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    const auto InputBytes = unitBytes * count;
    auto& obj = *reinterpret_cast<SimpleDownloader*>(pContext);
    if (true
        && !obj.m_BufferInfo.failure
        && InputBytes <= obj.m_BufferInfo.total - obj.m_BufferInfo.used)
    {
        auto address = reinterpret_cast<uintptr_t>(obj.m_BufferInfo.address) + obj.m_BufferInfo.used;
        std::memcpy(reinterpret_cast<void*>(address), ptr, InputBytes);
        obj.m_BufferInfo.used += InputBytes;
        return InputBytes;
    }
    obj.m_BufferInfo.failure = true;
    return 0;
}
int SimpleDownloader::XferInfoFunction(void*, curl_off_t, curl_off_t, curl_off_t, curl_off_t) NN_NOEXCEPT
{
    return 0;
}
size_t SimpleDownloader::DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFunctionContext);
    pFunctionContext->isCalled = true;
    if (!pFunctionContext->result.IsSuccess())
    {
        return 1;
    }

    auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
    auto r = context.Create(nn::ssl::Context::SslVersion_Auto);
    if (!r.IsSuccess())
    {
        pFunctionContext->result = r;
        return 1;
    }
    return 0;
}

void SimpleDownloader::Reset() NN_NOEXCEPT
{
    curl_easy_reset(m_CurlHandle);
    m_Initialized = false;
    m_SslCtxFunctionContext = SslCtxFunctionContext::Default;
    m_BufferInfo.address = nullptr;
    m_BufferInfo.total = 0u;
    m_BufferInfo.used = 0u;
    m_BufferInfo.failure = false;
}

void SimpleDownloader::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(buffer != nullptr);
    NN_SDK_ASSERT(bufferSize > 0);
    NN_ABORT_UNLESS(SetDefault() == CURLE_OK);
    m_BufferInfo.address = buffer;
    m_BufferInfo.total = bufferSize;
    m_BufferInfo.used = 0;
    m_BufferInfo.failure = false;
}

#define USERMIGRATION_SET_CURLOPT(handle, opt, ...) \
    do \
    { \
        CURLcode _r = curl_easy_setopt(handle, opt, __VA_ARGS__); \
        if (_r != CURLE_OK) \
        { \
            NN_LOG("[Migration] curl_easy_setopt(" #opt ") failed (%d)\n", _r); \
            return _r;\
        } \
    } while (NN_STATIC_CONDITION(false))

CURLcode SimpleDownloader::SetDefault() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_Initialized);
    Reset();

    /* basic */
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_NOPROGRESS,        0L);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_VERBOSE,           0L);

    /* advance*/
    const long RedirectCountMax =  0L; // リダイレクトは 0 回まで
    const long LowSpeedLimit    =  1L; // 転送ビットレートは 1bps を下限とする
    const long LowSpeedTime     = 30L; // LowSpeedLimit が 30 秒つづくと通信を中断する
    const long ConnectTimeout   = 30L; // サーバとの接続(CONNECT)のタイムアウト (30秒)
    const long TimeoutSeconds   = 60L; // 通信全体でのタイムアウト (1分)
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_FOLLOWLOCATION,    1L);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_MAXREDIRS,         RedirectCountMax);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_LOW_SPEED_LIMIT,   LowSpeedLimit);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_LOW_SPEED_TIME,    LowSpeedTime);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_CONNECTTIMEOUT,    ConnectTimeout);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_TIMEOUT,           TimeoutSeconds);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYPEER,    0L); // server validation is disabled for testing
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYHOST,    0L); // host name validation is disabled for testing
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYPEER,    1L);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYHOST,    2L);
#else
#   error "Unsupported OS specified"
#endif
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYDATE,    0L); // 時刻検証無効

    /* callback */
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERFUNCTION,    HeaderFunction);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERDATA,        nullptr);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEFUNCTION,     WriteFunction);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEDATA,         this);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_XFERINFOFUNCTION,  XferInfoFunction);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_XFERINFODATA,      this);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_CTX_FUNCTION,  DefaultSslCtxFunction);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_CTX_DATA,      &m_SslCtxFunctionContext);

    /* user agent */
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_USERAGENT,         GetUserAgent());

    m_Initialized = true;
    return CURLE_OK;
}

void SimpleDownloader::Setup(const char* url, const char* postData, bool verbose) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(SetUrl(url) == CURLE_OK);
    NN_ABORT_UNLESS(SetHttpPost(postData) == CURLE_OK);
    if (verbose)
    {
        NN_ABORT_UNLESS(SetDebugMode() == CURLE_OK);
    }
}
CURLcode SimpleDownloader::SetUrl(const char* url) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_URL, url);
    return CURLE_OK;
}
CURLcode SimpleDownloader::SetHttpPost(const char* postData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_POSTFIELDS, postData);
    return CURLE_OK;
}
CURLcode SimpleDownloader::SetDebugMode() NN_NOEXCEPT
{
    USERMIGRATION_SET_CURLOPT(m_CurlHandle, CURLOPT_VERBOSE, 1L);
    return CURLE_OK;
}

#undef USERMIGRATION_SET_CURLOPT

CURLcode SimpleDownloader::Invoke(size_t* pOutSizeActual) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_BufferInfo.address != nullptr);

    auto code = curl_easy_perform(m_CurlHandle);
    if (code != CURLE_OK)
    {
        if (code == CURLE_WRITE_ERROR)
        {
            NN_ABORT_UNLESS(!m_BufferInfo.failure, "[Migration] Insufficient buffer (internal)");
        }
        return code;
    }
    *pOutSizeActual = m_BufferInfo.used;
    return CURLE_OK;
}

long SimpleDownloader::GetHttpCode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    long httpCode = 0;
    auto curle = curl_easy_getinfo(m_CurlHandle, CURLINFO_RESPONSE_CODE, &httpCode);
    return curle == CURLE_OK ? httpCode : -1;
}

}} // ~namespace nnt::migration
