﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/idc/migration_SharedBufferConnection.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/detail/migration_ConnectionPolicy.h>
#include <nn/migration/detail/migration_Mutex.h>
#include <nn/migration/detail/migration_Settings.h>

namespace nnt { namespace migration {

template <int64_t Mbps>
class DelayedSharedBufferConnection
{
private:
    nn::migration::idc::SharedBufferConnection m_Connection;

public:
    DelayedSharedBufferConnection() NN_NOEXCEPT
    {
    }
    explicit DelayedSharedBufferConnection(nn::migration::idc::SharedBufferConnection&& connection) NN_NOEXCEPT
        : m_Connection(std::move(connection))
    {
    }
    DelayedSharedBufferConnection(DelayedSharedBufferConnection&& rhs) NN_NOEXCEPT
        : m_Connection(std::move(rhs.m_Connection))
    {
    }
    DelayedSharedBufferConnection& operator =(DelayedSharedBufferConnection&& rhs) NN_NOEXCEPT
    {
        m_Connection = std::move(rhs.m_Connection);
        return *this;
    }

    nn::Result Send(size_t* pOutSentSize, const void* stream, size_t size, int timeoutSeconds, const nn::migration::detail::Cancellable* pCancellable, nn::migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) const NN_NOEXCEPT
    {
        return m_Connection.Send(pOutSentSize, stream, size, timeoutSeconds, pCancellable, pSpeedMonitor);
    }
    nn::Result Receive(size_t* pOutReceivedSize, void* outStream, size_t outStreamSize, int timeoutSeconds, const nn::migration::detail::Cancellable* pCancellable, nn::migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) const NN_NOEXCEPT
    {
        // NOTE: SharedBufferConnection の Receive にかかる時間は無視する
        NN_RESULT_DO(m_Connection.Receive(pOutReceivedSize, outStream, outStreamSize, timeoutSeconds, pCancellable, pSpeedMonitor));
        auto duration = nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>((static_cast<double>((*pOutReceivedSize) * 8) / (Mbps * 1000 * 1000)) * 1000));
        nn::os::SleepThread(duration);
        NN_RESULT_THROW_UNLESS(duration < nn::TimeSpan::FromSeconds(timeoutSeconds), nn::migration::idc::ResultTimeout());
        NN_RESULT_SUCCESS;
    }
    void Close() NN_NOEXCEPT
    {
        m_Connection.Close();
    }
};

template <typename ServerInfo, typename Tag>
class SimpleRegistrar
{
public:
    struct MatchingHint
    {
        uint64_t value;
    };

private:
    static const size_t AdvertizeCountMax = 2;
    static nn::migration::detail::Mutex s_Lock;
    static bool s_Initialized;

    struct Advertisement
    {
        nn::util::Uuid id;
        MatchingHint hint;
        nn::migration::UserMigrationServerProfile profile;
        ServerInfo serverInfo;
    };
    static Advertisement s_Advertises[AdvertizeCountMax];

public:
    bool RegisterAdvertisement(
        nn::util::Uuid* pOut,
        const ServerInfo& serverInfo, const nn::migration::UserMigrationServerProfile& profile, MatchingHint hint) NN_NOEXCEPT;
    bool EjectServer(ServerInfo* pOut, const nn::util::Uuid& id) NN_NOEXCEPT;
    size_t ScanServers(
        nn::migration::UserMigrationServerInfo* pOut, size_t count,
        MatchingHint hint) NN_NOEXCEPT;
};

class InprocessConnectionCreatorBase
{
private:
    typedef SimpleRegistrar<int, InprocessConnectionCreatorBase>::MatchingHint MatchingHint;

    static SimpleRegistrar<int, InprocessConnectionCreatorBase> s_Registrar;

    static const size_t PerConnectionBufferSize = 1024 * 1024;
    typedef std::aligned_storage<3 * 1024 * 1024, 8>::type ResourceStorage;

    static nn::migration::detail::Mutex s_Lock;
    static uint32_t s_Counter;
    static ResourceStorage s_ResourceStorage;

    static nn::migration::idc::SharedBufferConnectionManager* s_pManager;
    static std::aligned_storage<sizeof(nn::migration::idc::SharedBufferConnectionManager), std::alignment_of<nn::migration::idc::SharedBufferConnectionManager>::value>::type s_ManagerStorage;

public:
    static void InitializeAllocator() NN_NOEXCEPT;
    static void FinalizeAllocator() NN_NOEXCEPT;

private:
    MatchingHint GetMatchingHint() const NN_NOEXCEPT
    {
        MatchingHint hint = {};
        return hint;
    }

protected:
    nn::Result CreateConnectionBase(
        nn::migration::idc::SharedBufferConnection* pOut,
        const nn::migration::UserMigrationServerProfile& profile, const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;
    nn::Result ConnectBase(
        nn::migration::idc::SharedBufferConnection* pOut,
        const nn::util::Uuid& serverId) NN_NOEXCEPT;

public:
    nn::Result ScanServers(
        size_t* pOutActual, nn::migration::UserMigrationServerInfo* pOut, size_t count,
        const nn::account::NetworkServiceAccountId& nsaId,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;
};

template <typename Connection>
class InprocessConnectionCreator
    : public InprocessConnectionCreatorBase
{
public:
    template <typename EncryptionPolicy>
    nn::Result CreateConnection(
        Connection* pOut,
        const nn::account::NetworkServiceAccountId& nsaId, const nn::migration::UserMigrationServerProfile& profile,
        nn::migration::detail::IdcServer<Connection, EncryptionPolicy>& server,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    template <typename EncryptionPolicy>
    nn::Result Connect(
        Connection* pOut,
        const nn::util::Uuid& serverId, const nn::account::NetworkServiceAccountId& nsaId,
        nn::migration::detail::IdcClient<Connection, EncryptionPolicy>& client,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

class LocalhostConnectionCreator
{
private:
    typedef SimpleRegistrar<uint16_t, LocalhostConnectionCreator>::MatchingHint MatchingHint;

    static SimpleRegistrar<uint16_t, LocalhostConnectionCreator> s_Registrar;

    bool m_IsConnected;
    nn::nifm::NetworkConnection m_Connection;

    uint16_t GeneratePort() NN_NOEXCEPT
    {
        static std::atomic<uint16_t> s_Counter(100u);
        return s_Counter.fetch_add(1u);
    }

    nn::Result EnableInternet(const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    MatchingHint GetMatchingHint() const NN_NOEXCEPT
    {
        MatchingHint hint = {};
        return hint;
    }

public:
    LocalhostConnectionCreator() NN_NOEXCEPT
        : m_IsConnected(false)
        , m_Connection(nn::os::EventClearMode_ManualClear)
    {
    }
    template <typename EncryptionPolicy>
    nn::Result CreateConnection(
        nn::migration::idc::SocketConnection* pOut,
        const nn::account::NetworkServiceAccountId& nsaId, const nn::migration::UserMigrationServerProfile& profile,
        nn::migration::detail::IdcServer<nn::migration::idc::SocketConnection, EncryptionPolicy>& server,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    nn::Result ScanServers(
        size_t* pOutActual, nn::migration::UserMigrationServerInfo* pOut, size_t count,
        const nn::account::NetworkServiceAccountId& nsaId,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    template <typename EncryptionPolicy>
    nn::Result Connect(
        nn::migration::idc::SocketConnection* pOut,
        const nn::util::Uuid& serverId, const nn::account::NetworkServiceAccountId& nsaId,
        nn::migration::detail::IdcClient<nn::migration::idc::SocketConnection, EncryptionPolicy>& client,
        const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;
};

using InprocessConnectionPolicy = nn::migration::detail::ConnectionPolicy<nn::migration::idc::SharedBufferConnection, InprocessConnectionCreator<nn::migration::idc::SharedBufferConnection>>;

template <int64_t Mbps>
using DelayedInprocessConnectionPolicy = nn::migration::detail::ConnectionPolicy<DelayedSharedBufferConnection<Mbps>, InprocessConnectionCreator<DelayedSharedBufferConnection<Mbps>>>;

using LocalhostConnectionPolicy = nn::migration::detail::ConnectionPolicy<nn::migration::idc::SocketConnection, LocalhostConnectionCreator>;

class IdcSubsystemInitializer
{
public:
    IdcSubsystemInitializer() NN_NOEXCEPT
    {
        InprocessConnectionCreatorBase::InitializeAllocator();
    }
    ~IdcSubsystemInitializer() NN_NOEXCEPT
    {
        InprocessConnectionCreatorBase::FinalizeAllocator();
    }
};

struct NoSubsystemInitializer
{
};

}} // ~namespace nnt::migration

#include <nn/nn_Log.h>
#include <nn/nifm/nifm_TypesRequirement.h>
#include <nn/migration/detail/migration_NetworkConnection.h>

namespace nnt { namespace migration {

template <typename ServerInfo, typename Tag>
nn::migration::detail::Mutex SimpleRegistrar<ServerInfo, Tag>::s_Lock = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);
template <typename ServerInfo, typename Tag>
bool SimpleRegistrar<ServerInfo, Tag>::s_Initialized = false;
template <typename ServerInfo, typename Tag>
typename SimpleRegistrar<ServerInfo, Tag>::Advertisement SimpleRegistrar<ServerInfo, Tag>::s_Advertises[AdvertizeCountMax] = {};

template <typename ServerInfo, typename Tag>
bool SimpleRegistrar<ServerInfo, Tag>::RegisterAdvertisement(
    nn::util::Uuid* pOut,
    const ServerInfo& serverInfo, const nn::migration::UserMigrationServerProfile& profile, MatchingHint hint) NN_NOEXCEPT
{
    std::lock_guard<nn::migration::detail::Mutex> lock(s_Lock);
    if (!s_Initialized)
    {
        for (auto& e : s_Advertises)
        {
            e.id = nn::util::InvalidUuid;
        }
    }

    for (auto& e : s_Advertises)
    {
        if (e.id == nn::util::InvalidUuid)
        {
            e.id = nn::util::GenerateUuid();
            e.hint = hint;
            e.profile = profile;
            e.serverInfo = serverInfo;
            *pOut = e.id;

            char str[32];
            e.id.ToString(str, sizeof(str));
            NN_LOG("[SimpleRegistrar] Server registered with hint %016llx, %s\n", e.hint.value, str);
            return true;
        }
    }
    return false;
}

template <typename ServerInfo, typename Tag>
bool SimpleRegistrar<ServerInfo, Tag>::EjectServer(ServerInfo* pOut, const nn::util::Uuid& id) NN_NOEXCEPT
{
    char str[32];
    id.ToString(str, sizeof(str));
    NN_LOG("[SimpleRegistrar] Finding %s\n", str);

    std::lock_guard<nn::migration::detail::Mutex > lock(s_Lock);
    for (auto& e : s_Advertises)
    {
        if (e.id != nn::util::InvalidUuid)
        {
            e.id.ToString(str, sizeof(str));
            NN_LOG("[SimpleRegistrar]   - %s, %016llx\n", str, e.hint.value);
            if (e.id == id)
            {
                e.id.ToString(str, sizeof(str));
                NN_LOG("[SimpleRegistrar]     -> Server ejected\n");

                *pOut = e.serverInfo;
                e.id = nn::util::InvalidUuid;
                return true;
            }
        }
    }
    return false;
}

template <typename ServerInfo, typename Tag>
size_t SimpleRegistrar<ServerInfo, Tag>::ScanServers(
    nn::migration::UserMigrationServerInfo* pOut, size_t count,
    MatchingHint hint) NN_NOEXCEPT
{
    NN_LOG("[SimpleRegistrar] Listing with hint %016llx\n", hint.value);

    size_t filled = 0;
    std::lock_guard<nn::migration::detail::Mutex> lock(s_Lock);
    for (const auto& e : s_Advertises)
    {
        if (true
            && filled < count
            && e.id != nn::util::InvalidUuid
            && e.hint.value == hint.value)
        {
            char str[32];
            e.id.ToString(str, sizeof(str));
            NN_LOG("[SimpleRegistrar]   [%d] %s\n", filled, str);

            pOut[filled].id = e.id;
            pOut[filled].profile = e.profile;
            ++ filled;
        }
    }
    return filled;
}

// ---------------------------------------------------------------------------------------------
// InprocessConnectionCreator

template <typename Connection>
template <typename EncryptionPolicy>
nn::Result InprocessConnectionCreator<Connection>::CreateConnection(
    Connection* pOut,
    const nn::account::NetworkServiceAccountId& nsaId, const nn::migration::UserMigrationServerProfile& profile,
    nn::migration::detail::IdcServer<Connection, EncryptionPolicy>& server,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    // クライアントとの接続
    nn::migration::idc::SharedBufferConnection con;
    NN_RESULT_DO(CreateConnectionBase(&con, profile, pCancellable));

    auto connection = Connection(std::move(con));
    NN_RESULT_DO(server.HandleOnetimeTokenRequest(connection, pCancellable));
    NN_RESULT_DO(server.HandleFsKeySeedVerificationRequest(connection, pCancellable));

    *pOut = std::move(connection);
    NN_RESULT_SUCCESS;
}

template <typename Connection>
template <typename EncryptionPolicy>
nn::Result InprocessConnectionCreator<Connection>::Connect(
    Connection* pOut,
    const nn::util::Uuid& serverId, const nn::account::NetworkServiceAccountId& nsaId,
    nn::migration::detail::IdcClient<Connection, EncryptionPolicy>& client,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    // サーバーとの接続
    nn::migration::idc::SharedBufferConnection con;
    NN_RESULT_DO(ConnectBase(&con, serverId));

    auto connection = Connection(std::move(con));
    NN_RESULT_DO(client.GetOnetimeToken(connection, pCancellable));

    // OTT の検証, fsKeySeed の取得
    nn::nifm::NetworkConnection nc;
    NN_RESULT_DO(nn::migration::detail::ConnectToInternet(nc, nn::nifm::RequirementPreset_InternetBestEffort, pCancellable));
    NN_RESULT_DO(client.AuthenticateWithOnetimeToken(pCancellable));
    nc.CancelRequest();

    NN_RESULT_DO(client.VerifyFsKeySeed(connection, pCancellable));

    *pOut = std::move(connection);
    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------------------------------------------------
// LocalhostConnectionCreator

template <typename EncryptionPolicy>
nn::Result LocalhostConnectionCreator::CreateConnection(
    nn::migration::idc::SocketConnection* pOut,
    const nn::account::NetworkServiceAccountId& nsaId, const nn::migration::UserMigrationServerProfile& profile,
    nn::migration::detail::IdcServer<nn::migration::idc::SocketConnection, EncryptionPolicy>& server,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    NN_LOG("[LocalhostConnectionCreator] Enabling internet\n");
    NN_RESULT_DO(EnableInternet(pCancellable));

    // クライアントとの接続
    NN_LOG("[LocalhostConnectionCreator] Create listen socket\n");
    auto acceptSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    NN_RESULT_THROW_UNLESS(acceptSocket >= 0, nn::migration::idc::detail::HandleSocketError(nn::socket::GetLastError()));
    auto port = GeneratePort();
    NN_RESULT_DO(nn::migration::detail::SetupConnectionPort(acceptSocket, "127.0.0.1", port));
    nn::migration::idc::SocketConnection listeningSocket(acceptSocket);

    NN_LOG("[LocalhostConnectionCreator] Registering\n");
    nn::util::Uuid id;
    NN_RESULT_THROW_UNLESS(
        s_Registrar.RegisterAdvertisement(&id, port, profile, GetMatchingHint()),
        nn::migration::detail::ResultResourceLimit());
    bool releaseRequired = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (releaseRequired)
        {
            uint16_t ejected;
            s_Registrar.EjectServer(&ejected, id);
        }
    };

    NN_LOG("[LocalhostConnectionCreator] Waiting connection\n");
    nn::migration::idc::SocketConnection con;
    NN_RESULT_DO(nn::migration::detail::WaitConnection(&con, acceptSocket, nn::migration::detail::GetLdnConnectionTimeoutSeconds(), pCancellable));

    NN_RESULT_DO(server.HandleOnetimeTokenRequest(con, pCancellable));
    NN_RESULT_DO(server.HandleFsKeySeedVerificationRequest(con, pCancellable));

    NN_LOG("[LocalhostConnectionCreator] done\n");
    *pOut = std::move(con);
    releaseRequired = false;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
nn::Result LocalhostConnectionCreator::Connect(
    nn::migration::idc::SocketConnection* pOut,
    const nn::util::Uuid& serverId, const nn::account::NetworkServiceAccountId& nsaId,
    nn::migration::detail::IdcClient<nn::migration::idc::SocketConnection, EncryptionPolicy>& client,
    const nn::migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    NN_RESULT_DO(EnableInternet(pCancellable));

    // サーバーとの接続
    uint16_t port;
    NN_ABORT_UNLESS(s_Registrar.EjectServer(&port, serverId));
    auto clientSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    NN_RESULT_THROW_UNLESS(clientSocket >= 0, nn::migration::idc::detail::HandleSocketError(nn::socket::GetLastError()));
    NN_RESULT_DO(nn::migration::detail::ConnectToPort(clientSocket, "127.0.0.1", port));
    nn::migration::idc::SocketConnection con(clientSocket);

    NN_RESULT_DO(client.GetOnetimeToken(con, pCancellable));
    NN_RESULT_DO(client.AuthenticateWithOnetimeToken(pCancellable));
    NN_RESULT_DO(client.VerifyFsKeySeed(con, pCancellable));

    *pOut = std::move(con);
    NN_RESULT_SUCCESS;
}

}} // ~namespace nnt::migration
