﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_SelectorForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/friends/friends_DaemonSuspension.h>
#include <nn/pdm/pdm_NotifyEventApi.h>

#if defined(NNT_MIGRATION_ENABLE_HEADLESS)
#include <nn/nifm/nifm_NetworkConnection.h>
#include "nntMigration_Buffer.h"
#include "nntMigration_Nas.h"
#endif

#include "nntMigration_SaveData.h"

namespace nnt { namespace migration {

inline bool LoginNetworkServiceAccount(nn::account::SessionId sessionId, const char* id, const char* password) NN_NOEXCEPT
{
#if defined(NNT_MIGRATION_ENABLE_HEADLESS)
    nn::nifm::NetworkConnection connection;
    connection.SubmitRequestAndWait();
    NN_ABORT_UNLESS(connection.IsAvailable());

    nn::account::ExternalNetworkServiceAccountIntroducingProcedure proc;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ProxyProcedureToIntroduceExternalNetworkServiceAccountForRegistration(&proc, sessionId));
    nn::account::RequestUrl request;
    nn::account::CallbackUri callback;
    NN_ABORT_UNLESS_RESULT_SUCCESS(proc.GetRequest(&request, &callback));

    nnt::migration::Buffer response(4096);
    NN_ABORT_UNLESS(nnt::migration::GetAuthorizationViaNasHelper(response, id, password, request.url, true));

    nn::account::AsyncContext async;
    NN_ABORT_UNLESS_RESULT_SUCCESS(proc.ApplyResponseAsync(&async, response.Get<char>(), response.GetSize()));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
    e.Wait();
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetResult());
#else
    NN_UNUSED(id);
    NN_UNUSED(password);
    // ゲストログインモードで Psel を呼び出し
    auto r = nn::account::ShowUiToIntroduceExternalNetworkServiceAccountForRegistration(sessionId);
    if (!r.IsSuccess())
    {
        if (nn::account::ResultCancelledByUser::Includes(r))
        {
            NN_LOG("[Client] Login canceled\n");
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
#endif
    return true;
}

inline nn::account::Uid SelectUserAccount() NN_NOEXCEPT
{
    nn::account::Uid uid;
#if defined(NNT_MIGRATION_ENABLE_HEADLESS)
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, &uid, 1));
    NN_ABORT_UNLESS(count > 0);
#else
    auto r = nn::account::ShowUserSelector(&uid);
    if (!r.IsSuccess())
    {
        if (nn::account::ResultCancelledByUser::Includes(r))
        {
            return nn::account::InvalidUid;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
#endif
    return uid;
}

inline void CleanupUsers() NN_NOEXCEPT
{
    nn::nifm::NetworkConnection connection;
    connection.SubmitRequestAndWait();
    NN_ABORT_UNLESS(connection.IsAvailable());

    int count = 0;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, std::extent<decltype(users)>::value));

    for (int i = 0; i < count; ++ i)
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[i]));

        bool isRegistered;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&isRegistered));

        if (isRegistered)
        {
            if (admin.CheckNetworkServiceAccountAvailability().IsSuccess())
            {
                nn::account::AsyncContext async;
                NN_ABORT_UNLESS_RESULT_SUCCESS(admin.UnregisterAsync(&async));
                nn::os::SystemEvent e;
                NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
                e.Wait();
                NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetResult());
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(admin.DeleteRegistrationInfoLocally());
            }
        }
    }

    nn::friends::DaemonSuspension friendsSuspension;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SuspendDaemon(&friendsSuspension));

    for (int i = 0; i < count; ++i)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pdm::PresentUserAccountEventServiceSuspend(
            users[i],
            [&]() -> nn::Result
            {
                CleanupSaveData(users[i]);
                NN_RESULT_DO(nn::account::DeleteUser(users[i]));
                NN_RESULT_SUCCESS;
            }));
#else
        CleanupSaveData(users[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(users[i]));
#endif
    }
}

inline nn::account::Uid SetupUser(const char* id, const char* password) NN_NOEXCEPT
{
    NN_ALIGNAS(4096) static char s_LoginBuffer[nn::account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar];
    nn::account::ExternalNetworkServiceAccountRegistrar registrar;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CreateExternalNetworkServiceAccountRegistrar(&registrar, s_LoginBuffer, sizeof(s_LoginBuffer)));
    nn::account::SessionId sessionId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(registrar.GetOAuthProcedureSessionId(&sessionId));

    NN_ABORT_UNLESS(LoginNetworkServiceAccount(sessionId, id, password));

    nn::account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(registrar.RegisterUser(&uid));

    nn::account::AsyncContext async;
    NN_ABORT_UNLESS_RESULT_SUCCESS(registrar.RegisterNetworkServiceAccountAsync(&async));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
    e.Wait();
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetResult());

    return uid;
}

inline nn::account::Uid SetupUser() NN_NOEXCEPT
{
    nn::account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(uid));
    return uid;
}

}} // ~namespace nnt::migration
