﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/os/os_SystemEvent.h>

namespace
{
    nn::Bit8 g_MbufBuffer[8 * 1024 * 1024];
}

TEST(MultiplePoolsDeath, TooSmallUnitSize)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin - 1, nn::mbuf::PoolCapacityMin, 0 };

    // ユニットサイズが小さすぎるためバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(&config, 1), "");

    // ユニットサイズが小さすぎるため mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooLargeUnitSize)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMax + 1, nn::mbuf::PoolCapacityMin, 0 };

    // ユニットサイズが大きすぎるためバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(&config, 1), "");

    // ユニットサイズが大きすぎるため mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooLittleCapacity)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMin - 1, 0 };

    // プールの容量が小さすぎるため必要なバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(&config, 1), "");

    // プールの容量が小さすぎるため mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooLargeCapacity)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMax + 1, 0 };

    // プールの容量が限界を超えているため必要なバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(&config, 1), "");

    // プールの容量が限界を超えているため mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooFewTypes)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMin, 0 };

    // Type 数 0 では mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        nullptr, 0, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 0, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
    ASSERT_DEATH(nn::mbuf::Initialize(
        nullptr, 1, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooManyTypes)
{
    // mbuf プールの設定です。
    nn::mbuf::Type types[2];
    nn::os::SystemEvent systemEvent1(nn::os::EventClearMode_AutoClear, false);
    nn::os::SystemEvent systemEvent2(nn::os::EventClearMode_AutoClear, false);
    types[0].pEvent = systemEvent1.GetBase();
    types[1].pEvent = systemEvent2.GetBase();
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMin, 0 };

    // プール数よりも Type が多いと mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        types, 2, &config, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooFewPools)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMin, 0 };

    // プール数 0 では必要なバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(nullptr, 0), "");
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(&config, 0), "");
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(nullptr, 1), "");

    // プール数 0 では mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, nullptr, 0, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, &config, 0, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, nullptr, 1, g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}

TEST(MultiplePoolsDeath, TooManyPools)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config configs[nn::mbuf::PoolCountMax + 1] = { };
    for (int i = 0; i < nn::mbuf::PoolCountMax + 1; ++i)
    {
        configs[i].unitSize = nn::mbuf::MbufUnitSizeMin;
        configs[i].count = nn::mbuf::PoolCapacityMin;
        configs[i].type = 0;
    }

    // プール数が限界を超えているためバッファサイズを計算できません。
    ASSERT_DEATH(nn::mbuf::GetRequiredBufferSize(configs, nn::mbuf::PoolCountMax + 1), "");

    // プール数が限界を超えているため mbuf ライブラリを初期化できません。
    ASSERT_DEATH(nn::mbuf::Initialize(
        &type, 1, configs, nn::mbuf::PoolCountMax + 1,
        g_MbufBuffer, sizeof(g_MbufBuffer)), "");
}
