﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs/fs_ContentStorage.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_AddOnContentLocationResolver.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>

#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include <nn/fs/fs_Bis.h>

#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <new>

namespace {
    typedef std::string String;

    class AddOnContentLocationResolverTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {

        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());

#if defined(NN_BUILD_CONFIG_OS_WIN32)
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::System, nullptr));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
#else
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::System));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
#endif
            nn::ncm::Initialize();
            nn::lr::Initialize();
        }

        static void TearDownTestCase()
        {
            nn::ncm::Finalize();
            nn::lr::Finalize();
            nn::fs::UnmountHostRoot();
        }

        nn::ncm::ApplicationId GetRandomApplicationId()
        {
            nn::ncm::ApplicationId id = { m_RandomEngine() };
            return id;
        }

        std::mt19937_64 m_RandomEngine;
    };

    void CreateAocPath(nn::lr::Path* path, const nn::ncm::AddOnContentId aocId)
    {
        nn::util::TSNPrintf(path->string, sizeof(path->string), "aoc_%016llx", aocId.value);
    }

    nn::ncm::AddOnContentId CreateAddOnContentId(nn::ncm::ApplicationId appId, int index)
    {
        return { appId.value + 0x1000 + index };
    }
}

TEST_F(AddOnContentLocationResolverTest, ResolvePath)
{
    nn::lr::AddOnContentLocationResolver locationResolver;
    NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenAddOnContentLocationResolver(&locationResolver));

    //auto id = GetRandomApplicationId();
    nn::ncm::ApplicationId id = { 0x0005000c10000000 };

    // ---------------- 通常登録

    // 未登録の状態では、ProgramNotFound を返す
    nn::lr::Path path;
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultAddOnContentNotFound, locationResolver.ResolveAddOnContentPath(&path, CreateAddOnContentId(id, 0)));

    // 2048個まで登録可能
    for (int i = 0; i < 2048; ++i)
    {
        auto aocId = CreateAddOnContentId(id, i + 1);
        CreateAocPath(&path, aocId);

        // 登録は成功する
        NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterAddOnContentStorage(aocId, nn::ncm::StorageId::BuildInUser));
    }

    // 2049個以降は登録不可
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultTooManyPathRegistered, locationResolver.RegisterAddOnContentStorage(CreateAddOnContentId(id, 9999), nn::ncm::StorageId::BuildInUser));

    // INFO: 2048 個インストールできたら有効に
#if 0
    // 登録した状態で Resolve すると、登録したパスを返す
    for (int i = 0; i < 2048; ++i)
    {
        auto aocId = CreateAddOnContentId(id, i);

        NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveAddOnContentPath(&path, aocId));

        nn::lr::Path masterPath;
        CreateAocPath(&masterPath, aocId);
        EXPECT_STRCASEEQ(path.string, masterPath.string);
    }
#endif
    // 登録を解除する
    locationResolver.UnregisterAllAddOnContentPath();

    // 解除した後は、再び ProgramNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultAddOnContentNotFound, locationResolver.ResolveAddOnContentPath(&path, CreateAddOnContentId(id, 0)));
}
