﻿using System;
using System.Threading;

namespace LdnTestBridge
{
    /// <summary>
    /// 同期を処理するオブジェクトです。
    /// </summary>
    class TestSynchronizer
    {
        /// <summary>
        /// キーワードの最小長です。
        /// </summary>
        public const int KEYWORD_LENGTH_MIN = 1;

        /// <summary>
        /// キーワードの最大長です。
        /// </summary>
        public const int KEYWORD_LENGTH_MAX = 128;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="keyword">同期に使用するキーワードです。</param>
        /// <param name="memberCountMax">同期するノードの最大数です。</param>
        public TestSynchronizer(string keyword, int memberCountMax)
        {
            m_Keyword = keyword;
            m_MemberCount = 0;
            m_MemberCountMax = memberCountMax;
            m_Event = new ManualResetEventSlim();
        }

        /// <summary>
        /// 同期します。
        /// </summary>
        /// <param name="timeout">タイムアウト時間です。</param>
        /// <returns>同期処理の成否です。</returns>
        public bool Sync(TimeSpan timeout)
        {
            lock (m_LockObject)
            {
                ++m_MemberCount;
                Console.WriteLine("Sync: {0} ({1}/{2})",
                    m_Keyword, m_MemberCount, m_MemberCountMax);
                if (m_MemberCount == m_MemberCountMax)
                {
                    m_Event.Set();
                }
            }
            bool isSucceeded = m_Event.Wait(timeout);
            lock (m_LockObject)
            {
                --m_MemberCount;
            }
            return isSucceeded;
        }

        /// <summary>
        /// 同期で共有するデータです。
        /// </summary>
        public string Data
        {
            get;
            set;
        }

        /// <summary>
        /// 同期に使用するキーワードを取得します。
        /// </summary>
        public string Keyword
        {
            get
            {
                return m_Keyword;
            }
        }
        private string m_Keyword;

        /// <summary>
        /// 現時点で同期待ちしているメンバーの数です。
        /// </summary>
        public int MemberCount
        {
            get
            {
                return m_MemberCount;
            }
        }
        private int m_MemberCount;

        /// <summary>
        /// 同期待ちに参加するメンバーの最大数です。
        /// </summary>
        public int MemberCountMax
        {
            get
            {
                return m_MemberCountMax;
            }
        }
        private int m_MemberCountMax;

        private object m_LockObject = new Object();
        private ManualResetEventSlim m_Event;
    }
}
