﻿using System;
using System.Collections.ObjectModel;

namespace LdnTestBridge
{
    /// <summary>
    /// テストグループを管理するリポジトリです。
    /// </summary>
    class TestGroupRepository : IRepository<TestGroup>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TestGroupRepository()
        {
            m_Collection = new ObservableCollection<TestGroup>();
            m_ReadOnlyCollection = new ReadOnlyObservableCollection<TestGroup>(m_Collection);
        }

        /// <summary>
        /// 全てのテストグループを取得します。
        /// </summary>
        public ReadOnlyObservableCollection<TestGroup> All
        {
            get
            {
                return m_ReadOnlyCollection;
            }
        }

        /// <summary>
        /// 新しいテストグループを登録します。
        /// </summary>
        /// <param name="group">対象のテストグループです。</param>
        /// <returns>処理の結果です。</returns>
        public bool Add(TestGroup group)
        {
            lock (m_LockObject)
            {
                if (Find((obj) => obj.Name == group.Name) == null)
                {
                    m_Collection.Add(group);
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// テストグループを削除します。
        /// </summary>
        /// <param name="group">対象のグループです。</param>
        public void Remove(TestGroup group)
        {
            lock (m_LockObject)
            {
                m_Collection.Remove(group);
            }
        }

        /// <summary>
        /// テストグループを検索します。
        /// </summary>
        /// <param name="group">対象のグループです。</param>
        public TestGroup Find(Func<TestGroup, bool> pred)
        {
            foreach (var entity in All)
            {
                pred(entity);
                if (pred(entity))
                {
                    return entity;
                }
            }
            return null;
        }

        private ObservableCollection<TestGroup> m_Collection;
        private ReadOnlyObservableCollection<TestGroup> m_ReadOnlyCollection;
        private object m_LockObject = new Object();
    }
}
