﻿using System;
using System.Collections.Generic;

namespace LdnTestBridge
{
    /// <summary>
    /// テストグループです。
    /// </summary>
    internal class TestGroup
    {
        /// <summary>
        /// クライアントの接続時に実行されるハンドラです。
        /// </summary>
        /// <param name="sender">イベントをシグナルしたオブジェクトです。</param>
        /// <returns></returns>
        public delegate void JoinedEventHandler(object sender);

        /// <summary>
        /// クライアントの接続時に実行されるイベントです。
        /// </summary>
        public event JoinedEventHandler Joined;

        /// <summary>
        /// グループ名の最小長です。
        /// </summary>
        public const int NAME_LENGTH_MIN = 1;

        /// <summary>
        /// グループ名の最台長です。
        /// </summary>
        public const int NAME_LENGTH_MAX = 64;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="name">テストグループの名前です。</param>
        /// <param name="clientCount">クライアント数です。</param>
        public TestGroup(string name, int clientCount)
        {
            m_Name = name;
            m_ClientCount = 0;
            m_ClientCountMax = clientCount;
            m_Synchronizers = new Dictionary<string, TestSynchronizer>();
        }

        /// <summary>
        /// グループに参加します。
        /// </summary>
        /// <returns>クライアントのインデックスです。失敗した場合は 0 以下の値が返ります。</returns>
        public int Join()
        {
            int clientIndex;
            lock (m_LockObject)
            {
                if (m_ClientCountMax <= m_ClientCount)
                {
                    return 0;
                }
                ++m_ClientCount;
                clientIndex = m_ClientCount;
            }
            if (Joined != null)
            {
                Joined(this);
            }
            return clientIndex;
        }

        /// <summary>
        /// グループから離脱します。
        /// </summary>
        public void Leave()
        {
            lock (m_LockObject)
            {
                --m_ClientCount;
            }
        }

        /// <summary>
        /// 同期に使用するオブジェクトを取得します。
        /// </summary>
        /// <param name="keyword">同期に使用するキーワードです。</param>
        /// <returns>同期に使用するオブジェクトです。</returns>
        public TestSynchronizer GetSynchronizer(string keyword)
        {
            TestSynchronizer sync;
            lock (m_LockObject)
            {
                // 指定されたキーワードに一致する同期オブジェクトを取得します。
                if (!m_Synchronizers.TryGetValue(keyword, out sync))
                {
                    // 存在しない場合には新しく生成します。
                    Console.WriteLine("Sync Started: {0}",  keyword);
                    sync = new TestSynchronizer(keyword, m_ClientCountMax + 1);
                    m_Synchronizers.Add(keyword, sync);
                }
            }
            return sync;
        }

        /// <summary>
        /// 同期に使用するオブジェクトを削除します。
        /// </summary>
        /// <param name="keyword">キーワードです。</param>
        public void RemoveSynchronizer(string keyword)
        {
            lock (m_LockObject)
            {
                m_Synchronizers.Remove(keyword);
            }
        }

        /// <summary>
        /// テストグループの名前を取得します。
        /// </summary>
        public string Name
        {
            get
            {
                return m_Name;
            }
        }
        private string m_Name;

        /// <summary>
        /// 現時点のクライアント数です。
        /// </summary>
        public int ClientCount
        {
            get
            {
                return m_ClientCount;
            }
        }
        private int m_ClientCount;

        /// <summary>
        /// テストクライアントの最大数です。
        /// </summary>
        public int ClientCountMax
        {
            get
            {
                return m_ClientCountMax;
            }
        }
        private int m_ClientCountMax;

        private object m_LockObject = new Object();
        private IDictionary<string, TestSynchronizer> m_Synchronizers;
    }
}
