﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "../../../../../../Net/Sources/Tests/Performance/testNet_P2P/testNet_P2P.h"
#include "../Common/Definition.h"
#include "../../Common/Util.h"

nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

NN_ALIGNAS(4096) char g_LdnThreadStack[8192];
NN_ALIGNAS(4096) char g_MonitorThreadStack[8192];

int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

const size_t BufferSize     = 512;

enum Type
{
    Type_P2P,
    Type_BroadcastServer,
    Type_BroadcastClient,
};

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::ThreadType          P2PThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::MutexType           mutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint32_t                    myCounter;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    int                         argc;
    char**                      argv;
    int                         cursor;
};

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);
    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_StationConnected)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 現在のネットワークの状態を取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        nn::os::LockMutex(&app.mutex);
        std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void P2PFunction(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    char argvAddr[BufferSize]     = "--NODES_ADDR_INFO=";
    char argvIndex[16];
    char address[32];
    nn::socket::InAddr addr;
    int nodeIndex = 0;

    for (int i = 0; i < app.networkInfo.ldn.nodeCount ; ++i)
    {
        if(app.networkInfo.ldn.nodes[i].ipv4Address.raw ==  app.ipv4Address.raw)
        {
            nodeIndex = i;
        }
        addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[i].ipv4Address.raw);
        sprintf(address, "%s:%d,", nn::socket::InetNtoa(addr), 8001 + i);
        strncat(argvAddr, address, BufferSize - strlen(argvAddr) - 1);
    }
    argvAddr[strlen(argvAddr) - 1] = '\0';

    sprintf(argvIndex, "--NODE_INDEX=%d", nodeIndex);

    int    argC = 3 + (app.argc - 1);

    char** arguments = (char **)std::malloc(sizeof(char *) * argC);

    arguments[0] = app.argv[0];
    arguments[1] = argvIndex;
    arguments[2] = argvAddr;

    for (int i =0; i < (app.argc - 1); ++i)
    {
        arguments[i + 3] = app.argv[i + 1];
    }

    const char** argV = (const char**)arguments;

    NN_LOG("start(argc:%d) ", argC);
    for(int i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i], argV[i]);
    }
    NN_LOG("\n");

    nnt::net::p2p::PacketRateTest::RunTest(argC, argV);

    std::free(arguments);
/*
    auto& app = *static_cast<ApplicationResource*>(arg);

    char argvName[]             = "p2p";
    char argvIndex[]            = "--NODE_INDEX=1";
    char argvAddr[128];
    char argvAddress1[17];
    char argvAddress2[17];
    nn::socket::InAddr addr1;
    nn::socket::InAddr addr2;

    addr1.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[0].ipv4Address.raw);
    strncpy(argvAddress1, nn::socket::InetNtoa(addr1), sizeof(argvAddress1) / sizeof(argvAddress1[0]) );
NN_LOG("addr1 : %x\n", addr1.S_addr);
NN_LOG("argvAddress1 : %s\n", argvAddress1);

    addr2.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[1].ipv4Address.raw);
    strncpy(argvAddress2, nn::socket::InetNtoa(addr2), sizeof(argvAddress2) / sizeof(argvAddress2[0]) );
NN_LOG("addr2 : %x\n", addr2.S_addr);
NN_LOG("argvAddress1 : %s / argvAddress2 : %s\n", argvAddress1, argvAddress2);

    sprintf(argvAddr, "--NODES_ADDR_INFO=%s:8000,%s:8001", argvAddress1, argvAddress2);
    NN_LOG("argvAddr : %s\n", argvAddr);

    char** arguments = (char **)std::malloc(sizeof(char *) * (2));

    arguments[0] = argvName;
    arguments[1] = argvIndex;
    arguments[2] = argvAddr;

    int    argC = 3;
    const char** argV = (const char**)arguments;

    NN_LOG("start(argc:%d) ", argC);
    for(int i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i], argV[i]);
    }
    NN_LOG("\n");

    P2Pmain(argC, argV);

    std::free(arguments);
*/
}

nn::Result Scan(ApplicationResource* pApp, int* pOutScanResultCount) NN_NOEXCEPT
{
    // IntentId が一致する LDN ネットワークのみを探索対象とします。
    nn::ldn::ScanFilter filter = { };
    filter.networkType = nn::ldn::NetworkType_Ldn;
    filter.networkId.intentId = BasicSampleIntentId;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

    // LDN ネットワークを探索します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
        g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
        filter, nn::ldn::AutoChannel));

    return nn::ResultSuccess();
}

nn::Result Connect(ApplicationResource* pApp, const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
{
    // 接続に必要なセキュリティパラメータを設定します。
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "p2pSta", nn::ldn::UserNameBytesMax);

    // LDN ネットワークに接続します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Connect(
        network, security, user, 0, nn::ldn::ConnectOption_None));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void Disconnect(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークから切断します。
    nn::ldn::Disconnect();
}

void LdnStation(void* arg) NN_NOEXCEPT
{
    nn::Result result;
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ステーションとして起動します。
    result = nn::ldn::OpenStation();
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークを探索します。
    for (int retryCount = 0; retryCount < 10; ++retryCount)
    {
        NN_LOG("LDN ネットワークを探索中です・・・。\n\n");
        int scanResultCount;
        result = Scan(&app, &scanResultCount);
        if (result <= nn::ldn::ResultInvalidState())
        {
            // 無線スイッチオフなど通信機能を利用できない状態です。
            // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
            return;
        }
        else if (scanResultCount == 0)
        {
            NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
            nn::ldn::CloseStation();
        }
        else
        {
            break;
        }
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークへの接続を試行します。
    for (int retryCount = 0; retryCount < 10; ++retryCount)
    {
        result = Connect(&app, g_ScanResult[0]);
        if (result <= nn::ldn::ResultInvalidState())
        {
            // 無線スイッチオフなど通信機能を利用できない状態です。
            // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
            return;
        }
        else if (result <= nn::ldn::ResultConnectionFailed())
        {
            NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
            nn::ldn::CloseStation();
        }
        else
        {
            break;
        }
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("LDN ネットワークへの接続に成功しました。\n");
    NN_LOG("IP Address (AP)  : %s/%s\n",
        ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address),
        ConvertToString(app.subnetMask));
    NN_LOG("IP Address (STA) : %s/%s\n\n",
        ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));

    // サンプルの操作方法を出力します。
    NN_LOG("A ボタンで 計測を開始します。\n");

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // ソケットを破棄します。
    nn::socket::Shutdown(app.socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(app.socket);

    // LDN ネットワークから切断し、データの送受信を終了します。
    Disconnect(&app);
    nn::ldn::CloseStation();
    NN_LOG("LDN ネットワークから切断されました。\n\n");
}

extern "C" void nnMain()
{
    nn::Result result;

    // デバッグパッドから入力を取得するために HID ライブラリを取得します。
    nn::hid::InitializeDebugPad();

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);


    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    app.argc = ::nnt::GetHostArgc();
    app.argv = ::nnt::GetHostArgv();

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    // メインループです。
    nn::hid::DebugPadState padState;
    nn::os::ThreadType ldnThread;

    nn::os::InitializeEvent(
        &app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMutex(&app.mutex, false, 0);
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &ldnThread, LdnStation, &app,
        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
    nn::os::StartThread(&ldnThread);

    for (;;)
    {
        // デバッグパッドの入力を取得します。
        nn::hid::GetDebugPadState(&padState);

        if (padState.buttons.Test<nn::hid::DebugPadButton::A>())
        {
            NN_LOG("入力を受け付けました。計測を開始します。\n\n");

            P2PFunction(&app);

            nn::os::WaitThread(&ldnThread);
            nn::os::DestroyThread(&ldnThread);
            nn::os::FinalizeMutex(&app.mutex);
            nn::os::FinalizeEvent(&app.cancelEvent);

            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    // socket ライブラリの使用を終了します。
    result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ライブラリの使用を終了します。
    nn::ldn::Finalize();
}
