﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "Definition.h"
#include "Types.h"
#include "Util.h"
#include "Graphics.h"
#include "HidController.h"
#include "DisplayInfo.h"
#include "LdnAccessPoint.h"
#include "LdnStation.h"
#include "UpdateState.h"
#include "PacketTransfer.h"

#define ARG_FLAG(s) (::std::strcmp((s), argv[i]) == 0)
#define ARG_SPARAM(s) ((::std::strcmp((s), argv[i]) == 0) && (i + 1) < argc && (( r = argv[++i]) != NULL))
#define ARG_LPARAM(s) ((::std::strncmp((s), argv[i], ::std::strlen(s)) == 0) && ((l = ::std::strlen(s)) < ::std::strlen(argv[i])) && (( r =& argv[i][l]) != NULL))

NN_ALIGNAS(4096) char g_StateThreadThreadStack[8192];

const int StateThreadPriority = nn::os::DefaultThreadPriority + 1;

const int    MovementThMax =  50;
const int    MovementThMin = -50;
const Region regionA = { 961, 1280,   0, 720 };
const Region regionB = { 321,  960, 361, 720 };
const Region regionX = { 321,  960,   0, 360 };
const Region regionY = {   0,  320,   0, 720 };

int g_LastX;
int g_LastY;
int g_PreviousX;
int g_PreviousY;



int GetArg(ApplicationResource* pApp, int argc, char*argv[])
{
    if (argc < 2)
    {
        return 0;
    }

    int i;
    for (i = 1; i < argc; i++) {
        char *r = NULL;
        int l = 0;

        if (argv[i][0] != '-') {
            break;
        }
        else if (ARG_FLAG("--")) {
            i++;
            break;
        }
        else if (ARG_SPARAM("-g") || ARG_LPARAM("--group=")) {
            for (int i = 0; i < 5; i++)
            {
                if (::std::strncmp(r, strSettingGroup[i], 1) == 0)
                {
                    pApp->ldnSetting.group = i;
                    break;
                }
            }
        }
        else if (ARG_SPARAM("-m") || ARG_LPARAM("--mode=")) {
            if (::std::strncmp(r, "ap", 2) == 0 || ::std::strncmp(r, "AP", 2) == 0)
            {
                pApp->ldnSetting.mode = LdnMode_AccessPoint;
            }
            else if (::std::strncmp(r, "sta", 3) == 0 || ::std::strncmp(r, "STA", 3) == 0)
            {
                pApp->ldnSetting.mode = LdnMode_Station;
            }
        }
        else if (ARG_SPARAM("-c") || ARG_LPARAM("--channel=")) {
            if (::std::strncmp(r, strSettingChannnel2g[0], ::std::strlen(strSettingChannnel2g[0])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_2GHz;
                pApp->ldnSetting.channel2GHz = LdnChannel2G_1;
            }
            else if (::std::strncmp(r, strSettingChannnel2g[1], ::std::strlen(strSettingChannnel2g[1])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_2GHz;
                pApp->ldnSetting.channel2GHz = LdnChannel2G_6;
            }
            else if (::std::strncmp(r, strSettingChannnel2g[2], ::std::strlen(strSettingChannnel2g[2])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_2GHz;
                pApp->ldnSetting.channel2GHz = LdnChannel2G_11;
            }
            else if (::std::strncmp(r, strSettingChannnel5g[0], ::std::strlen(strSettingChannnel5g[0])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_5GHz;
                pApp->ldnSetting.channel5GHz = LdnChannel5G_36;
            }
            else if (::std::strncmp(r, strSettingChannnel5g[1], ::std::strlen(strSettingChannnel5g[1])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_5GHz;
                pApp->ldnSetting.channel5GHz = LdnChannel5G_40;
            }
            else if (::std::strncmp(r, strSettingChannnel5g[2], ::std::strlen(strSettingChannnel5g[2])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_5GHz;
                pApp->ldnSetting.channel5GHz = LdnChannel5G_44;
            }
            else if (::std::strncmp(r, strSettingChannnel5g[3], ::std::strlen(strSettingChannnel5g[3])) == 0)
            {
                pApp->ldnSetting.band = LdnBand_5GHz;
                pApp->ldnSetting.channel5GHz = LdnChannel5G_48;
            }
        }
        else if (ARG_SPARAM("-s") || ARG_LPARAM("--security=")) {
            if (::std::strncmp(r, "aes", 3) == 0 || ::std::strncmp(r, "AES", 3) == 0)
            {
                pApp->ldnSetting.secrity = LdnSecurity_StaticAES;
            }
            else if (::std::strncmp(r, "none", 4) == 0 || ::std::strncmp(r, "NONE", 4) == 0)
            {
                pApp->ldnSetting.secrity = LdnSecurity_None;
            }
        }
        else if (ARG_SPARAM("-n") || ARG_LPARAM("--nodemax=")) {
            int nodeCount = ::std::atoi(r);
            if (ModeCount <= nodeCount && nodeCount <= nn::ldn::NodeCountMax)
            {
                pApp->ldnSetting.nodeCountMax = nodeCount;
            }
        }
        else if (ARG_SPARAM("-t") || ARG_LPARAM("--castto=")) {
            if (::std::strncmp(r, "Broadcast", 9) == 0)
            {
                pApp->translateSetting.castTo = PacketCastTo_Broadcast;
            }
            else if (::std::strncmp(r, "UnicastToAP", 11) == 0)
            {
                pApp->translateSetting.castTo = PacketCastTo_UnicastToAP;
            }
            else if (::std::strncmp(r, "UnicastToAll", 12) == 0)
            {
                pApp->translateSetting.castTo = PacketCastTo_UnicastToAll;
            }
            else if (::std::strncmp(r, "PingToAll", 9) == 0)
            {
                pApp->translateSetting.castTo = PacketCastTo_Ping;
            }
        }
        else if (ARG_SPARAM("-r") || ARG_LPARAM("--rate=")) {
            int rate = ::std::atoi(r);
            if (RateMin <= rate && rate <= RateMax)
            {
                pApp->translateSetting.rate = rate;
            }
        }
        else if (ARG_SPARAM("-p") || ARG_LPARAM("--packetsize=")) {
            int size = ::std::atoi(r);
            if (PacketSizeMin <= size && size <= PacketSizeMax)
            {
                pApp->translateSetting.packetSize = size;
            }
        }
        else {
            NN_LOG("Option Error\n");
        }
    }

    return 0;
}// NOLINT(readability/fn_size)

bool isTouchInput(int x, int y, Region region)
{
    if (region.xMin < x && x < region.xMax &&
        region.yMin < y && y < region.yMax)
    {
        return true;
    }
    return false;
}

Touch UpdateTouch(ApplicationResource* pApp, int x, int y)
{
    Touch retValue = Touch_None;

    if (x != 0 && y != 0 && g_PreviousX != 0 && g_PreviousY != 0)
    {
        //pApp->diffX = g_PreviousX - x;
        //pApp->diffY = g_PreviousY - y;
        pApp->diffX = g_LastX - x;
        pApp->diffY = g_LastY - y;
    }
    else
    {
        pApp->diffX = 0;
        pApp->diffY = 0;
    }
   /*
    NN_LOG("Touch : [%d, %d] - [%d, %d] - [%d, %d]\n",
        x, y, g_PreviousX, g_PreviousY, pApp->diffX, pApp->diffY);
    */

    if ((x != 0 && y != 0) && (g_LastX == 0 && g_LastY == 0))
    {
        g_LastX = x;
        g_LastY = y;
    }
    else if ((x == 0 && y == 0) && (g_LastX != 0 && g_LastY != 0))
    {
        int movementX = g_PreviousX - g_LastX;
        int movementY = g_PreviousY - g_LastY;
        /*
        NN_LOG("Touch : [%d, %d] - [%d, %d] - [%d, %d] - [%d, %d]\n",
            x, y, g_PreviousX, g_PreviousY, g_LastX, g_LastY, movementX, movementY);
        */
        if (MovementThMin < movementX && movementX < MovementThMax &&
            MovementThMin < movementY && movementY < MovementThMax)
        {
            if (isTouchInput(g_LastX, g_LastY, regionA))
            {
                retValue = Touch_A;
            }
            else if (isTouchInput(g_LastX, g_LastY, regionB))
            {
                retValue = Touch_B;
            }
            else if (isTouchInput(g_LastX, g_LastY, regionX))
            {
                retValue = Touch_X;
            }
            else if (isTouchInput(g_LastX, g_LastY, regionY))
            {
                retValue = Touch_Y;
            }
        }
        else
        {
            if (MovementThMin < movementX && movementX < MovementThMax && movementY < 0)
            {
                retValue = Touch_Up;
            }
            else if (MovementThMin < movementX && movementX < MovementThMax && movementY > 0)
            {
                retValue = Touch_Down;
            }
            else if (movementX > 0 && MovementThMin < movementY && movementY < MovementThMax)
            {
                retValue = Touch_Right;
            }
            else if (movementX < 0 && MovementThMin < movementY && movementY < MovementThMax)
            {
                retValue = Touch_Left;
            }
        }

        g_LastX = 0;
        g_LastY = 0;
    }

    g_PreviousX = x;
    g_PreviousY = y;

    return retValue;
}

extern "C" void nnMain()
{
    nn::Result result;

    InitializeGraphicSystem();

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    ::std::memset(&app, 0, sizeof(app));
    app.initCursor = 1;
    app.settingCursor = 1;
    app.ldnSetting.channel5GHz = LdnChannel5G_36;
    app.ldnSetting.channel2GHz = LdnChannel2G_1;
    app.ldnSetting.nodeCountMax = nn::ldn::NodeCountMax;
    app.translateSetting.castTo = PacketCastTo_Broadcast;
    app.translateSetting.rate = 1;
    app.translateSetting.packetSize = 100;

    // 引数を処理します。
    GetArg(&app, nn::os::GetHostArgc(), nn::os::GetHostArgv());

     // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    InitializeHidController();

    nn::hid::InitializeTouchScreen();

    nn::os::InitializeMutex(&app.mutex, false, 0);

    nn::os::ThreadType StateThread;

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &StateThread, UpdateState, &app,
        g_StateThreadThreadStack, sizeof(g_StateThreadThreadStack), StateThreadPriority));
    nn::os::StartThread(&StateThread);

    nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> states[nn::hid::TouchScreenStateCountMax];

    // メインループです。
    for (;;)
    {
        nn::hid::GetTouchScreenStates(states, nn::hid::TouchScreenStateCountMax);
        app.touch = UpdateTouch(&app, states->touches[0].x, states->touches[0].y);

        if (app.appState == ApplicationState_Exit)
        {
            break;
        }

        DisplayInfo(app);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(33));
    }

    nn::os::WaitThread(&StateThread);
    nn::os::DestroyThread(&StateThread);

    nn::os::FinalizeMutex(&app.mutex);

    FinalizeHidController();

    nn::ldn::Finalize();

    FinalizeGraphicSystem();
}
