﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/ldn/detail/ldn_Config.h>
#include <nn/ldn/detail/Utility/ldn_Crypto.h>
#include <nnt.h>

namespace
{
    char ConvertToHexChar(int n) NN_NOEXCEPT
    {
        if (0 <= n && n <= 9)
        {
            return static_cast<char>('0' + n);
        }
        else
        {
            return static_cast<char>('a' + (n - 10));
        }
    }

    void ConvertToHexString(void* outBuffer, size_t bufferSize,
                            const void* input, size_t inputSize) NN_NOEXCEPT
    {
        ASSERT_TRUE(inputSize * 2 < bufferSize);
        auto out = static_cast<char*>(outBuffer);
        auto in  = static_cast<const uint8_t*>(input);
        for (size_t i = 0; i < inputSize; ++i)
        {
            auto high = (in[i] >> 4) & 0x0F;
            auto low  = in[i] & 0x0F;
            out[i * 2 + 0] = ConvertToHexChar(high);
            out[i * 2 + 1] = ConvertToHexChar(low);
        }
        out[inputSize * 2] = 0;
    }

    template <int N>
    void Verify(const void* hash, size_t hashSize, const char* expected) NN_NOEXCEPT
    {
        ASSERT_EQ(hashSize, N);
        char buffer[N * 2 + 1];
        ConvertToHexString(buffer, sizeof(buffer), hash, hashSize);
        ASSERT_STREQ(expected, buffer);
    }

    char g_Buffer[1024 * 1024];
};

//
// 製品環境か否かの判定が正しく動作することを確認します。
//
TEST(IsProductionEnvironment, MultipleTimes)
{
    // 製品機でテストを動かすことはないので、常に false を返すはずです。
    const int retryCount = 256;
    for (int i = 0; i < retryCount; ++i)
    {
        ASSERT_EQ(false, nn::ldn::detail::IsProductionEnvironment());
    }
}

//
// 生成される鍵の互換性を確認します。つまり、初期仕様と同じ鍵が生成されていれば成功します。
// 開発機用のテストなので、製品機では鍵が一致しない可能性があります。
//
TEST(GenerateAesKey, Compatiblity)
{
    uint8_t key[16];
    uint8_t kekSource[16] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };

    nn::ldn::detail::GenerateAesKey(key, nullptr, 0, kekSource, sizeof(kekSource));
    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    Verify<16>(key, sizeof(key), "454b896f7859490f0f63aaf730b93d3c");
    #else
    Verify<16>(key, sizeof(key), "0d0db758df32df1763b996c3f2582664");
    #endif

    char keySource1[] = "abc";
    nn::ldn::detail::GenerateAesKey(key, &keySource1, 3, kekSource, sizeof(kekSource));
    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    Verify<16>(key, sizeof(key), "207ba7205613f0a3c237e37813e23023");
    #else
    Verify<16>(key, sizeof(key), "54c565a5c8cf0ce9b80322d53699bd63");
    #endif

    char keySource2[] = "abcdefghijklmnop";
    nn::ldn::detail::GenerateAesKey(key, &keySource2, 16, kekSource, sizeof(kekSource));
    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    Verify<16>(key, sizeof(key), "bec453ec3d1999becc02d7ea5772b6fb");
    #else
    Verify<16>(key, sizeof(key), "1d20df76fd74905dd5621edba6b88a09");
    #endif

    char keySource3[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    nn::ldn::detail::GenerateAesKey(key, &keySource3, 26, kekSource, sizeof(kekSource));
    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    Verify<16>(key, sizeof(key), "a7214c9f86b276d0d6244b4f405dae46");
    #else
    Verify<16>(key, sizeof(key), "38511b8299491eaf9719541a1bbde5e1");
    #endif

    std::memset(g_Buffer, 'a', sizeof(g_Buffer));
    nn::ldn::detail::GenerateAesKey(key, g_Buffer, 1000000, kekSource, sizeof(kekSource));
    #if defined(NN_LDN_BUILD_CONFIG_SPL_ENABLED)
    Verify<16>(key, sizeof(key), "4915859b8829caeaaa2e60550d10e40b");
    #else
    Verify<16>(key, sizeof(key), "237a1d46deda389178e3a5e9efe0a127");
    #endif
}
