﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Wlan.h>
#include <nnt.h>

namespace
{
    // このテストでは Wlan を使用します。
    typedef nn::ldn::detail::Wlan NetworkInterface;
    const size_t NetworkInterfaceBufferSize = sizeof(nn::ldn::detail::impl::WlanBuffer);

    // ネットワーク・インタフェース用のバッファです。
    NN_ALIGNAS(4096) char g_NetworkInterfaceBuffer[NetworkInterfaceBufferSize];

    // 標準の SSID です。
    const nn::ldn::Ssid DefaultSsid = nn::ldn::MakeSsid("DefaultSsid");

    // スキャン用のバッファです。
    nn::ldn::detail::L2ScanResult g_ScanBuffer[nn::ldn::ScanResultCountMax];

} // namespace <unnamed>

//
// OpenAccessPoint/CloseAccessPoint を繰り返し実行します。
//
TEST(OpenAccessPoint, MultipleTimes)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    const int repeatCountMax = 64;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
        ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
        ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
    }
}

//
// OpenAccessPoint や CloseAccessPoint でイベントがシグナルされないことを検証します。
//
TEST(OpenAccessPoint, NotSignaled)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // 状態変化を取得するイベントを取得します。
    auto& stateChangeEvent = pNetworkInterface->GetStateChangeEvent();
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // OpenAccessPoint 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // CloseAccessPoint 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());
}

//
// OpenAccessPoint の後、 CloseAccessPoint をとばして Finalize します。
//
TEST(OpenAccessPoint, CloseAccessPointByFinalize)
{
    // ネットワークインタフェースを初期化します。
    auto* pNetworkInterface = new NetworkInterface(
        g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize);
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // CloseAccessPoint を実行せず、 ネットワーク・インタフェースを解放して AP としての動作を停止します。
    delete pNetworkInterface;
}

//
// CreateNetwork や DestroyNetwork でイベントがシグナルされることを確認します。
//
TEST(CreateNetwork, NotSignaled)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // 状態変化を取得するイベントを取得します。
    auto& stateChangeEvent = pNetworkInterface->GetStateChangeEvent();
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
    ASSERT_FALSE(stateChangeEvent.TryWait());
}

//
// ネットワークを構築した状態で CloseAccessPoint() を実行します。
//
TEST(CreateNetwork, DestroyNetworkByCloseAccessPoint)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// ネットワークを構築した状態でネットワーク・インタフェースを解放します。
//
TEST(CreateNetwork, DestroyNetworkByFinalize)
{
    auto* pNetworkInterface = new NetworkInterface(
        g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize);
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワーク・インタフェースを解放します。
    delete pNetworkInterface;
}

//
// AutoChannel でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, AutoChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークの構築を繰り返します。
    const int repeatCountMax = 16;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
            DefaultSsid, nn::ldn::AutoChannel, 2, nullptr, 0));
        ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

        // ネットワークの情報を取得し、値を確認します。
        nn::ldn::detail::NetworkProfile network;
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
        ASSERT_EQ(interface.physicalAddress, network.bssid);
        ASSERT_EQ(DefaultSsid, network.ssid);
        ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

        // ネットワークを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
        ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());
    }

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 無線チャンネルを指定してネットワークを構築できることを確認します。
//
TEST(CreateNetwork, ManualChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークの構築を繰り返します。
    for (int i = 0; i < interface.supportedChannelCount; ++i)
    {
        // ネットワークを構築します。
        int channel = interface.supportedChannels[i];
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
            DefaultSsid, channel, 2, nullptr, 0));
        ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

        // ネットワークの情報を取得し、値を確認します。
        nn::ldn::detail::NetworkProfile network;
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
        ASSERT_EQ(interface.physicalAddress, network.bssid);
        ASSERT_EQ(DefaultSsid, network.ssid);
        ASSERT_EQ(channel, network.channel);

        // ネットワークを破棄します。
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
        ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());
    }

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最小サイズの SSID でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, MinimumSsid)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    const nn::ldn::Ssid ssid = nn::ldn::MakeSsid("a");
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        ssid, nn::ldn::AutoChannel, 1, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(ssid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最大サイズの SSID でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, LargeSsid)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    const nn::ldn::Ssid ssid = nn::ldn::MakeSsid("0123456789abcdef0123456789ABCDEF");
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        ssid, nn::ldn::AutoChannel, 1, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(ssid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最小のノード数でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, NodeCountMin)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 1, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最大のノード数でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, NodeCountMax)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, interface.nodeCountMax, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最小の鍵長でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, KeyLengthMin)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    const char key[] = "0123456789abcdef";
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 2, key, sizeof(key) - 1));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最大の鍵長でネットワークを構築できることを確認します。
//
TEST(CreateNetwork, KeyLengthMax)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    const char key[] = "0123456789abcdef0123456789ABCDEF0123456789abcdef0123456789ABCDEF";
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 2, key, sizeof(key) - 1));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最小のビーコンデータを設定できることを確認します。
//
TEST(SetBeaconData, MinimumDataSize)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 2, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // 最小のビーコンデータを設定します。
    char minimumData;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->SetBeaconData(&minimumData, sizeof(minimumData)));

    // ビーコンデータを削除します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->SetBeaconData(nullptr, 0));

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 最大のビーコンデータを設定できることを確認します。
//
TEST(SetBeaconData, MaximumDataSize)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 2, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // 最大のビーコンデータを設定します。
    char* maximumData = new char[interface.beaconDataSizeMax];
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->SetBeaconData(
        maximumData, interface.beaconDataSizeMax));
    delete[] maximumData;

    // ビーコンデータを削除します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->SetBeaconData(nullptr, 0));

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 誰も接続していない状態で GetStations を実行します。
//
TEST(GetStations, NoStations)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ステーション情報を取得します。
    int stationCount;
    nn::ldn::detail::L2StationInfo stations[nn::ldn::StationCountMax];
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetStations(
        stations, &stationCount, nn::ldn::StationCountMax));
    ASSERT_EQ(0, stationCount);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// アクセスポイントに誰も接続していない状態で GetLinkLevel を実行します。
//
TEST(GetLinkLevel, NoStations)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // LinkLevel を取得します。
    ASSERT_EQ(nn::ldn::LinkLevelMax, pNetworkInterface->GetLinkLevel());

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// 自動チャンネルで既接続スキャンを実行します。
//
TEST(ScanOnConnection, AutoChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // 自動チャンネルスキャンを繰り返します。
    const int repeatCountMax = 16;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        int count;
        NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->Scan(
            g_ScanBuffer, &count, nn::ldn::ScanResultCountMax, nn::ldn::AutoChannel));
        ASSERT_GE(count, 0);
        ASSERT_LE(count, nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(g_ScanBuffer[i].channel));
            ASSERT_LT(g_ScanBuffer[i].rssi, 0);
            ASSERT_LE(g_ScanBuffer[i].dataSize, interface.beaconDataSizeMax);
        }
    }

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}

//
// チャンネルを指定して既接続スキャンを実行します。
//
TEST(ScanOnConnection, ManualChannel)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->OpenAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CreateNetwork(
        DefaultSsid, nn::ldn::AutoChannel, 4, nullptr, 0));
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPointCreated, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile interface;
    pNetworkInterface->GetNetworkInterfaceProfile(&interface);

    // ネットワークの情報を取得し、値を確認します。
    nn::ldn::detail::NetworkProfile network;
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->GetNetworkProfile(&network));
    ASSERT_EQ(interface.physicalAddress, network.bssid);
    ASSERT_EQ(DefaultSsid, network.ssid);
    ASSERT_TRUE(pNetworkInterface->IsAutoSelectableChannel(network.channel));

    // 自動チャンネルスキャンを繰り返します。
    for (int i = 0; i < interface.supportedChannelCount; ++i)
    {
        int channel = interface.supportedChannels[i];

        // TODO: 全チャンネルをテストの対象にする
        if (pNetworkInterface->IsAutoSelectableChannel(channel))
        {
            int count;
            NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->Scan(
                g_ScanBuffer, &count, nn::ldn::ScanResultCountMax, channel));
            ASSERT_GE(count, 0);
            ASSERT_LE(count, nn::ldn::ScanResultCountMax);
            for (int i = 0; i < count; ++i)
            {
                ASSERT_EQ(channel, g_ScanBuffer[i].channel);
                ASSERT_LT(g_ScanBuffer[i].rssi, 0);
                ASSERT_LE(g_ScanBuffer[i].dataSize, interface.beaconDataSizeMax);
            }
        }
    }

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->DestroyNetwork());
    ASSERT_EQ(nn::ldn::detail::L2State_AccessPoint, pNetworkInterface->GetState());

    // アクセスポイントとしての役割を終了します。
    NNT_ASSERT_RESULT_SUCCESS(pNetworkInterface->CloseAccessPoint());
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
}
