﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Advertise/ldn_Aes128CtrSha256Advertise.h>
#include <nn/ldn/detail/Advertise/ldn_PlainSha256Advertise.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>
#include <nnt.h>

namespace
{
    char g_BuilderBuffer[1 * 1024 * 1024];
    char g_ParserBuffer[1 * 1024 * 1024];

    const uint64_t LocalCommunicationId = UINT64_C(0x0123456789ABCDEF);
    const uint16_t SceneId = UINT16_C(0x1234);
    const nn::ldn::IntentId IntentId = nn::ldn::MakeIntentId(LocalCommunicationId, SceneId);
    const nn::ldn::SessionId SessionId = {{
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    }};
    const nn::ldn::NetworkId NetworkId = { IntentId, SessionId };

    static const uint8_t AesKey[] = {
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
    };

    template <typename Helper>
    class AdvertiseBuildParseTest : public ::testing::Test
    {
    protected:

        virtual void SetUp() NN_OVERRIDE
        {
            m_pBuilder = Helper::CreateBuilder();
            m_pParser = Helper::CreateParser();
        }

        virtual void TearDown() NN_OVERRIDE
        {
            delete m_pBuilder;
            delete m_pParser;
        }

    protected:

        nn::ldn::detail::IAdvertiseBuilder* m_pBuilder;
        nn::ldn::detail::IAdvertiseParser* m_pParser;
    };

    struct PlainSha256AdvertiseTestHelper
    {
        typedef nn::ldn::detail::PlainSha256AdvertiseBuilder Builder;
        typedef nn::ldn::detail::PlainSha256AdvertiseParser Parser;
        // static const auto Format = nn::ldn::detail::AdvertiseFormat_PlainSha256;

        static nn::ldn::detail::IAdvertiseBuilder* CreateBuilder() NN_NOEXCEPT
        {
            return new Builder();
        }

        static nn::ldn::detail::IAdvertiseParser* CreateParser() NN_NOEXCEPT
        {
            return new Parser(g_ParserBuffer, Parser::GetRequiredBufferSize());
        }
    };
    typedef ::testing::Types<PlainSha256AdvertiseTestHelper> PlainSha256Advertise;
    //TYPED_TEST_CASE(AdvertiseBuildParseTest, PlainSha256Advertise);

    struct Aes128CtrSha256AdvertiseTestHelper
    {
        typedef nn::ldn::detail::Aes128CtrSha256AdvertiseBuilder Builder;
        typedef nn::ldn::detail::AesCtr128Sha256AdvertiseParser Parser;
        // static const auto Format = nn::ldn::detail::AdvertiseFormat_Aes128CtrSha256;

        static nn::ldn::detail::IAdvertiseBuilder* CreateBuilder() NN_NOEXCEPT
        {
            return new Builder(g_BuilderBuffer, Builder::GetRequiredBufferSize(), AesKey);
        }

        static nn::ldn::detail::IAdvertiseParser* CreateParser() NN_NOEXCEPT
        {
            return new Parser(g_ParserBuffer, Parser::GetRequiredBufferSize(), AesKey);
        }
    };
    typedef ::testing::Types<
        PlainSha256AdvertiseTestHelper,
        Aes128CtrSha256AdvertiseTestHelper> AdvertiseBuildParseTestTypes;
    TYPED_TEST_CASE(AdvertiseBuildParseTest, AdvertiseBuildParseTestTypes);
};

//
// 空の入力に対する署名が仕様と一致することを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, EmptyData)
{
    // アドバータイズデータを構築します。
    char advertise[nn::ldn::detail::AdvertiseSizeMin];
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(advertise, advertiseSize));
    ASSERT_EQ(NetworkId, parser.GetNetworkId());
    ASSERT_EQ(nn::ldn::detail::CurrentVersion, parser.GetVersion());
    ASSERT_EQ(0U, parser.GetDataSize());
}

//
// 最小の入力に対する署名が仕様と一致することを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, MinimumSizeData)
{
    // アドバータイズデータを構築します。
    char data = 'a';
    char advertise[nn::ldn::detail::AdvertiseSizeMin + sizeof(data)];
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(advertise, &advertiseSize, sizeof(advertise), &data, sizeof(data));
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin + sizeof(data), advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(advertise, advertiseSize));
    ASSERT_EQ(NetworkId, parser.GetNetworkId());
    ASSERT_EQ(nn::ldn::detail::CurrentVersion, parser.GetVersion());
    ASSERT_EQ(sizeof(data), parser.GetDataSize());
    ASSERT_EQ(data, *static_cast<const char*>(parser.GetData()));
}

//
// 最大の入力に対する署名が仕様と一致することを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, MaximumSizeData)
{
    // データを生成します。
    uint8_t data[nn::ldn::detail::AdvertiseBodySizeMax];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(0xA5 + i);
    }

    // アドバータイズデータを構築します。
    char advertise[nn::ldn::detail::AdvertiseSizeMax];
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(advertise, &advertiseSize, sizeof(advertise), data, sizeof(data));
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMax, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(advertise, advertiseSize));
    ASSERT_EQ(NetworkId, parser.GetNetworkId());
    ASSERT_EQ(nn::ldn::detail::CurrentVersion, parser.GetVersion());
    ASSERT_EQ(sizeof(data), parser.GetDataSize());
    ASSERT_EQ(0, std::memcmp(data, parser.GetData(), sizeof(data)));
}

//
// カウンタ値を利用してキャッシュが働くことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, CounterCache)
{
    nn::ldn::detail::Advertise advertise;

    // アドバータイズデータを構築します。
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // 同じアドバータイズを 2 回解析すると失敗します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise, advertiseSize));
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_NotUpdated,
        parser.Parse(&advertise, advertiseSize));

    // カウンタを 1 つ進めたアドバータイズの受信には成功します。
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise, advertiseSize));

    // カウンタ値を進めたアドバータイズは一定範囲内であれば正常に受信できます。
    for (int i = 0; i < nn::ldn::detail::AdvertiseRejectCounterThreshold - 1; ++i)
    {
        builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
        ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);
    }
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise, advertiseSize));
}

//
// 小さすぎるアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, TooSmall)
{
    // アドバータイズデータを構築します。
    char advertise[nn::ldn::detail::AdvertiseSizeMin];
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InvalidAdvertise,
        parser.Parse(advertise, advertiseSize - 1));
}

//
// 大きすぎるアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, TooLarge)
{
    // アドバータイズデータを構築します。
    char advertise[nn::ldn::detail::AdvertiseSizeMin];
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InvalidAdvertise,
        parser.Parse(advertise, nn::ldn::detail::AdvertiseSizeMax + 1));
}

//
// 未知のフォーマットを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, OtherFormat)
{
    // アドバータイズデータを構築します。
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    advertise.header.format = nn::ldn::detail::AdvertiseFormat_None;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InvalidAdvertise,
        parser.Parse(&advertise, advertiseSize));
}

//
// Body のサイズが最大値を超えているアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, InconsistentBodyLength)
{
    // アドバータイズデータを構築します。
    char data = 'a';
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), &data, sizeof(data));
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin + 1, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    advertise.header.length = nn::ldn::detail::ConvertToNetworkByteOrder(
        static_cast<uint16_t>(sizeof(data) + 1));
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InvalidAdvertise,
        parser.Parse(&advertise, advertiseSize));
    advertise.header.length = nn::ldn::detail::ConvertToNetworkByteOrder(
        static_cast<uint16_t>(sizeof(data) - 1));
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InvalidAdvertise,
        parser.Parse(&advertise, advertiseSize));
}

//
// 互換性の無いバージョンのアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, IncompatibleVersion)
{
    // アドバータイズデータを構築します。
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    advertise.header.version = nn::ldn::detail::MakeVersion(0, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_IncompatibleVersion,
        parser.Parse(&advertise, advertiseSize));
}

//
// NetworkID が変化したアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, InconsistentNetworkId)
{
    // アドバータイズデータを構築します。
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // 一度アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise, advertiseSize));

    // SessionID を変化させてからもう一度アドバータイズデータを解析します。
    nn::ldn::NetworkId networkId;
    networkId = NetworkId;
    networkId.sessionId.random[15] ^= 1;
    builder.SetNetworkId(networkId);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InconsistentUpdate,
        parser.Parse(&advertise, advertiseSize));

    // IntentID を変化させてからもう一度アドバータイズデータを解析します。
    networkId = NetworkId;
    networkId.intentId.localCommunicationId ^= 1;
    builder.SetNetworkId(networkId);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InconsistentUpdate,
        parser.Parse(&advertise, advertiseSize));
}

//
// バージョンが変化したアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, InconsistentVersion)
{
    // アドバータイズデータを構築します。
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // 一度アドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise, advertiseSize));

    // Version を変化させてからもう一度アドバータイズデータを解析します。
    nn::ldn::detail::Version version;
    version= nn::ldn::detail::MakeVersion(
        nn::ldn::detail::CurrentMajorVersion, nn::ldn::detail::CurrentMinorVersion + 1);
    builder.SetVersion(version);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InconsistentUpdate,
        parser.Parse(&advertise, advertiseSize));
}

//
// カウンタ値が離れたアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, InconsistentCounter)
{
    nn::ldn::detail::Advertise advertise1;
    nn::ldn::detail::Advertise advertise2;
    nn::ldn::detail::Advertise advertise3;

    // アドバータイズデータを構築します。
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise1, &advertiseSize, sizeof(advertise1), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // もう 1 つアドバータイズデータを構築します。
    builder.Build(&advertise2, &advertiseSize, sizeof(advertise2), nullptr, 0);
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);

    // 2 つ目のアドバータイズを解析してから 1 つ目のアドバータイズを解析すると失敗します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_Success,
        parser.Parse(&advertise2, advertiseSize));
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InconsistentUpdate,
        parser.Parse(&advertise1, advertiseSize));

    // カウンタ値を進めます。
    for (int i = 0; i < nn::ldn::detail::AdvertiseRejectCounterThreshold; ++i)
    {
        builder.Build(&advertise3, &advertiseSize, sizeof(advertise3), nullptr, 0);
        ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMin, advertiseSize);
    }

    // カウンタ値が大きく進んだアドバータイズを解析すると失敗します。
    ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_InconsistentUpdate,
        parser.Parse(&advertise3, advertiseSize));
}

//
// 署名が一致しないアドバータイズを Parse できないことを確認します。
//
TYPED_TEST(AdvertiseBuildParseTest, InvalidSign)
{
    // データを生成します。
    uint8_t data[nn::ldn::detail::AdvertiseBodySizeMax];
    for (size_t i = 0; i < sizeof(data); ++i)
    {
        data[i] = static_cast<uint8_t>(0xA5 + i);
    }

    // アドバータイズデータを構築します。
    nn::ldn::detail::Advertise advertise;
    size_t advertiseSize;
    nn::ldn::detail::IAdvertiseBuilder& builder = *this->m_pBuilder;
    builder.SetNetworkId(NetworkId);
    builder.SetVersion(nn::ldn::detail::CurrentVersion);
    builder.Build(&advertise, &advertiseSize, sizeof(advertise), data, sizeof(data));
    ASSERT_EQ(nn::ldn::detail::AdvertiseSizeMax, advertiseSize);

    // 署名の先頭を書き換えてアドバータイズデータを解析します。
    nn::ldn::detail::IAdvertiseParser& parser = *this->m_pParser;
    {
        nn::ldn::detail::Advertise tamperedAdvertise = advertise;
        tamperedAdvertise.digest[0] ^= 1;
        ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_SignVerificationError,
            parser.Parse(&tamperedAdvertise, advertiseSize));
    }

    // 署名の末尾を書き換えてアドバータイズデータを解析します。
    {
        nn::ldn::detail::Advertise tamperedAdvertise = advertise;
        tamperedAdvertise.digest[nn::ldn::detail::AdvertiseDigestSize - 1] ^= 1;
        ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_SignVerificationError,
            parser.Parse(&tamperedAdvertise, advertiseSize));
    }

    // ペイロードの先頭を書き換えてアドバータイズデータを解析します。
    {
        nn::ldn::detail::Advertise tamperedAdvertise = advertise;
        tamperedAdvertise.body[0] ^= 1;
        ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_SignVerificationError,
            parser.Parse(&tamperedAdvertise, advertiseSize));
    }

    // ペイロードの末尾を書き換えてアドバータイズデータを解析します。
    {
        nn::ldn::detail::Advertise tamperedAdvertise = advertise;
        tamperedAdvertise.body[nn::ldn::detail::AdvertiseBodySizeMax - 1] ^= 1;
        ASSERT_EQ(nn::ldn::detail::AdvertiseParserResult_SignVerificationError,
            parser.Parse(&tamperedAdvertise, advertiseSize));
    }
}
