﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * 1 台の開発機だけで実行できるテストです。
 */

#include <nn/ldn.h>
#include <nn/ldn/ldn_MonitorApi.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/os.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_Config.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>

namespace
{
    // テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    const nn::Bit64 LocalId = UINT64_C(0x0005000c10000000);
    const nn::Bit16 SceneId = UINT16_C(0x1100);
    const nn::ldn::IntentId IntentId = nn::ldn::MakeIntentId(LocalId, SceneId);
    const char Passphrase[] = "LdnTestPassphrase";
    const size_t PassphraseSize = std::strlen(Passphrase);
    const char UserName[] = "TestAp";

    nn::ldn::NetworkInfo g_Networks[nn::ldn::ScanResultCountMax];

    void CreateRandom(void* data, size_t dataSize) NN_NOEXCEPT
    {
        nn::Bit8* binary = static_cast<nn::Bit8*>(data);
        for (size_t i = 0; i < dataSize; ++i)
        {
            binary[i] = static_cast<nn::Bit8>(nn::os::GetSystemTick().GetInt64Value() >> 1);
        }
    }

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetWirelessControllerRestriction(
            static_cast<nn::ldn::WirelessControllerRestriction>(
                g_TestConfig.wirelessControllerRestriction)));
    }

} // namespace <unnamed>

//
// Initialize/Finalize を繰り返し実行します。
//
TEST(Initialize, MultipleTimes)
{
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    const int repeatCount = 32;
    for (int i = 0; i < repeatCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Initialize());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        nn::ldn::Finalize();
        ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    }
}

//
// InitializeSystem/FinalizeSystem を繰り返し実行します。
//
TEST(InitializeSystem, MultipleTimes)
{
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    const int repeatCount = 32;
    for (int i = 0; i < repeatCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        nn::ldn::FinalizeSystem();
        ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    }
}

//
// InitializeMonitor/FinalizeMonitor を繰り返し実行します。
//
TEST(InitializeMonitor, MultipleTimes)
{
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    const int repeatCount = 32;
    for (int i = 0; i < repeatCount; ++i)
    {
        nn::ldn::InitializeMonitor();
        nn::ldn::FinalizeMonitor();
    }
}

//
// OpenAccessPoint/CloseAccessPoint を繰り返し実行します。
//
TEST(OpenAccessPoint, MultipleTimes)
{
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    const int repeatCount = 32;
    for (int i = 0; i < repeatCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }
}

//
// OpenAccessPoint や CloseAccessPoint でイベントがシグナルされないことを検証します。
//
TEST(OpenAccessPoint, NotSignaled)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    nn::os::SystemEventType stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

    // OpenAccessPoint 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // CloseAccessPoint 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 取得したイベントを削除します。
    nn::os::DestroySystemEvent(&stateChangeEvent);
}

//
// OpenAccessPoint の後、 CloseAccessPoint をとばして Finalize します。
//
TEST(OpenAccessPoint, CloseAccessPointByFinalize)
{
    // LDN ライブラリを初期化します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    SetLdnSettings();
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());

    // OpenAccessPoint 関数でアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // CloseAccessPoint を実行せず、 Finalize でステーションとしての動作を停止します。
    nn::ldn::FinalizeSystem();
}

//
// CreateNetwork や Destroynetwork でイベントがシグナルされることを検証します。
//
TEST(CreateNetwork, Signaled)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    nn::os::SystemEventType stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(&stateChangeEvent);
    SetLdnSettings();

    // アクセスポイントとしての動作を開始します。
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 適当なパラメータでネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // AutoReset の設定でイベントがシグナルされていることを検証します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // AutoReset の設定でイベントがシグナルされていることを検証します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
}

//
// 最小のノード数でネットワークを構築します。
//
TEST(CreateNetwork, NodeCountMin)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 1;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCountMax);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大のノード数でネットワークを構築します。
//
TEST(CreateNetwork, NodeCountMax)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = nn::ldn::NodeCountMax;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::NodeCountMax, g_Networks[0].ldn.nodeCountMax);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最小の鍵長でネットワークを構築します。
//
TEST(CreateNetwork, PassphraseLengthMin)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大の鍵長でネットワークを構築します。
//
TEST(CreateNetwork, PassphraseLengthMax)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMax;
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最小のローカル通信バージョンでネットワークを構築します。
//
TEST(CreateNetwork, LocalCommunicationVersionMin)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = nn::ldn::LocalCommunicationVersionMin;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMax;
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ローカル通信バージョンを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::LocalCommunicationVersionMin,
              g_Networks[0].ldn.nodes[0].localCommunicationVersion);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大のローカル通信バージョンでネットワークを構築します。
//
TEST(CreateNetwork, LocalCommunicationVersionMax)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = nn::ldn::LocalCommunicationVersionMax;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMax;
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ローカル通信バージョンを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::LocalCommunicationVersionMax,
              g_Networks[0].ldn.nodes[0].localCommunicationVersion);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// AutoChannel で 1, 6, 11 チャンネルにネットワークが展開されることを確認します。
//
TEST(CreateNetwork, AutoChannel)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークの構築・破壊を繰り返しながら無線チャンネルの検証を行います。
    const int repeatCount = 16;
    for (int i = 0; i < repeatCount; ++i)
    {
        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // 無線チャンネルが 1, 6, 11 のいずれかであることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        const int channel = g_Networks[0].common.channel;
        ASSERT_TRUE(channel == 1 || channel == 6 || channel == 11);

        // ネットワークを破壊します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// 指定したチャンネルにネットワークが展開されることを確認します。
//
TEST(CreateNetwork, ManualChannel)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークの構築・破壊を繰り返しながら無線チャンネルの検証を行います。
    const int channels[] = { 1, 6, 11, 36, 40, 44, 48 };
    for (int channel : channels)
    {
        // ネットワークを構築します。
        network.channel = static_cast<int16_t>(channel);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // 指定された無線チャンネルにネットワークが展開されていることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(channel, g_Networks[0].common.channel);

        // ネットワークを破壊します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// DestroyNetwork を飛ばして CloseAccessPoint でネットワークを遮断します。
//
TEST(CreateNetwork, DestroyNetworkByCloseAccessPoint)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // CloseAccessPoint でネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
    nn::ldn::FinalizeSystem();
}

//
// DestroyNetwork を飛ばして Finalize でネットワークを遮断します。
//
TEST(CreateNetwork, DestroyNetworkByFinalize)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // Finalize でネットワークを破棄します。
    nn::ldn::FinalizeSystem();
}

//
// CreateNetworkPrivate でイベントがシグナルされることを検証します。
//
TEST(CreateNetworkPrivate, Signaled)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    nn::os::SystemEventType stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

    // アクセスポイントとしての動作を開始します。
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 適当なパラメータでネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // AutoReset の設定でイベントがシグナルされていることを検証します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // AutoReset の設定でイベントがシグナルされていることを検証します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
}

//
// 最小のノード数でネットワークを構築します。
//
TEST(CreateNetworkPrivate, NodeCountMin)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 1;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCountMax);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大のノード数でネットワークを構築します。
//
TEST(CreateNetworkPrivate, NodeCountMax)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = nn::ldn::NodeCountMax;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::NodeCountMax, g_Networks[0].ldn.nodeCountMax);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最小の鍵長でネットワークを構築します。
//
TEST(CreateNetworkPrivate, PassphraseLengthMin)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大の鍵長でネットワークを構築します。
//
TEST(CreateNetworkPrivate, PassphraseLengthMax)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    security.passphraseSize = nn::ldn::PassphraseSizeMax;
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最小の通信バージョンでネットワークを構築します。
//
TEST(CreateNetworkPrivate, LocalCommunicationVersionMin)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = nn::ldn::NodeCountMax;
    network.localCommunicationVersion = nn::ldn::LocalCommunicationVersionMin;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::LocalCommunicationVersionMin,
              g_Networks[0].ldn.nodes[0].localCommunicationVersion);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// 最大の通信バージョンでネットワークを構築します。
//
TEST(CreateNetworkPrivate, LocalCommunicationVersionMax)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = nn::ldn::NodeCountMax;
    network.localCommunicationVersion = nn::ldn::LocalCommunicationVersionMax;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // ノード数を確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(nn::ldn::LocalCommunicationVersionMax,
              g_Networks[0].ldn.nodes[0].localCommunicationVersion);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);

    // ネットワークを破壊します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// AutoChannel で 1, 6, 11 チャンネルにネットワークが展開されることを確認します。
//
TEST(CreateNetworkPrivate, AutoChannel)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークの構築・破壊を繰り返しながら無線チャンネルの検証を行います。
    const int repeatCount = 16;
    for (int i = 0; i < repeatCount; ++i)
    {
        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
            network, security, securityParam, user, 0, nullptr));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // 無線チャンネルが 1, 6, 11 のいずれかであることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        const int channel = g_Networks[0].common.channel;
        ASSERT_TRUE(channel == 1 || channel == 6 || channel == 11);

        // ネットワークを破壊します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// 指定したチャンネルにネットワークが展開されることを確認します。
//
TEST(CreateNetworkPrivate, ManualChannel)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークの構築・破壊を繰り返しながら無線チャンネルの検証を行います。
    const int channels[] = { 1, 6, 11, 36, 40, 44, 48 };
    for (int channel : channels)
    {
        // ネットワークを構築します。
        network.channel = static_cast<int16_t>(channel);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
            network, security, securityParam, user, 0, nullptr));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // 指定された無線チャンネルにネットワークが展開されていることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(channel, g_Networks[0].common.channel);

        // ネットワークを破壊します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// DestroyNetwork を飛ばして CloseAccessPoint でネットワークを遮断します。
//
TEST(CreateNetworkPrivate, DestroyNetworkByCloseAccessPoint)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // CloseAccessPoint でネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
    nn::ldn::FinalizeSystem();
}

//
// DestroyNetwork を飛ばして Finalize でネットワークを遮断します。
//
TEST(CreateNetworkPrivate, DestroyNetworkByFinalize)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter securityParam = { };

    // ネットワークを構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        network, security, securityParam, user, 0, nullptr));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // Finalize でネットワークを破棄します。
    nn::ldn::FinalizeSystem();
}

//
// ネットワークの再構築でパラメータを引き継ぎできることを検証します。
//
TEST(CreateNetworkPrivate, Resume)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 適当なパラメータを使用してネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // 現在のネットワークに対するセキュリティ・パラメータとネットワーク情報を取得します。
    nn::ldn::SecurityParameter securityParam;
    nn::ldn::NetworkConfig networkConfig;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetSecurityParameter(&securityParam));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkConfig(&networkConfig));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(network.intentId, networkConfig.intentId);
    ASSERT_EQ(network.nodeCountMax, networkConfig.nodeCountMax);
    ASSERT_EQ(network.localCommunicationVersion, networkConfig.localCommunicationVersion);
    ASSERT_NE(nn::ldn::AutoChannel, networkConfig.channel);
    ASSERT_EQ(g_Networks[0].networkId.sessionId, securityParam._sessionId);
    ASSERT_TRUE(std::memcmp(g_Networks[0].ldn.serverRandom, securityParam._serverRandom,
                            sizeof(securityParam._serverRandom)) == 0);

    // IPv4 アドレスを再利用するためテーブルを保存しておきます。
    nn::ldn::AddressEntry entry;
    entry.ipv4Address = g_Networks[0].ldn.nodes[0].ipv4Address;
    entry.macAddress = g_Networks[0].ldn.nodes[0].macAddress;

    // ネットワークを一旦破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワークを再構築します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        networkConfig, security, securityParam, user, 1, &entry));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // 再構築されたネットワークのパラメータを検証します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
    ASSERT_EQ(networkConfig.channel, g_Networks[0].common.channel);
    ASSERT_EQ(IntentId, g_Networks[0].networkId.intentId);
    ASSERT_EQ(securityParam._sessionId, g_Networks[0].networkId.sessionId);
    ASSERT_TRUE(std::memcmp(securityParam._serverRandom, g_Networks[0].ldn.serverRandom,
                            sizeof(securityParam._serverRandom)) == 0);
    ASSERT_EQ(4, g_Networks[0].ldn.nodeCountMax);
    ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);
    ASSERT_EQ(entry.ipv4Address, g_Networks[0].ldn.nodes[0].ipv4Address);

    // ネットワークを破棄します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
}

//
// ネットワーク構築前に Advertise を設定できることを確認します。
//
TEST(SetAdvertiseData, StateAccessPoint)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // Advertise で配信するデータを定義します。
    char smallData = 'a';
    char largeData[nn::ldn::AdvertiseDataSizeMax];
    CreateRandom(largeData, sizeof(largeData));
    void* data[] = { &smallData, largeData, nullptr };
    size_t dataSize[] = { sizeof(smallData), sizeof(largeData), 0U };
    const int dataCount = static_cast<int>(sizeof(data) / sizeof(data[0]));

    // ネットワークの構築前に Advertise データを登録します。
    for (int i = 0; i < dataCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetAdvertiseData(data[i], dataSize[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(dataSize[i], g_Networks[0].ldn.advertiseDataSize);
        ASSERT_EQ(0, std::memcmp(data[i], g_Networks[0].ldn.advertiseData, dataSize[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// ネットワーク構築後に Advertise を設定できることを確認します。
//
TEST(SetAdvertiseData, StateAccessPointCreated)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // Advertise で配信するデータを定義します。
    char smallData = 'a';
    char largeData[nn::ldn::AdvertiseDataSizeMax];
    CreateRandom(largeData, sizeof(largeData));
    void* data[] = {
        nullptr,        // 初期状態
        &smallData,
        largeData,
        nullptr
    };
    size_t dataSize[] = { 0U, sizeof(smallData), sizeof(largeData), 0U };
    const int dataCount = static_cast<int>(sizeof(data) / sizeof(data[0]));

    // ネットワークの構築後に Advertise データを登録します。
    for (int i = 1; i < dataCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(dataSize[i - 1], g_Networks[0].ldn.advertiseDataSize);
        ASSERT_EQ(0, std::memcmp(data[i - 1], g_Networks[0].ldn.advertiseData, dataSize[i - 1]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetAdvertiseData(data[i], dataSize[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(dataSize[i], g_Networks[0].ldn.advertiseDataSize);
        ASSERT_EQ(0, std::memcmp(data[i], g_Networks[0].ldn.advertiseData, dataSize[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// ネットワーク構築前に StationAcceptPolicy を設定できることを確認します。
//
TEST(SetStationAcceptPolicy, StateAccessPoint)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ポリシーのリストを定義します。
    const nn::ldn::AcceptPolicy policies[] = {
        nn::ldn::AcceptPolicy_WhiteList,
        nn::ldn::AcceptPolicy_BlackList,
        nn::ldn::AcceptPolicy_AlwaysReject,
        nn::ldn::AcceptPolicy_AlwaysAccept,
    };
    const int policyCount = static_cast<int>(sizeof(policies) / sizeof(policies[0]));

    // ネットワークの構築前に StationAcceptPlicy を登録します。
    for (int i = 0; i < policyCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(policies[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(policies[i], g_Networks[0].ldn.stationAcceptPolicy);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// ネットワーク構築後に StationAcceptPolicy を設定できることを確認します。
//
TEST(SetStationAcceptPolicy, StateAccessPointCreated)
{
    // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // ネットワーク構築に使用する適当なパラメータを生成します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ポリシーのリストを定義します。
    const nn::ldn::AcceptPolicy policies[] = {
        nn::ldn::AcceptPolicy_AlwaysAccept, // 初期状態
        nn::ldn::AcceptPolicy_WhiteList,
        nn::ldn::AcceptPolicy_BlackList,
        nn::ldn::AcceptPolicy_AlwaysReject,
        nn::ldn::AcceptPolicy_AlwaysAccept,
    };
    const int policyCount = static_cast<int>(sizeof(policies) / sizeof(policies[0]));

    // ネットワークの構築後に StationAcceptPlicy を登録します。
    for (int i = 1; i < policyCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(policies[i - 1], g_Networks[0].ldn.stationAcceptPolicy);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetStationAcceptPolicy(policies[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(g_Networks));
        ASSERT_EQ(policies[i], g_Networks[0].ldn.stationAcceptPolicy);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// OpenStation/CloseStation を繰り返し実行します。
//
TEST(OpenStation, MultipleTimes)
{
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    const int repeatCount = 32;
    for (int i = 0; i < repeatCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }
}

//
// OpenStation や CloseStation でイベントがシグナルされないことを検証します。
//
TEST(OpenStation, NotSignaled)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    nn::os::SystemEventType stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

    // OpenStation 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // CloseStation 関数でイベントがシグナルされないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 取得したイベントを削除します。
    nn::os::DestroySystemEvent(&stateChangeEvent);
}

//
// OpenStation の後、 CloseStation をとばして Finalize します。
//
TEST(OpenStation, CloseStationByFinalize)
{
    // LDN ライブラリを初期化してイベントを取得します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // OpenStation 関数でステーションとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // CloseStation を実行せず、 Finalize でステーションとしての動作を停止します。
    nn::ldn::FinalizeSystem();
}

//
// Station の接続前のスキャンです。
// 最小・最大のバッファサイズで実行して成功することを確認します。
//
TEST(Scan, BufferSizeMinMax)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // 最小のバッファサイズでスキャンを実行します。
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, 1, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= 1);

    // 最大のバッファサイズでスキャンを実行します。
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// 全てのチャンネルでスキャンに成功することを確認します。
//
TEST(Scan, ManualChannel)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { };
    int count;
    int channels[] = { 1, 6, 11, 36, 40, 44, 48 };
    for (int channel : channels)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, channel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_EQ(channel, g_Networks[i].common.channel);
        }
    }
}

//
// Station の接続前のスキャンです。
// 自動チャンネル選択でスキャンに成功することを確認します。
//
TEST(Scan, AutoChannel)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { };
    int count;
    for (int i = 0; i < 16; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int j = 0; j < count; ++j)
        {
            int ch = g_Networks[j].common.channel;
            ASSERT_TRUE(ch == 1 || ch == 6 || ch == 11);
        }
    }
}

//
// Station の接続前のスキャンです。
// SSID フィルタを使用できることを確認します。
//
TEST(Scan, SsidFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // SSID をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_Ssid;
    filter.ssid = nn::ldn::MakeSsid(" !\"#$%&'()*+,-./0123456789:;<=>");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
    filter.ssid = nn::ldn::MakeSsid("?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
    filter.ssid = nn::ldn::MakeSsid("_`abcdefghijklmnopqrstuvwxyz{|}~");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);

    // SSID のフィルタを有効に設定していない場合に、
    // 無効な SSID を Scan() に渡してもアサートが発生しないことを確認します。
    filter.ssid.length = 33;
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// NetworkType フィルタを使用できることを確認します。
//
TEST(Scan, NetworkTypeFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // 全ての NetworkType を使用してスキャンできることを確認します。
    nn::ldn::NetworkType types[] = { nn::ldn::NetworkType_General, nn::ldn::NetworkType_Ldn };
    for (auto type : types)
    {
        filter.networkType = static_cast<nn::Bit32>(type);
        filter.flag = nn::ldn::ScanFilterFlag_NetworkType;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_EQ(type, g_Networks[i].common.networkType);
        }
    }

    // NetworkType のフィルタを有効に設定していない場合に、
    // 無効な NetworkType を Scan() に渡してもアサートが発生しないことを確認します。
    filter.networkType = static_cast<nn::Bit32>(1 << 2);
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// NetworkId のフィルタを使用できることを確認します。
//
TEST(Scan, NetworkIdFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    filter.networkId.intentId = IntentId;
    int count;

    // ローカル通信識別子をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_LocalCommunicationId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(
            IntentId.localCommunicationId,
            g_Networks[i].networkId.intentId.localCommunicationId);
    }

    // シーン識別子をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_SceneId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(
            IntentId.sceneId, g_Networks[i].networkId.intentId.sceneId);
    }

    // IntentID 全体をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_IntentId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(IntentId, g_Networks[i].networkId.intentId);
    }

    // SessionID をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_SessionId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);

    // NetworkID 全体をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_NetworkId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
}

//
// Station の接続前のスキャンです。
// 最小・最大のバッファサイズで実行して成功することを確認します。
//
TEST(ScanPrivate, BufferSizeMinMax)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // 最小のバッファサイズでスキャンを実行します。
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, 1, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= 1);

    // 最大のバッファサイズでスキャンを実行します。
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// 全てのチャンネルでスキャンに成功することを確認します。
//
TEST(ScanPrivate, ManualChannel)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { };
    int count;
    int channels[] = { 1, 6, 11, 36, 40, 44, 48 };
    for (int channel : channels)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, channel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_EQ(channel, g_Networks[i].common.channel);
        }
    }
}

//
// Station の接続前のスキャンです。
// 自動チャンネル選択でスキャンに成功することを確認します。
//
TEST(ScanPrivate, AutoChannel)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { };
    int count;
    for (int i = 0; i < 16; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int j = 0; j < count; ++j)
        {
            int ch = g_Networks[j].common.channel;
            ASSERT_TRUE(ch == 1 || ch == 6 || ch == 11);
        }
    }
}

//
// Station の接続前のスキャンです。
// SSID フィルタを使用できることを確認します。
//
TEST(ScanPrivate, SsidFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // SSID をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_Ssid;
    filter.ssid = nn::ldn::MakeSsid(" !\"#$%&'()*+,-./0123456789:;<=>");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
    filter.ssid = nn::ldn::MakeSsid("?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
    filter.ssid = nn::ldn::MakeSsid("_`abcdefghijklmnopqrstuvwxyz{|}~");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);

    // SSID のフィルタを有効に設定していない場合に、
    // 無効な SSID を Scan() に渡してもアサートが発生しないことを確認します。
    filter.ssid.length = 33;
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// BSSID フィルタを使用できることを確認します。
//
TEST(ScanPrivate, BssidFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // BSSID をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_Bssid;
    filter.bssid = nn::ldn::MakeMacAddress(0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);

    // BSSID のフィルタを有効に設定していない場合に、
    // 無効な BSSID を Scan() に渡してもアサートが発生しないことを確認します。
    filter.flag = 0;
    filter.bssid = nn::ldn::ZeroMacAddress;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    filter.bssid = nn::ldn::BroadcastMacAddress;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// NetworkType フィルタを使用できることを確認します。
//
TEST(ScanPrivate, NetworkTypeFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    int count;

    // 全ての NetworkType を使用してスキャンできることを確認します。
    nn::ldn::NetworkType types[] = { nn::ldn::NetworkType_General, nn::ldn::NetworkType_Ldn };
    for (auto type : types)
    {
        filter.networkType = static_cast<nn::Bit32>(type);
        filter.flag = nn::ldn::ScanFilterFlag_NetworkType;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
        ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
        for (int i = 0; i < count; ++i)
        {
            ASSERT_EQ(type, g_Networks[i].common.networkType);
        }
    }

    // NetworkType のフィルタを有効に設定していない場合に、
    // 無効な NetworkType を Scan() に渡してもアサートが発生しないことを確認します。
    filter.networkType = static_cast<nn::Bit32>(1 << 2);
    filter.flag = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
}

//
// Station の接続前のスキャンです。
// NetworkId のフィルタを使用できることを確認します。
//
TEST(ScanPrivate, NetworkIdFilter)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter;
    filter.networkId.intentId = IntentId;
    int count;

    // ローカル通信識別子をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_LocalCommunicationId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(
            IntentId.localCommunicationId,
            g_Networks[i].networkId.intentId.localCommunicationId);
    }

    // シーン識別子をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_SceneId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(
            IntentId.sceneId, g_Networks[i].networkId.intentId.sceneId);
    }

    // IntentID 全体をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_IntentId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_TRUE(0 <= count && count <= nn::ldn::ScanResultCountMax);
    for (int i = 0; i < count; ++i)
    {
        ASSERT_EQ(IntentId, g_Networks[i].networkId.intentId);
    }

    // SessionID をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_SessionId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);

    // NetworkID 全体をフィルタに設定してスキャンします。
    filter.flag = nn::ldn::ScanFilterFlag_NetworkId;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel));
    ASSERT_EQ(0, count);
}

//
// Monitor の機能を使用して初期化前後の状態を確認します。
//
TEST(Monitor, State_Initialized)
{
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::MonitorInitializer monitorInitializer;

    // 監視モードで初期化しても、 LDN ライブラリがローカル通信を開始していることは無いので、
    // nn::ldn::Initialize 関数を実行した場合とは異なり State_None のまま変化しません。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());

    // LDN ライブラリでローカル通信を開始していないため、失敗します。
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
        nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
        nn::ldn::GetIpv4AddressForMonitor(&address, &mask));

    // nn::ldn::Initialize 関数で初期化すると State_Initialized に遷移します。
    {
        nnt::ldn::Initializer initializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
    }
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());

    // nn::ldn::InitializeSystem 関数で初期化すると State_Initialized に遷移します。
    {
        nnt::ldn::SystemInitializer initializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        SetLdnSettings();
    }
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());
}

//
// Monitor の機能を使用して OpenAccessPoint 前後の状態を確認します。
//
TEST(Monitor, StateAccessPoint)
{
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;

    // nn::ldn::Initialize() 前に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());
        nnt::ldn::SystemInitializer initializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        SetLdnSettings();
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // アクセスポイントとしての動作を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetStateForMonitor());

        // ネットワークを構築していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));

        // アクセスポイントとしての動作を終了します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());

        // ネットワークを構築していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
    }

    // nn::ldn::Initialize() 後に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // アクセスポイントとしての動作を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetStateForMonitor());

        // ネットワークを構築していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));

        // アクセスポイントとしての動作を終了します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());

        // ネットワークを構築していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
    }
}

//
// Monitor の機能を使用してネットワーク構築後の状態を確認します。
//
TEST(Monitor, StateAccessPointCreated)
{
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;

    // ネットワーク構築に仕様する適当なパラメータを定義します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // nn::ldn::Initialize() 前に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());
        nnt::ldn::SystemInitializer initializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        SetLdnSettings();
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // アクセスポイントとしての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetStateForMonitor());

        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetStateForMonitor());

        // ネットワークの情報を正しく取得できます。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfoForMonitor(g_Networks));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&address, &mask));
        ASSERT_EQ(IntentId, g_Networks[0].networkId.intentId);
        ASSERT_EQ(network.nodeCountMax, g_Networks[0].ldn.nodeCountMax);
        ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);
    }

    // nn::ldn::Initialize() 後に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // アクセスポイントとしての動作を開始します。
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetStateForMonitor());

        // ネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetStateForMonitor());

        // ネットワークの情報を正しく取得できます。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfoForMonitor(g_Networks));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&address, &mask));
        ASSERT_EQ(IntentId, g_Networks[0].networkId.intentId);
        ASSERT_EQ(network.nodeCountMax, g_Networks[0].ldn.nodeCountMax);
        ASSERT_EQ(1, g_Networks[0].ldn.nodeCount);
    }
}

//
// Monitor の機能を使用して OpenStation 前後の状態を確認します。
//
TEST(Monitor, StateStation)
{
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;

    // nn::ldn::Initialize() 前に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetStateForMonitor());
        nnt::ldn::SystemInitializer initializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        SetLdnSettings();
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // ステーションとしての動作を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetStateForMonitor());

        // ネットワークに接続していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));

        // ステーションとしての動作を終了します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());

        // ネットワークに接続していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
    }

    // nn::ldn::Initialize() 後に nn::ldn::InitializeMonitor() を実行し、通信状態を監視します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::SystemInitializer initializer;
        nnt::ldn::MonitorInitializer monitorInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);
        SetLdnSettings();

        // ステーションとしての動作を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetStateForMonitor());

        // ネットワークに接続していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));

        // ステーションとしての動作を終了します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetStateForMonitor());

        // ネットワークに接続していないため、失敗します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfoForMonitor(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
    }
}

//
// LDN ライブラリの初期化直後、別の権限を使ってローカル通信の使用権取得を試行します。
//
TEST(Occupy, StateInitialized)
{
    // User 権限で初期化したローカル通信を System 権限で奪い取ります。
    // Initialize, InitializeSystem, FinalizeSystem, Finalize
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Initialize());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    nn::ldn::FinalizeSystem();
    ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    nn::ldn::Finalize();
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());

    // User 権限で初期化したローカル通信を System 権限で奪い取ります。
    // Initialize, InitializeSystem, Finalize, FinalizeSystem
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Initialize());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    nn::ldn::Finalize();
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    nn::ldn::FinalizeSystem();
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());

    // System 権限で初期化したローカル通信は User 権限では奪い取ることはできません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultDeviceOccupied, nn::ldn::Initialize());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    nn::ldn::FinalizeSystem();
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
}

//
// OpenAccessPoint の後、通信デバイスの占有権を剥奪します。
//
TEST(Occupy, StateAccessPoint)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
        nnt::ldn::Initializer initializer;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        // 通信デバイスの占有権を剥奪します。
        nnt::ldn::SystemInitializer systemInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        SetLdnSettings();
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::Initializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
}

//
// CreateNetwork の後、通信デバイスの占有権を剥奪します。
//
TEST(Occupy, StateAccessPointCreated)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
        nnt::ldn::Initializer initializer;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // 適当なパラメータでネットワークを構築します。
        nn::ldn::NetworkConfig network = { };
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 通信デバイスの占有権を剥奪します。
        nnt::ldn::SystemInitializer systemInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        SetLdnSettings();

        // 新しい権限でネットワークを構築します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // 取得したイベントを削除します。
        nn::os::DestroySystemEvent(&stateChangeEvent);
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::Initializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
}

#if defined(NNT_LDN_BUILD_CONFIG_LOCK_ENABLED)
//
// NIFM の Lock 後に Monitor で取得できる状態が正常であることを確認します。
//
TEST(Lock, Monitor)
{
    // NIFM をロック状態にします。
    nnt::ldn::NifmInitializer nifmInitializer;
    nnt::ldn::ScopedNifmLock nifmLock;

    // 監視モードの初期化は成功します。
    nnt::ldn::MonitorInitializer initializer;

    // 監視モードで初期化しても、 LDN ライブラリがローカル通信を開始していることは無いので、
    // nn::ldn::Initialize 関数を実行した場合とは異なり State_None のまま変化しません。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());

    // 通信状態の取得は失敗します。
    nn::ldn::NetworkInfo network;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    NNT_ASSERT_RESULT_FAILURE(
        nn::ldn::ResultInvalidState, nn::ldn::GetNetworkInfoForMonitor(&network));
    NNT_ASSERT_RESULT_FAILURE(
        nn::ldn::ResultInvalidState, nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
}

//
// LDN ライブラリが初期化されていない状態で NIFM をロックし、
// LDN ライブラリを初期化できなくなることを確認します。
//
TEST(Lock, StateNone)
{
    {
        // NIFM をロック状態にします。
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::ScopedNifmLock nifmLock;

        // 無線オフ状態で LDN ライブラリの初期化に失敗することを確認します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultLocked, nn::ldn::InitializeSystem());
    }

    // NIFM のロック状態が解除されれば LDN ライブラリの初期化は成功します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    SetLdnSettings();
    nn::ldn::FinalizeSystem();
}
#endif

#if defined(NNT_LDN_BUILD_CONFIG_FLIGHT_MODE_ENABLED)
//
// 無線オフ後に Monitor で取得できる状態が正常であることを確認します。
//
TEST(WifiOff, Monitor)
{
    // 無線スイッチをオフにします。
    nnt::ldn::NifmInitializer nifmInitializer;
    nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

    // 監視モードの初期化は成功します。
    nnt::ldn::MonitorInitializer initializer;

    // 監視モードで初期化しても、 LDN ライブラリがローカル通信を開始していることは無いので、
    // nn::ldn::Initialize 関数を実行した場合とは異なり State_None のまま変化しません。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());

    // 通信状態の取得は失敗します。
    nn::ldn::NetworkInfo network;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    NNT_ASSERT_RESULT_FAILURE(
        nn::ldn::ResultInvalidState, nn::ldn::GetNetworkInfoForMonitor(&network));
    NNT_ASSERT_RESULT_FAILURE(
        nn::ldn::ResultInvalidState, nn::ldn::GetIpv4AddressForMonitor(&address, &mask));
}

//
// LDN ライブラリが初期化されていない状態で無線オフし、LDN ライブラリを初期化できなくなることを確認します。
//
TEST(WifiOff, StateNone)
{
    {
        // 無線スイッチをオフにします。
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

        // 無線オフ状態で LDN ライブラリの初期化に失敗することを確認します。
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::InitializeSystem());
    }

    // 再度無線オン状態になれば LDN ライブラリの初期化は成功します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    SetLdnSettings();
    nn::ldn::FinalizeSystem();
}

//
// LDN ライブラリが初期化された状態で無線オフします。
//
TEST(WifiOff, StateInitialized)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();

        {
            // 無線スイッチをオフにします。
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

            // 一定時間後、 LDN プロセスが Error 状態に遷移します。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());

            // この状態での API 呼び出しは失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::OpenAccessPoint());
            NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::OpenStation());

            // イベント割り当ては無線オフ後であっても成功します。
            nn::os::SystemEvent stateChangeEvent;
            nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());
        }

        // 無線スイッチをオンに戻しても状態は変わりません。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
}

//
// OpenAccessPoint の後、無線スイッチをオフにします。
//
TEST(WifiOff, StateAccessPoint)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());

        {
            // 無線スイッチをオフにします。
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

            // 一定時間後、 LDN プロセスが Error 状態に遷移します。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());

            // この状態での API 呼び出しは失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::CloseAccessPoint());
        }

        // 無線スイッチをオンに戻しても状態は変わりません。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
}

//
// CreateNetwork の後、無線スイッチをオフにします。
//
TEST(WifiOff, StateAccessPointCreated)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化してアクセスポイントとしての動作を開始します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
        nn::os::SystemEventType stateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        // 適当なパラメータでネットワークを構築します。
        nn::ldn::NetworkConfig network;
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        {
            // 無線スイッチをオフにします。
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

            // 一定時間後、 LDN プロセスが Error 状態に遷移します。
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(
                &stateChangeEvent, nn::TimeSpan::FromSeconds(1)));
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // この状態での API 呼び出しは失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::DestroyNetwork());
        }

        // 無線スイッチをオンに戻しても状態は変わりません。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());

        // 取得したイベントを削除します。
        nn::os::DestroySystemEvent(&stateChangeEvent);
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
}

//
// OpenStation の後、無線スイッチをオフにします。
//
TEST(WifiOff, StateStation)
{
    // NIFM ライブラリを初期化しておきます。
    nnt::ldn::NifmInitializer nifmInitializer;

    {
        // LDN ライブラリを初期化してステーションとしての動作を開始します。
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

        {
            // 無線スイッチをオフにします。
            nnt::ldn::ScopedWirelessCommunicationSwitchSetter wirelessSwitch(false);

            // 一定時間後、 LDN プロセスが Error 状態に遷移します。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());

            // この状態での API 呼び出しは失敗します。
            NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::CloseStation());
        }

        // 無線スイッチをオンに戻しても状態は変わりません。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    }

    // 再度 LDN ライブラリを初期化すれば成功します。
    ASSERT_EQ(nn::ldn::State_None, nn::ldn::GetState());
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();
}
#endif

//
// State_Initialized 以外で OpenAccessPoint を実行します。
//
TEST(InvalidState, OpenAccessPoint)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_AccessPoint, State_AccessPointCreated 状態で OpenAccessPoint を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        nn::ldn::NetworkConfig network = { };
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }

    // State_Station 状態で OpenAccessPoint を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::OpenAccessPoint());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// State_AccessPoint, State_AccessPointCreated 以外で CloseAccessPoint を実行します。
//
TEST(InvalidState, CloseAccessPoint)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_Initialized 状態で CloseAccessPoint を実行します。
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CloseAccessPoint());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }

    // State_Station 状態で CloseAccessPoint を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CloseAccessPoint());
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// State_AccessPoint 以外で CreateNetwork を実行します。
//
TEST(InvalidState, CreateNetwork)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // ネットワーク構築に使用する適当なパラメータを定義します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // State_Initialized 状態で CreateNetwork を実行します。
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }

    // State_AccessPointCreated 状態で CreateNetwork を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }

    // State_Station 状態で CreateNetwork を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// State_AccessPoint 以外で CreateNetworkPrivate を実行します。
//
TEST(InvalidState, CreateNetworkPrivate)
{
    nn::ldn::SecurityParameter securityParam;

    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // ネットワーク構築に使用する適当なパラメータを定義します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // State_AccessPointCreated 状態で CreateNetworkPrivate を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetSecurityParameter(&securityParam));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CreateNetworkPrivate(
            network, security, securityParam, user, 0, nullptr));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }

    // State_Station 状態で CreateNetworkPrivate を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CreateNetworkPrivate(
            network, security, securityParam, user, 0, nullptr));
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }

    // State_Initialized 状態で CreateNetworkPrivate を実行します。
    {
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CreateNetworkPrivate(
            network, security, securityParam, user, 0, nullptr));
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }
}

//
// State_Initialized 以外で OpenAccessPoint を実行します。
//
TEST(InvalidState, OpenStation)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_AccessPoint, State_AccessPointCreated 状態で OpenStation を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        nn::ldn::NetworkConfig network = { };
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::OpenStation());
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }
}

//
// State_Station, State_StationConnected 以外で CloseStation を実行します。
//
TEST(InvalidState, CloseStation)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_Initialized 状態で CloseStation を実行します。
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CloseStation());
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    }

    // State_AccessPoint 状態で CloseStation を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::CloseStation());
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
    }
}

//
// ネットワークに参加していない状態で GetNetworkInfo を実行します。
//
TEST(InvalidState, GetNetworkInfo)
{
    nnt::ldn::SystemInitializer initializer;
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    SetLdnSettings();

    // State_Initialized
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfo(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    }

    // State_AccessPoint
    {
        nnt::ldn::AccessPointStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfo(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    }

    // State_Station
    {
        nnt::ldn::StationStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetNetworkInfo(&networkInfo));
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    }
}

//
// ネットワークに参加していない状態で GetIpv4Address を実行します。
//
TEST(InvalidState, GetIpv4Address)
{
    nnt::ldn::SystemInitializer initializer;
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    SetLdnSettings();

    // State_Initialized
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::ldn::ResultInvalidState, nn::ldn::GetIpv4Address(&address, &mask));
    }

    // State_AccessPoint
    {
        nnt::ldn::AccessPointStarter started;
        NNT_ASSERT_RESULT_FAILURE(
            nn::ldn::ResultInvalidState, nn::ldn::GetIpv4Address(&address, &mask));
    }

    // State_Station
    {
        nnt::ldn::StationStarter started;
        NNT_ASSERT_RESULT_FAILURE(
            nn::ldn::ResultInvalidState, nn::ldn::GetIpv4Address(&address, &mask));
    }
}

//
// ネットワークに参加していない状態で GetSecurityParameter を実行します。
//
TEST(InvalidState, GetSecurityParameter)
{
    nnt::ldn::SystemInitializer initializer;
    nn::ldn::SecurityParameter param;
    SetLdnSettings();

    // State_Initialized
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetSecurityParameter(&param));
    }

    // State_AccessPoint
    {
        nnt::ldn::AccessPointStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetSecurityParameter(&param));
    }

    // State_Station
    {
        nnt::ldn::StationStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::GetSecurityParameter(&param));
    }
}

//
// ネットワークに参加していない状態で GetNetworkConfig を実行します。
//
TEST(InvalidState, GetNetworkConfig)
{
    nnt::ldn::SystemInitializer initializer;
    nn::ldn::NetworkConfig config;
    SetLdnSettings();

    // State_Initialized
    {
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkConfig(&config));
    }

    // State_AccessPoint
    {
        nnt::ldn::AccessPointStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkConfig(&config));
    }

    // State_Station
    {
        nnt::ldn::StationStarter started;
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::GetNetworkConfig(&config));
    }
}

//
// State_Initialized 以外で SetOperationMode を実行します。
//
TEST(InvalidState, SetOperationMode)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_AccessPoint, State_AccessPointCreated 状態で SetOperationMode を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        nn::ldn::NetworkConfig network = { };
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }

    // State_Station 状態で SetOperationMode を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState, nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
    }
}


//
// State_Initialized 以外で SetWirelessControllerRestriction を実行します。
//
TEST(InvalidState, SetWirelessControllerRestriction)
{
    // LDN ライブラリを初期化します。
    nnt::ldn::SystemInitializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    SetLdnSettings();

    // State_AccessPoint, State_AccessPointCreated 状態で
    // SetWirelessControllerRestriction を実行します。
    {
        nnt::ldn::AccessPointStarter starter;
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::SetWirelessControllerRestriction(
                static_cast<nn::ldn::WirelessControllerRestriction>(
                    g_TestConfig.wirelessControllerRestriction)));
        ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

        nn::ldn::NetworkConfig network = { };
        network.channel = nn::ldn::AutoChannel;
        network.intentId = IntentId;
        network.nodeCountMax = 4;
        network.localCommunicationVersion = 1;
        nn::ldn::SecurityConfig security;
        security.securityMode = nn::ldn::SecurityMode_Product;
        std::memcpy(security.passphrase, Passphrase, PassphraseSize);
        security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
        nn::ldn::UserConfig user= {};
        std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::SetWirelessControllerRestriction(
                static_cast<nn::ldn::WirelessControllerRestriction>(
                    g_TestConfig.wirelessControllerRestriction)));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());
    }

    // State_Station 状態で SetWirelessControllerRestriction を実行します。
    {
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultInvalidState,
            nn::ldn::SetWirelessControllerRestriction(
                static_cast<nn::ldn::WirelessControllerRestriction>(
                    g_TestConfig.wirelessControllerRestriction)));
    }
}

//
// イベントシグナルがコンテキスト毎に独立していることを確認します。
//
TEST(Context, Signal)
{
    // LDN ライブラリを初期化してイベントを取得します。
    nnt::ldn::Initializer initializer;
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    nn::os::SystemEventType applicationStateChangeEvent;
    nn::ldn::AttachStateChangeEvent(&applicationStateChangeEvent);

    // アクセスポイントとしての動作を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    // 適当なパラメータでネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
    ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

    // AutoReset の設定でイベントがシグナルされていることを検証します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&applicationStateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&applicationStateChangeEvent));

    // システム権限でローカル通信を開始します。
    {
        // LDN ライブラリを初期化してイベントを取得します。
        nnt::ldn::SystemInitializer systemInitializer;
        ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
        nn::os::SystemEventType systemStateChangeEvent;
        nn::ldn::AttachStateChangeEvent(&systemStateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&systemStateChangeEvent));

        // ステーションとしての動作を開始します。
        nnt::ldn::StationStarter starter;
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }

    // ネットワークが破棄されたため、イベントがシグナルされていることを確認します。
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&applicationStateChangeEvent));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&applicationStateChangeEvent));
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_OperationMode |
        nnt::ldn::CommandLineOptionFlag_WirelessControllerRestriction);

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 無線スイッチをオンにします。
    {
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::SetWirelessCommunicationSwitch(true);
    }

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
