﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * 1 台の開発機だけで実行できるテストです。
 */

#include <nn/ldn.h>
#include <nn/ldn/ldn_MonitorApi.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/os.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_Config.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>

namespace
{
    const nn::Bit64 LocalId = UINT64_C(0x0005000c10000000);
    const nn::Bit16 SceneId = UINT16_C(0x1100);
    const nn::ldn::IntentId IntentId = nn::ldn::MakeIntentId(LocalId, SceneId);
    const char Passphrase[] = "LdnTestPassphrase";
    const size_t PassphraseSize = std::strlen(Passphrase);
    const char UserName[] = "TestAp";

    nn::ldn::NetworkInfo g_Networks[nn::ldn::ScanResultCountMax];

    void CreateRandom(void* data, size_t dataSize) NN_NOEXCEPT
    {
        nn::Bit8* binary = static_cast<nn::Bit8*>(data);
        for (size_t i = 0; i < dataSize; ++i)
        {
            binary[i] = static_cast<nn::Bit8>(nn::os::GetSystemTick().GetInt64Value() >> 1);
        }
    }

} // namespace <unnamed>

TEST(NotInitializedDeathTest, Finalize)
{
    ASSERT_DEATH(nn::ldn::Finalize(), "");
}

TEST(NotInitializedDeathTest, AttachStateChangeEvent)
{
    nn::os::SystemEvent stateChangeEvent;
    ASSERT_DEATH(nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase()), "");
}

TEST(NotInitializedDeathTest, Scan)
{
    int count;
    nn::ldn::ScanFilter filter = { };
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(NotInitializedDeathTest, ScanPrivate)
{
    int count;
    nn::ldn::ScanFilter filter = { };
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(NotInitializedDeathTest, GetNetworkInfo)
{
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    ASSERT_DEATH(nn::ldn::GetNetworkInfo(g_Networks, updates, nn::ldn::NodeCountMax), "");
    ASSERT_DEATH(nn::ldn::GetNetworkInfo(g_Networks), "");
}

TEST(NotInitializedDeathTest, GetIpv4Address)
{
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    ASSERT_DEATH(nn::ldn::GetIpv4Address(&address, &mask), "");
}

TEST(NotInitializedDeathTest, GetSecurityParameter)
{
    nn::ldn::SecurityParameter param;
    ASSERT_DEATH(nn::ldn::GetSecurityParameter(&param), "");
}

TEST(NotInitializedDeathTest, GetNetworkConfig)
{
    nn::ldn::NetworkConfig config;
    ASSERT_DEATH(nn::ldn::GetNetworkConfig(&config), "");
}

TEST(NotInitializedDeathTest, OpenAccessPoint)
{
    ASSERT_DEATH(nn::ldn::OpenAccessPoint(), "");
}

TEST(NotInitializedDeathTest, CloseAccessPoint)
{
    ASSERT_DEATH(nn::ldn::CloseAccessPoint(), "");
}

TEST(NotInitializedDeathTest, CreateNetwork)
{
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
}

TEST(NotInitializedDeathTest, CreateNetworkPrivate)
{
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter param = { };
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, 0, nullptr), "");
}

TEST(NotInitializedDeathTest, DestroyNetwork)
{
    ASSERT_DEATH(nn::ldn::DestroyNetwork(), "");
}

TEST(NotInitializedDeathTest, SetAdvertiseData)
{
    ASSERT_DEATH(nn::ldn::SetAdvertiseData(nullptr, 0U), "");
}

TEST(NotInitializedDeathTest, SetStationAcceptPolicy)
{
    ASSERT_DEATH(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept), "");
}

TEST(NotInitializedDeathTest, Reject)
{
    nn::ldn::Ipv4Address address = nn::ldn::MakeIpv4Address(192, 168, 10, 2);
    ASSERT_DEATH(nn::ldn::Reject(address), "");
}

TEST(NotInitializedDeathTest, AddAcceptFilterEntry)
{
    nn::ldn::NodeInfo node = { };
    nn::ldn::MacAddress address = nn::ldn::MakeMacAddress(1, 2, 3, 4, 5, 6);
    ASSERT_DEATH(nn::ldn::AddAcceptFilterEntry(node), "");
    ASSERT_DEATH(nn::ldn::AddAcceptFilterEntry(address), "");
}

TEST(NotInitializedDeathTest, ClearAcceptFilterEntry)
{
    ASSERT_DEATH(nn::ldn::ClearAcceptFilter(), "");
}

TEST(NotInitializedDeathTest, OpenStation)
{
    ASSERT_DEATH(nn::ldn::OpenStation(), "");
}

TEST(NotInitializedDeathTest, CloseStation)
{
    ASSERT_DEATH(nn::ldn::CloseStation(), "");
}

TEST(NotInitializedDeathTest, Connect)
{
    nn::ldn::NetworkInfo network = { };
    nn::ldn::SecurityConfig security = { };
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user = { };
    ASSERT_DEATH(nn::ldn::Connect(network, security, user, 1, nn::ldn::ConnectOption_None), "");
}

TEST(NotInitializedDeathTest, ConnectPrivate)
{
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::SecurityParameter param = { };
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user = { };
    ASSERT_DEATH(nn::ldn::ConnectPrivate(
        network, security, param, user, 1, nn::ldn::ConnectOption_None), "");
}

TEST(NotInitializedDeathTest, Disconnect)
{
    ASSERT_DEATH(nn::ldn::Disconnect(), "");
}

TEST(NotInitializedDeathTest, GetDisconnectReason)
{
    ASSERT_DEATH(nn::ldn::GetDisconnectReason(), "");
}

TEST(NotInitializedDeathTest, SetOperationMode)
{
    ASSERT_DEATH(nn::ldn::SetOperationMode(nn::ldn::OperationMode_HighSpeed), "");
}

TEST(MonitorDeathTest, Finalize)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::Finalize(), "");
}

TEST(MonitorDeathTest, AttachStateChangeEvent)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::os::SystemEvent stateChangeEvent;
    ASSERT_DEATH(nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase()), "");
}

TEST(MonitorDeathTest, Scan)
{
    nnt::ldn::MonitorInitializer initializer;
    int count;
    nn::ldn::ScanFilter filter = { };
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(MonitorDeathTest, ScanPrivate)
{
    nnt::ldn::MonitorInitializer initializer;
    int count;
    nn::ldn::ScanFilter filter = { };
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(MonitorDeathTest, GetSecurityParameter)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::SecurityParameter param;
    ASSERT_DEATH(nn::ldn::GetSecurityParameter(&param), "");
}

TEST(MonitorDeathTest, GetNetworkConfig)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NetworkConfig config;
    ASSERT_DEATH(nn::ldn::GetNetworkConfig(&config), "");
}

TEST(MonitorDeathTest, OpenAccessPoint)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::OpenAccessPoint(), "");
}

TEST(MonitorDeathTest, CloseAccessPoint)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::CloseAccessPoint(), "");
}

TEST(MonitorDeathTest, CreateNetwork)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= {};
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
}

TEST(MonitorDeathTest, CreateNetworkPrivate)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NetworkConfig network = { };
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 4;
    network.localCommunicationVersion = 1;
    nn::ldn::SecurityConfig security;
    security.securityMode = nn::ldn::SecurityMode_Product;
    std::memcpy(security.passphrase, Passphrase, PassphraseSize);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user= { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    nn::ldn::SecurityParameter param = { };
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, 0, nullptr), "");
}

TEST(MonitorDeathTest, DestroyNetwork)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::DestroyNetwork(), "");
}

TEST(MonitorDeathTest, SetAdvertiseData)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::SetAdvertiseData(nullptr, 0U), "");
}

TEST(MonitorDeathTest, SetStationAcceptPolicy)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::SetStationAcceptPolicy(nn::ldn::AcceptPolicy_AlwaysAccept), "");
}

TEST(MonitorDeathTest, Reject)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::Ipv4Address address = nn::ldn::MakeIpv4Address(192, 168, 10, 2);
    ASSERT_DEATH(nn::ldn::Reject(address), "");
}

TEST(MonitorDeathTest, AddAcceptFilterEntry)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NodeInfo node = { };
    nn::ldn::MacAddress address = nn::ldn::MakeMacAddress(1, 2, 3, 4, 5, 6);
    ASSERT_DEATH(nn::ldn::AddAcceptFilterEntry(node), "");
    ASSERT_DEATH(nn::ldn::AddAcceptFilterEntry(address), "");
}

TEST(MonitorDeathTest, ClearAcceptFilterEntry)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::ClearAcceptFilter(), "");
}

TEST(MonitorDeathTest, OpenStation)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::OpenStation(), "");
}

TEST(MonitorDeathTest, CloseStation)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::CloseStation(), "");
}

TEST(MonitorDeathTest, Connect)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NetworkInfo network = { };
    nn::ldn::SecurityConfig security = { };
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user = { };
    ASSERT_DEATH(nn::ldn::Connect(network, security, user, 1, nn::ldn::ConnectOption_None), "");
}

TEST(MonitorDeathTest, ConnectPrivate)
{
    nnt::ldn::MonitorInitializer initializer;
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::SecurityParameter param = { };
    security.passphraseSize = nn::ldn::PassphraseSizeMin;
    nn::ldn::UserConfig user = { };
    ASSERT_DEATH(nn::ldn::ConnectPrivate(
        network, security, param, user, 1, nn::ldn::ConnectOption_None), "");
}

TEST(MonitorDeathTest, Disconnect)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::Disconnect(), "");
}

TEST(MonitorDeathTest, GetDisconnectReason)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::GetDisconnectReason(), "");
}

TEST(MonitorDeathTest, SetOperationMode)
{
    nnt::ldn::MonitorInitializer initializer;
    ASSERT_DEATH(nn::ldn::SetOperationMode(nn::ldn::OperationMode_HighSpeed), "");
}

TEST(InvalidArgumentDeathTest, Scan)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { };
    int count;

    // 出力バッファに nullptr を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::Scan(
        nullptr, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 結果の出力先に nullptr を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, nullptr, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // バッファサイズに異常値を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, 0, filter, nn::ldn::AutoChannel), "");

    // 無効なチャンネルを指定して失敗することを確認します。
    int channels[] = {
        2, 3, 4, 5, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
        27, 28, 29, 30, 31, 32, 33, 34, 35, 37, 38, 39, 41, 42, 43, 45, 46, 47, 49, 50, 51, 52
    };
    for (int channel : channels)
    {
        ASSERT_DEATH(nn::ldn::Scan(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, channel), "");
    }

    // BSSID をフィルタに使用します。
    filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_Bssid);
    filter.bssid = nn::ldn::MakeMacAddress(0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc);
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 最小文字数以下の SSID を指定してフィルタに使用します。
    filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_Ssid);
    filter.ssid = nn::ldn::MakeSsid("");
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 最大文字数以下の SSID を指定してフィルタに使用します。
    filter.flag = nn::ldn::ScanFilterFlag_Ssid;
    filter.ssid.length = nn::ldn::SsidLengthMax + 1;
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 異常な NetworkType を指定してフィルタに使用します。
    filter.flag = nn::ldn::ScanFilterFlag_NetworkType;
    filter.networkType = 0xFFFFFFFF;
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 異常な flag を指定してフィルタに使用します。
    filter.flag = ~static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_All);
    ASSERT_DEATH(nn::ldn::Scan(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(InvalidArgumentDeathTest, ScanPrivate)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::StationStarter starter;
    nn::ldn::ScanFilter filter = { 0 };
    int count;

    // 出力バッファに nullptr を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        nullptr, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 結果の出力先に nullptr を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, nullptr, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // バッファサイズに異常値を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, 0, filter, nn::ldn::AutoChannel), "");

    // 無効なチャンネルを指定して失敗することを確認します。
    int channels[] = {
        2, 3, 4, 5, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
        27, 28, 29, 30, 31, 32, 33, 34, 35, 37, 38, 39, 41, 42, 43, 45, 46, 47, 49, 50, 51, 52
    };
    for (int channel : channels)
    {
        ASSERT_DEATH(nn::ldn::ScanPrivate(
            g_Networks, &count, nn::ldn::ScanResultCountMax, filter, channel), "");
    }

    // 最小文字数以下の SSID を指定してフィルタに使用します。
    filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_Ssid);
    filter.ssid = nn::ldn::MakeSsid("");
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 最大文字数以下の SSID を指定してフィルタに使用します。
    filter.flag = nn::ldn::ScanFilterFlag_Ssid;
    filter.ssid.length = nn::ldn::SsidLengthMax + 1;
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 異常な NetworkType を指定してフィルタに使用します。
    filter.flag = nn::ldn::ScanFilterFlag_NetworkType;
    filter.networkType = 0xFFFFFFFF;
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");

    // 異常な flag を指定してフィルタに使用します。
    filter.flag = ~static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_All);
    ASSERT_DEATH(nn::ldn::ScanPrivate(
        g_Networks, &count, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel), "");
}

TEST(InvalidArgumentDeathTest, GetNetworkInfo)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = IntentId;
    network.nodeCountMax = 2;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 64;
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));

    // GetNetworkInfo に nullptr を指定すると失敗します。
    ASSERT_DEATH(nn::ldn::GetNetworkInfo(nullptr), "");
}

TEST(InvalidArgumentDeathTest, GetIpv4Address)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワークを構築します。
    nn::ldn::NetworkConfig network = { };
    network.intentId = IntentId;
    network.nodeCountMax = 2;
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 64;
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));

    // 引数に nullptr を指定すると失敗します。
    nn::ldn::Ipv4Address address;
    nn::ldn::SubnetMask mask;
    ASSERT_DEATH(nn::ldn::GetIpv4Address(nullptr, &mask), "");
    ASSERT_DEATH(nn::ldn::GetIpv4Address(&address, nullptr), "");
}


TEST(InvalidArgumentDeathTest, GetSecurityParameter)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワーク構築に使用する適当なパラメータを設定します。
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 32;
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // 最小のノード数でネットワークを構築します。
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 1;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));

    // セキュリティ・パラメータを取得します。
    ASSERT_DEATH(nn::ldn::GetSecurityParameter(nullptr), "");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
}

TEST(InvalidArgumentDeathTest, GetNetworkConfig)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワーク構築に使用する適当なパラメータを設定します。
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 32;
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // 最小のノード数でネットワークを構築します。
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 1;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));

    // ネットワーク設定を取得します。
    ASSERT_DEATH(nn::ldn::GetNetworkConfig(nullptr), "");
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
}

TEST(InvalidArgumentDeathTest, CreateNetwork)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワーク構築に使用する適当なパラメータを設定します。
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 32;
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ノード数 0 でネットワークを構築します。
    network.intentId = IntentId;
    network.nodeCountMax = 0;
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");

    // 最大を超えるノード数でネットワークを構築します。
    network.nodeCountMax = nn::ldn::NodeCountMax + 1;
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
    network.nodeCountMax = nn::ldn::NodeCountMax;

    // 無効な無線チャンネルでネットワークを構築します。
    int channels[] = {
        2, 3, 4, 5, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
        27, 28, 29, 30, 31, 32, 33, 34, 35, 37, 38, 39, 41, 42, 43, 45, 46, 47, 49, 50, 51, 52
    };
    for (int channel : channels)
    {
        network.channel = static_cast<int16_t>(channel);
        ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
    }
    network.channel = nn::ldn::AutoChannel;

    // 最小を下回る鍵長でネットワークを構築します。
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = nn::ldn::PassphraseSizeMin - 1;
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");

    // 最大を上回る鍵長でネットワークを構築します。
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Debug);
    security.passphraseSize = nn::ldn::PassphraseSizeMax + 1;
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
    security.passphraseSize = nn::ldn::PassphraseSizeMin;

    // 最小を下回るローカル通信バージョンでネットワークを構築します。
    network.localCommunicationVersion = static_cast<int16_t>(
        nn::ldn::LocalCommunicationVersionMin - 1);
    ASSERT_DEATH(nn::ldn::CreateNetwork(network, security, user), "");
    network.localCommunicationVersion = 0;
}

TEST(InvalidArgumentDeathTest, CreateNetworkPrivate)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // ネットワーク構築に使用する適当なパラメータを設定します。
    nn::ldn::NetworkConfig network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::SecurityParameter param = { };
    nn::ldn::UserConfig user = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = 32;
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // ノード数 0 でネットワークを構築します。
    network.intentId = IntentId;
    network.nodeCountMax = 0;
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(network, security, param, user, 0, nullptr), "");

    // 最大を超えるノード数でネットワークを構築します。
    network.nodeCountMax = nn::ldn::NodeCountMax + 1;
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(network, security, param, user, 0, nullptr), "");
    network.nodeCountMax = nn::ldn::NodeCountMax;

    // 無効な無線チャンネルでネットワークを構築します。
    int channels[] = {
        2, 3, 4, 5, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
        27, 28, 29, 30, 31, 32, 33, 34, 35, 37, 38, 39, 41, 42, 43, 45, 46, 47, 49, 50, 51, 52
    };
    for (int channel : channels)
    {
        network.channel = static_cast<int16_t>(channel);
        ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
            network, security, param, user, 0, nullptr), "");
    }
    network.channel = nn::ldn::AutoChannel;

    // 最小を下回る鍵長でネットワークを構築します。
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = nn::ldn::PassphraseSizeMin - 1;
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, 0, nullptr), "");

    // 最大を上回る鍵長でネットワークを構築します。
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Debug);
    security.passphraseSize = nn::ldn::PassphraseSizeMax + 1;
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, 0, nullptr), "");
    security.passphraseSize = nn::ldn::PassphraseSizeMin;

    // 最小を下回るローカル通信バージョンでネットワークを構築します。
    network.localCommunicationVersion = static_cast<int16_t>(
        nn::ldn::LocalCommunicationVersionMin - 1);
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(network, security, param, user, 0, nullptr), "");
    network.localCommunicationVersion = 0;

    // 最大数を超えるアドレスエントリを与えます。
    nn::ldn::AddressEntry entries[nn::ldn::AddressEntryCountMax + 1] = { };
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, nn::ldn::AddressEntryCountMax + 1, entries), "");

    // 最小数を下回るアドレスエントリを与えます。
    ASSERT_DEATH(nn::ldn::CreateNetworkPrivate(
        network, security, param, user, -1, entries), "");
}

TEST(InvalidArgumentDeathTest, SetAdvertiseData)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;

    // nullptr と 1 バイト以上のサイズを指定すると失敗します。
    ASSERT_DEATH(nn::ldn::SetAdvertiseData(nullptr, 1), "");

    // 最大サイズを超えた Advertise データを設定すると失敗します。
    char data[nn::ldn::AdvertiseDataSizeMax + 1];
    ASSERT_DEATH(nn::ldn::SetAdvertiseData(data, sizeof(data)), "");
}

TEST(InvalidArgumentDeathTest, SetStationAcceptPolicy)
{
    nnt::ldn::Initializer initializer;
    nnt::ldn::AccessPointStarter starter;
    ASSERT_DEATH(nn::ldn::SetStationAcceptPolicy(
        static_cast<nn::ldn::AcceptPolicy>(-1)), "");
    ASSERT_DEATH(nn::ldn::SetStationAcceptPolicy(
        static_cast<nn::ldn::AcceptPolicy>(4)), "");
}

TEST(InvalidArgumentDeathTest, SetOperationMode)
{
    nnt::ldn::SystemInitializer initializer;
    ASSERT_DEATH(nn::ldn::SetOperationMode(
        static_cast<nn::ldn::OperationMode>(-1)), "");
    ASSERT_DEATH(nn::ldn::SetOperationMode(
        static_cast<nn::ldn::OperationMode>(2)), "");
}
