﻿
Param(
    [Parameter(Mandatory=$TRUE)][string] $TestName,
    [string] $Build = "Develop",
    [string] $Specs = "NX",
    [string] $Platforms = "NXFP2-a64",
    [int] $SceneId = -1
)

# スクリプトをロードします。
$SCRIPT_DIR = (Split-Path -Parent $MyInvocation.MyCommand.Path)
. $SCRIPT_DIR\Siglo\Bundle
. $SCRIPT_DIR\Runner\RunParallel

# テストの NSP ファイルのパスを取得します。
Function TestPath($Name)
{
    $FileName = "testLdn_$Name"
    $BuildTarget = "$Specs-$Platforms"
    return "$SIGLO_ROOT\Tests\Outputs\$BuildTarget\Tests\$FileName\$Build\$FileName.nsp"
}

# Stable モードで Network テストを実行します。
Function NetworkStable($Targets, $SceneId)
{
    $AccessPointPath = (TestPath "IntegrationNetworkAccessPoint")
    $StationPath = (TestPath "IntegrationNetworkStation")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $AccessPointPath $StationPath)
    return $ErrorCount
}

# HighSpeed モードで Network テストを実行します。
Function NetworkHighSpeed($Targets, $SceneId)
{
    $AccessPointPath = (TestPath "IntegrationNetworkAccessPoint")
    $StationPath = (TestPath "IntegrationNetworkStation")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $AccessPointPath $StationPath `
        "--mode" "highspeed")
    return $ErrorCount
}

# コントローラの接続制限を無効化して Network テストを実行します。
Function NetworkCross($Targets, $SceneId)
{
    $AccessPointPath = (TestPath "IntegrationNetworkAccessPoint")
    $StationPath = (TestPath "IntegrationNetworkStation")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $AccessPointPath $StationPath `
        "--wireless-controller-restriction" "disabled")
    return $ErrorCount
}

# Stable モードで UDP テストを実行します。
Function UdpStable($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationUdp")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $Path $Path `
        "-e" "-b" "60" "-u" "4" "-p" "1472" "-d" "30s")
    return $ErrorCount
}

# HighSpeed モードで UDP テストを実行します。
Function UdpHighSpeed($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationUdp")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $Path $Path `
        "--mode" "highspeed" "-e" "-b" "60" "-u" "4" "-p" "1472" "-d" "30s")
    return $ErrorCount
}

# コントローラの接続制限を無効化して UDP テストを実行します。
Function UdpCross($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationUdp")
    $ErrorCount = 0
    $ErrorCount += (RunParallel $Targets $SceneId $Path $Path `
        "-w" "disabled" "-e" "-b" "60" "-u" "4" "-p" "1472" "-d" "30s")
    return $ErrorCount
}

# 全てのテストを実行します。
Function All($Targets, $SceneId)
{
    $ErrorCount = 0
    $ErrorCount += (NetworkStable $Targets $SceneId)
    $ErrorCount += (NetworkHighSpeed $Targets $SceneId)
    $ErrorCount += (NetworkCross $Targets $SceneId)
    $ErrorCount += (UdpStable $Targets $SceneId)
    $ErrorCount += (UdpHighSpeed $Targets $SceneId)
    $ErrorCount += (UdpCross $Targets $SceneId)
    return $ErrorCount
}

# 指定されたテストを実行します。
Function Run($SceneId)
{
    # LdnTestBridge を終了しておきます。
    if (Get-Process | Where-Object { $_.Name -eq "LdnTestBridge" })
    {
        Stop-Process -Name "LdnTestBridge"
    }

    # PC に接続されている開発機を取得します。
    $Targets = GetTargetList

    # Scene Id が指定されなかった場合、ランダムに生成します。
    if ($SceneId -lt 0)
    {
        $SceneId = (Get-Random 65536)
    }

    # 引数で指定されたテストを実行します。
    if ($TestName -eq "All")
    {
        return (All $Targets $SceneId)
    }
    elseif ($TestName -eq "NetworkStable")
    {
        return (NetworkStable $Targets $SceneId)
    }
    elseif ($TestName -eq "NetworkHighSpeed")
    {
        return (NetworkHighSpeed $Targets $SceneId)
    }
    elseif ($TestName -eq "NetworkCross")
    {
        return (NetworkCross $Targets $SceneId)
    }
    elseif ($TestName -eq "UdpStable")
    {
        return (UdpStable $Targets $SceneId)
    }
    elseif ($TestName -eq "UdpHighSpeed")
    {
        return (UdpHighSpeed $Targets $SceneId)
    }
    elseif ($TestName -eq "UdpCross")
    {
        return (UdpCross $Targets $SceneId)
    }
    else
    {
        throw "Unknown Test: $TestName"
    }
}

$ExitCode = (Run $SceneId)
if ($ExitCode -ne 0)
{
    throw "Test Failed: $TestName ($ExitCode)"
}
return $ExitCode
