﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/htcs.h>
#include <nn/os.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/detail/testLdn_SynchronizationInterpreter.h>

namespace nnt { namespace ldn { namespace
{
    /**
     * @brief       Test Bridge に接続します。
     * @param[out]  pOutSocket      socket ディスクリプタの出力先です。
     * @param[in]   timeout         タイムアウト時間です。
     * @return      処理の結果です。
     */
    SynchronizationResult ConnectToTestBridge(
        int* pOutSocket, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        auto start = nn::os::GetSystemTick();

        // ソケットを生成します。
        int socket = nn::htcs::Socket();
        if (socket < 0)
        {
            int error = nn::htcs::GetLastError();
            if (error == nn::htcs::HTCS_ENOBUFS)
            {
                return SynchronizationResult_InsufficientBuffer;
            }
            else
            {
                return SynchronizationResult_Unknown;
            }
        }

        // Test Bridge に接続します。
        nn::htcs::SockAddrHtcs addr;
        addr.family = nn::htcs::HTCS_AF_HTCS;
        addr.peerName = nn::htcs::GetPeerNameAny();
        std::strncpy(addr.portName.name, "LdnTestBridge", nn::htcs::PortNameBufferLength);
        for (;;)
        {
            if (nn::htcs::Connect(socket, &addr) == 0)
            {
                break;
            }
            auto elapsed = (nn::os::GetSystemTick() - start).ToTimeSpan();
            if (timeout < elapsed)
            {
                nn::htcs::Close(socket);
                return SynchronizationResult_Timeout;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        // Test Bridge への接続に成功しました。
        *pOutSocket = socket;
        return SynchronizationResult_Success;
    }

    /**
     * @brief       エラーコードを Result に変換します。
     * @param[in]   error       変換対象のエラーコードです。
     * @return      エラーコードに対応する Result です。
     */
    SynchronizationResult ConvertErrorCodeToSynchronizationResult(
        detail::SynchronizationErrorCode error) NN_NOEXCEPT
    {
        switch (error)
        {
        case detail::SynchronizationErrorCode_Success:
            return SynchronizationResult_Success;
        case detail::SynchronizationErrorCode_InvalidState:
            return SynchronizationResult_InvalidState;
        case detail::SynchronizationErrorCode_Timeout:
            return SynchronizationResult_Timeout;
        default:
            return SynchronizationResult_Unknown;
        }
    }

    int Recv(int socket, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        int offset = 0;
        char* p = static_cast<char*>(buffer);
        do
        {
            auto readSize = static_cast<int>(
                nn::htcs::Recv(socket, p + offset, bufferSize - offset, 0));
            if (readSize <= 0)
            {
                return readSize;
            }
            NNT_LDN_LOG_DEBUG("Received %d Bytes\n", readSize);
            offset += readSize;
        } while (p[offset - 1] != '\n');
        p[offset] = 0;
        return offset;
    }

}}} // namespace nnt::ldn::<unnamed>

namespace nnt { namespace ldn
{
    HtcsSynchronizationServer::HtcsSynchronizationServer(
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(static_cast<detail::HtcsSynchronizationServerBuffer*>(buffer)),
          m_Socket(-1),
          m_ClientCount(0),
          m_DataSize(0U),
          m_Timeout(nn::TimeSpan::FromSeconds(60)),
          m_IsRunning(false)
    {
        NN_ASSERT(sizeof(detail::HtcsSynchronizationServerBuffer) <= bufferSize);
        NN_UNUSED(bufferSize);
        nn::htcs::Initialize(m_Buffer->htcs, sizeof(m_Buffer->htcs));
    }

    HtcsSynchronizationServer::~HtcsSynchronizationServer() NN_NOEXCEPT
    {
        if (m_IsRunning)
        {
            DestroyServer();
        }
        nn::htcs::Finalize();
    }

    int HtcsSynchronizationServer::SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_ASSERT(nn::TimeSpan() < timeout);
        m_Timeout = timeout;
        return SynchronizationResult_Success;
    }

    int HtcsSynchronizationServer::CreateServer(
        const char* groupName, int clientCount) NN_NOEXCEPT
    {
        NN_ASSERT(!m_IsRunning);
        NN_ASSERT_NOT_NULL(groupName);
        NN_ASSERT_MINMAX(std::strlen(groupName), 1U, SynchronizationGroupNameLengthMax);
        NN_ASSERT_MINMAX(clientCount, 0, SynchronizationClientCountMax);

        // 同期に必要な情報を記憶しておきます。
        NNT_LDN_LOG_DEBUG("Create group: %s\n", groupName);
        std::strncpy(m_GroupName, groupName, SynchronizationGroupNameLengthMax);
        m_ClientCount = clientCount;

        // Test Bridge に接続します。
        int result = ConnectToTestBridge(&m_Socket, m_Timeout);
        if (result != SynchronizationResult_Success)
        {
            NNT_LDN_LOG_DEBUG("Failed to connect to the test bridge\n");
            return result;
        }

        // グループを作成します。
        result = SendCreateGroupRequest();
        if (result == SynchronizationResult_Success)
        {
            m_IsRunning = true;
        }
        else
        {
            nn::htcs::Close(m_Socket);
        }
        return result;
    }

    int HtcsSynchronizationServer::DestroyServer() NN_NOEXCEPT
    {
        NN_ASSERT(m_IsRunning);
        SendLeaveRequest();
        nn::htcs::Shutdown(m_Socket, nn::htcs::HTCS_SHUT_RDWR);
        nn::htcs::Close(m_Socket);
        m_IsRunning = false;
        m_DataSize = 0U;
        return SynchronizationResult_Success;
    }

    int HtcsSynchronizationServer::SetData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsRunning);
        NN_ASSERT(dataSize <= SynchronizationDataSizeMax);
        if (0U < dataSize)
        {
            NN_SDK_ASSERT_NOT_NULL(data);
            std::memcpy(m_Data, data, dataSize);
        }
        m_DataSize = dataSize;
        return SynchronizationResult_Success;
    }

    int HtcsSynchronizationServer::Synchronize(const char* keyword) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsRunning);
        NN_ASSERT_NOT_NULL(keyword);
        NN_ASSERT_MINMAX(std::strlen(keyword), 1U, SynchronizationKeywordLengthMax);
        NNT_LDN_LOG_DEBUG("Sync: %s\n", keyword);
        return SendSyncRequest(keyword);
    }

    int HtcsSynchronizationServer::GetClientCount() const NN_NOEXCEPT
    {
        NN_ASSERT(m_IsRunning);
        return m_ClientCount;
    }

    int HtcsSynchronizationServer::SendCreateGroupRequest() NN_NOEXCEPT
    {
        // グループの生成に必要なリクエストを生成します。
        size_t dataSize;
        detail::BuildCreateGroupRequest(
            m_Buffer->send, &dataSize, sizeof(m_Buffer->send), m_GroupName, m_ClientCount);

        // リクエストを送信します。
        if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
        {
            NNT_LDN_LOG_DEBUG("Failed to create group: send error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスの受信まで待機します。
        if (Recv(m_Socket, m_Buffer->recv, sizeof(m_Buffer->recv)) <= 0)
        {
            NNT_LDN_LOG_DEBUG("Failed to create group: receive error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスを解析します。
        detail::SynchronizationErrorCode error;
        if (!detail::ParseCreateGroupResponse(&error, m_Buffer->recv))
        {
            NNT_LDN_LOG_DEBUG("Failed to create group: invalid response\n");
            return SynchronizationResult_Unknown;
        }
        auto result = ConvertErrorCodeToSynchronizationResult(error);
        if (result == SynchronizationResult_Success)
        {
            NNT_LDN_LOG_DEBUG("Succeeded to create group\n");
        }
        else
        {
            NNT_LDN_LOG_DEBUG("Failed to create group: %d\n", result);
        }
        return result;
    }

    int HtcsSynchronizationServer::SendSyncRequest(const char* keyword) NN_NOEXCEPT
    {
        // 同期に必要なリクエストを生成します。
        size_t dataSize;
        detail::BuildSyncRequest(
            m_Buffer->send, &dataSize, sizeof(m_Buffer->send),
            keyword, m_Data, m_DataSize, m_Timeout);

        // リクエストを送信します。
        if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: send error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスの受信まで待機します。
        if (Recv(m_Socket, m_Buffer->recv, sizeof(m_Buffer->recv)) <= 0)
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: receive error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスを解析します。
        detail::SynchronizationErrorCode error;
        if (!detail::ParseSyncResponse(&error, m_Buffer->recv, nullptr, nullptr, 0U))
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: invalid response\n");
            return SynchronizationResult_Unknown;
        }
        auto result = ConvertErrorCodeToSynchronizationResult(error);
        if (result == SynchronizationResult_Success)
        {
            NNT_LDN_LOG_DEBUG("Succeeded to sync\n");
        }
        else
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: %d\n", result);
        }
        return result;
    }

    int HtcsSynchronizationServer::SendLeaveRequest() NN_NOEXCEPT
    {
        // グループから離脱するリクエストを生成します。
        size_t dataSize;
        detail::BuildLeaveRequest(m_Buffer->send, &dataSize, sizeof(m_Buffer->send));

        // リクエストを送信します。
        NNT_LDN_LOG_DEBUG("Leave\n");
        if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
        {
            return SynchronizationResult_Unknown;
        }
        return SynchronizationResult_Success;
    }

    HtcsSynchronizationClient::HtcsSynchronizationClient(
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(static_cast<detail::HtcsSynchronizationClientBuffer*>(buffer)),
          m_Socket(-1),
          m_ClientCount(0),
          m_DataSize(0U),
          m_Timeout(nn::TimeSpan::FromSeconds(60)),
          m_IsRunning(false)
    {
        NN_ASSERT(sizeof(detail::HtcsSynchronizationClientBuffer) <= bufferSize);
        NN_UNUSED(bufferSize);
        nn::htcs::Initialize(m_Buffer->htcs, sizeof(m_Buffer->htcs));
    }

    HtcsSynchronizationClient::~HtcsSynchronizationClient() NN_NOEXCEPT
    {
        if (m_IsRunning)
        {
            Disconnect();
        }
        nn::htcs::Finalize();
    }

    int HtcsSynchronizationClient::SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_ASSERT(nn::TimeSpan() < timeout);
        m_Timeout = timeout;
        return SynchronizationResult_Success;
    }

    int HtcsSynchronizationClient::Connect(const char* groupName) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(groupName);
        NN_SDK_ASSERT_MINMAX(std::strlen(groupName), 1U, SynchronizationGroupNameLengthMax);

        // 同期に必要な情報を記憶しておきます。
        NNT_LDN_LOG_DEBUG("Join: %s\n", groupName);
        auto start = nn::os::GetSystemTick();
        std::strncpy(m_GroupName, groupName, SynchronizationGroupNameLengthMax);

        // Test Bridge に接続します。
        int result = ConnectToTestBridge(&m_Socket, m_Timeout);
        if (result != SynchronizationResult_Success)
        {
            NNT_LDN_LOG_DEBUG("Failed to connect to the test bridge\n");
            return result;
        }

        // グループに参加します。
        auto elapsed = (nn::os::GetSystemTick() - start).ToTimeSpan();
        result = SendJoinGroupRequest(m_Timeout - elapsed);
        if (result == SynchronizationResult_Success)
        {
            m_IsRunning = true;
        }
        else
        {
            nn::htcs::Close(m_Socket);
        }
        return result;
    }

    int HtcsSynchronizationClient::Disconnect() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        SendLeaveRequest();
        nn::htcs::Shutdown(m_Socket, nn::htcs::HTCS_SHUT_RDWR);
        nn::htcs::Close(m_Socket);
        m_IsRunning = false;
        m_DataSize = 0U;
        return SynchronizationResult_Success;
    }

    int HtcsSynchronizationClient::GetData(
        void* buffer, size_t* pOutSize, size_t bufferSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        if (m_DataSize <= bufferSize)
        {
            if (0U < m_DataSize)
            {
                std::memcpy(buffer, m_Data, m_DataSize);
            }
            *pOutSize = m_DataSize;
            return SynchronizationResult_Success;
        }
        else
        {
            return SynchronizationResult_InsufficientBuffer;
        }
    }

    int HtcsSynchronizationClient::Synchronize(const char* keyword) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(keyword);
        NN_SDK_ASSERT_MINMAX(std::strlen(keyword), 1U, SynchronizationKeywordLengthMax);
        NNT_LDN_LOG_DEBUG("Sync: %s\n", keyword);
        return SendSyncRequest(keyword);
    }

    int HtcsSynchronizationClient::GetClientCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        return m_ClientCount;
    }

    int HtcsSynchronizationClient::GetClientIndex() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        return m_ClientIndex;
    }

    int HtcsSynchronizationClient::SendJoinGroupRequest(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // グループの生成に必要なリクエストを生成します。
        size_t dataSize;
        detail::BuildJoinGroupRequest(
            m_Buffer->send, &dataSize, sizeof(m_Buffer->send), m_GroupName);

        // タイムアウトするまで繰り返します。
        nn::TimeSpan elapsed;
        auto start = nn::os::GetSystemTick();
        do
        {
            // リクエストを送信します。
            if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
            {
                NNT_LDN_LOG_DEBUG("Failed to join: send error\n");
                return SynchronizationResult_Unknown;
            }

            // レスポンスの受信まで待機します。
            if (Recv(m_Socket, m_Buffer->recv, sizeof(m_Buffer->recv)) <= 0)
            {
                NNT_LDN_LOG_DEBUG("Failed to join: receive error\n");
                return SynchronizationResult_Unknown;
            }

            // レスポンスの正当性を確認します。
            int clientCount;
            int clientIndex;
            detail::SynchronizationErrorCode error;
            if (!detail::ParseJoinGroupResponse(
                &error, m_Buffer->recv, &clientCount, &clientIndex))
            {
                NNT_LDN_LOG_DEBUG("Failed to join: invalid response\n");
                return SynchronizationResult_Unknown;
            }
            if (error == detail::SynchronizationErrorCode_Success)
            {
                NNT_LDN_LOG_DEBUG("Succeeded to join\n");
                m_ClientCount = clientCount;
                m_ClientIndex = clientIndex;
                return SynchronizationResult_Success;
            }
            else
            {
                NNT_LDN_LOG_DEBUG("Failed to join: %d\n", error);
            }

            // 登録対象のグループが見つからなかった場合は一定時間待機してから再試行します。
            elapsed = (nn::os::GetSystemTick() - start).ToTimeSpan();
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        } while (elapsed < timeout);
        NNT_LDN_LOG_DEBUG("Failed to join: timeout\n");

        return SynchronizationResult_Timeout;
    }

    int HtcsSynchronizationClient::SendSyncRequest(const char* keyword) NN_NOEXCEPT
    {
        // 同期に必要なリクエストを生成します。
        size_t dataSize;
        detail::BuildSyncRequest(
            m_Buffer->send, &dataSize, sizeof(m_Buffer->send),
            keyword, nullptr, 0U, m_Timeout);

        // リクエストを送信します。
        if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: send error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスの受信まで待機します。
        if (Recv(m_Socket, m_Buffer->recv, sizeof(m_Buffer->recv)) <= 0)
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: receive error\n");
            return SynchronizationResult_Unknown;
        }

        // レスポンスを解析します。
        detail::SynchronizationErrorCode error;
        if (!detail::ParseSyncResponse(
            &error, m_Buffer->recv, m_Data, &m_DataSize, sizeof(m_Data)))
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: invalid response\n");
            return SynchronizationResult_Unknown;
        }
        auto result = ConvertErrorCodeToSynchronizationResult(error);
        if (result == SynchronizationResult_Success)
        {
            NNT_LDN_LOG_DEBUG("Succeeded to sync\n");
        }
        else
        {
            NNT_LDN_LOG_DEBUG("Failed to sync: %d\n", result);
        }
        return result;
    }

    int HtcsSynchronizationClient::SendLeaveRequest() NN_NOEXCEPT
    {
        // グループから離脱するリクエストを生成します。
        size_t dataSize;
        detail::BuildLeaveRequest(m_Buffer->send, &dataSize, sizeof(m_Buffer->send));

        // リクエストを送信します。
        if (nn::htcs::Send(m_Socket, m_Buffer->send, dataSize, 0) != static_cast<int>(dataSize))
        {
            return SynchronizationResult_Unknown;
        }
        return SynchronizationResult_Success;
    }

}} // namespace nnt::ldn
