﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nnt/ldn/testLdn_Config.h>
#include <nnt/ldn/detail/testLdn_XmlParser.h>

#if defined(NNT_LDN_BUILD_CONFIG_USE_PUGIXML)
    #include <pugixml.hpp>
#endif

namespace nnt { namespace ldn { namespace detail { namespace
{
    #if defined(NNT_LDN_BUILD_CONFIG_USE_PUGIXML)
    inline XmlNode* ToHandle(pugi::xml_node impl) NN_NOEXCEPT
    {
        return *reinterpret_cast<XmlNode**>(&impl);
    }
    inline const pugi::xml_node ToImpl(const XmlNode* pNode) NN_NOEXCEPT
    {
        return *reinterpret_cast<const pugi::xml_node*>(&pNode);
    }
    class Pred
    {
    public:

        explicit Pred(const nn::util::string_view name) NN_NOEXCEPT
            : m_Name(name)
        {
        }

        bool operator()(pugi::xml_node node) const NN_NOEXCEPT
        {
            return m_Name.compare(node.name()) == 0;
        }

    private:

        const nn::util::string_view m_Name;
    };
    #endif

}}}} // namespace nnt::ldn::detail::<unnamed>

namespace nnt { namespace ldn { namespace detail
{
    #if defined(NNT_LDN_BUILD_CONFIG_USE_PUGIXML)
    class XmlParser::Impl
    {
    public:

        Impl() NN_NOEXCEPT
            : m_pDocument(new pugi::xml_document())
        {
        }

        ~Impl() NN_NOEXCEPT
        {
            delete m_pDocument;
            m_pDocument = nullptr;
        }

        bool Parse(char* document) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(document);
            pugi::xml_parse_result result = m_pDocument->load_buffer_inplace(
                document, std::strlen(document), pugi::parse_default, pugi::encoding_utf8);
            return result.status == pugi::status_ok;
        }

        void Clear() NN_NOEXCEPT
        {
            m_pDocument->reset();
        }

        XmlNode* GetRoot() NN_NOEXCEPT
        {
            return ToHandle(m_pDocument->document_element());
        }

    private:

        pugi::xml_document* m_pDocument;
    };

    nn::util::string_view XmlNode::GetName() const NN_NOEXCEPT
    {
        return ToImpl(this).name();
    }

    nn::util::string_view XmlNode::GetText() const NN_NOEXCEPT
    {
        return ToImpl(this).child_value();
    }

    const XmlNode* XmlNode::FindChild(const nn::util::string_view& name) const NN_NOEXCEPT
    {
        return ToHandle(ToImpl(this).find_child(Pred(name)));
    }
    #endif

    XmlParser::XmlParser() NN_NOEXCEPT
        : m_pImpl(new Impl())
    {
    }

    XmlParser::~XmlParser() NN_NOEXCEPT
    {
        delete m_pImpl;
        m_pImpl = nullptr;
    }

    bool XmlParser::Parse(char* document) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(document);
        return m_pImpl->Parse(document);
    }

    void XmlParser::Clear() NN_NOEXCEPT
    {
        return m_pImpl->Clear();
    }

    const XmlNode* XmlParser::GetRoot() const NN_NOEXCEPT
    {
        return m_pImpl->GetRoot();
    }

}}} // namespace nnt::ldn::detail
