﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nnt/ldn/detail/testLdn_XmlBuilder.h>

namespace nnt { namespace ldn { namespace detail { namespace
{
    const char XmlHeader[] = "<?xml version=\"1.0\" encoding=\"utf-8\"?>";

    const char HexTable[] =
    {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        'a', 'b', 'c', 'd', 'e', 'f'
    };

}}}} // namespace nnt::ldn::detail::<unnamed>

namespace nnt { namespace ldn { namespace detail
{
    XmlBuilder::XmlBuilder(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(static_cast<char*>(buffer)),
          m_MaxSize(bufferSize - 2U),
          m_Pos(0U)
    {
        NN_UNUSED(m_MaxSize);
        NN_ASSERT(sizeof(XmlHeader) < bufferSize);
        Clear();
    }

    XmlBuilder::~XmlBuilder() NN_NOEXCEPT
    {
    }

    void XmlBuilder::Clear() NN_NOEXCEPT
    {
        std::strncpy(m_Buffer, XmlHeader, m_MaxSize);
        m_Pos = std::strlen(XmlHeader);
        Terminate();
    }

    void XmlBuilder::OpenTag(const char* tag) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(tag);
        size_t nameSize = std::strlen(tag);
        size_t tagSize = nameSize + 2U;
        NN_ASSERT(m_Pos + tagSize <= m_MaxSize);
        m_Buffer[m_Pos] = '<';
        std::memcpy(m_Buffer + m_Pos + 1U, tag, nameSize);
        m_Buffer[m_Pos + 1U + nameSize] = '>';
        m_Pos += tagSize;
        Terminate();
    }

    void XmlBuilder::CloseTag(const char* tag) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(tag);
        size_t nameSize = std::strlen(tag);
        size_t tagSize = nameSize + 3U;
        NN_ASSERT(m_Pos + tagSize <= m_MaxSize);
        m_Buffer[m_Pos] = '<';
        m_Buffer[m_Pos + 1U] = '/';
        std::memcpy(m_Buffer + m_Pos + 2U, tag, nameSize);
        m_Buffer[m_Pos + 2U + nameSize] = '>';
        m_Pos += tagSize;
        Terminate();
    }

    void XmlBuilder::Add(const char* text) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(text);
        size_t textSize = std::strlen(text);
        NN_ASSERT(m_Pos + textSize <= m_MaxSize);
        std::memcpy(m_Buffer + m_Pos, text, textSize);
        m_Pos += textSize;
        Terminate();
    }

    void XmlBuilder::Add(int data) NN_NOEXCEPT
    {
        char buffer[16];
        nn::util::SNPrintf(buffer, sizeof(buffer), "%d", data);
        Add(buffer);
    }

    void XmlBuilder::Add(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(data);
        NN_ASSERT(m_Pos + dataSize * 2 <= m_MaxSize);
        const uint8_t* p = static_cast<const uint8_t*>(data);
        for (size_t i = 0; i < dataSize; ++i)
        {
            uint8_t high = (p[i] >> 4) & 0x0F;
            uint8_t low = p[i] & 0x0F;
            m_Buffer[m_Pos + 0] = HexTable[high];
            m_Buffer[m_Pos + 1] = HexTable[low];
            m_Pos += 2;
        }
        Terminate();
    }

    const char* XmlBuilder::GetData() const NN_NOEXCEPT
    {
        return m_Buffer;
    }

    size_t XmlBuilder::GetDataSize() const NN_NOEXCEPT
    {
        return m_Pos + 1U;
    }

    void XmlBuilder::Terminate() NN_NOEXCEPT
    {
        m_Buffer[m_Pos + 0] = '\n';
        m_Buffer[m_Pos + 1] = '\0';
    }

}}} // namespace nnt::ldn::detail
