﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/detail/testLdn_SynchronizationInterpreter.h>
#include <nnt/ldn/detail/testLdn_XmlBuilder.h>
#include <nnt/ldn/detail/testLdn_XmlParser.h>

namespace nnt { namespace ldn { namespace detail { namespace
{
    int ConvertHexCharToInteger(char ch) NN_NOEXCEPT
    {
        if ('0' <= ch && ch <= '9')
        {
            return ch - '0';
        }
        else if ('a' <= ch && ch <= 'f')
        {
            return ch - 'a' + 10;
        }
        else if ('A' <= ch && ch <= 'F')
        {
            return ch - 'A' + 10;
        }
        else
        {
            return 0;
        }
    }

    /**
     * @brief       XML ノードから整数値を取得します。
     * @param[out]  pOut        整数値の出力先です。
     * @param[in]   pNode       対象の数字文字列を InnerText として含む XML ノードです。
     * @return      整数値の取得に成功した場合は true です。
     */
    bool GetInteger(int* pOut, const XmlNode* pNode) NN_NOEXCEPT
    {
        if (pNode != nullptr)
        {
            char* e;
            int n = std::strtol(pNode->GetText().data(), &e, 10);
            if (*e == 0)
            {
                *pOut = n;
                return true;
            }
        }
        return false;
    }

    /**
    * @brief       XML ノードからバイナリデータを取得します。
    * @param[out]  buffer      バイナリデータの出力先です。
    * @param[out]  dataSize    バイナリデータの出力サイズです。
    * @param[out]  bufferSize  buffer のバッファサイズです。
    * @param[in]   pNode       対象のバイナリデータを InnerText として含む XML ノードです。
    * @return      整数値の取得に成功した場合は true です。
    */
    bool GetData(
        void* buffer, size_t* dataSize, size_t bufferSize, const XmlNode* pNode) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(buffer);
        NN_ASSERT_NOT_NULL(dataSize);
        if (pNode != nullptr)
        {
            auto str = pNode->GetText();
            size_t i;
            for (i = 0; i < bufferSize && i * 2 + 1 < str.length(); ++i)
            {
                uint8_t high = static_cast<uint8_t>((ConvertHexCharToInteger(str.at(i * 2))));
                uint8_t low  = static_cast<uint8_t>((ConvertHexCharToInteger(str.at(i * 2 + 1))));
                static_cast<uint8_t*>(buffer)[i] = static_cast<uint8_t>((high << 4) | low);
            }
            if (i * 2 == str.length())
            {
                *dataSize = i;
                return true;
            }
        }
        return false;
    }

    /**
     * @brief       XML ノードからエラーコードを取得します。
     * @param[out]  pOut        エラーコードの出力先です。
     * @param[in]   pNode       エラーコードを InnerText として含む XML ノードです。
     * @return      整数値の取得に成功した場合は true です。
     */
    bool GetResult(SynchronizationErrorCode* pOut, const XmlNode* pNode) NN_NOEXCEPT
    {
        if (pNode != nullptr)
        {
            if (pNode->GetName().compare("Result") == 0)
            {
                int result;
                if (GetInteger(&result, pNode))
                {
                    *pOut = static_cast<SynchronizationErrorCode>(result);
                    return true;
                }
            }
        }
        return false;
    }

}}}} // namespace nnt::ldn::detail::<unnamed>

namespace nnt { namespace ldn { namespace detail
{
    void BuildCreateGroupRequest(
        void* buffer, size_t* pOutSize, size_t bufferSize,
        const char* groupName, int clientCount) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(buffer);
        NN_ASSERT_NOT_NULL(pOutSize);
        NN_ASSERT_NOT_NULL(groupName);
        NN_ASSERT(0 < clientCount);
        XmlBuilder builder(buffer, bufferSize);
        builder.OpenTag("CreateGroupRequest");
        builder.OpenTag("GroupName");
        builder.Add(groupName);
        builder.CloseTag("GroupName");
        builder.OpenTag("ClientCount");
        builder.Add(clientCount);
        builder.CloseTag("ClientCount");
        builder.CloseTag("CreateGroupRequest");
        *pOutSize = builder.GetDataSize();
    }

    void BuildJoinGroupRequest(
        void* buffer, size_t* pOutSize, size_t bufferSize,
        const char* groupName) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(buffer);
        NN_ASSERT_NOT_NULL(pOutSize);
        NN_ASSERT_NOT_NULL(groupName);
        XmlBuilder builder(buffer, bufferSize);
        builder.OpenTag("JoinGroupRequest");
        builder.OpenTag("GroupName");
        builder.Add(groupName);
        builder.CloseTag("GroupName");
        builder.CloseTag("JoinGroupRequest");
        *pOutSize = builder.GetDataSize();
    }

    void BuildSyncRequest(
        void* buffer, size_t* pOutSize, size_t bufferSize, const char* keyword,
        const void* data, size_t dataSize, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(buffer);
        NN_ASSERT_NOT_NULL(pOutSize);
        NN_ASSERT_NOT_NULL(keyword);
        XmlBuilder builder(buffer, bufferSize);
        builder.OpenTag("SyncRequest");
        builder.OpenTag("Keyword");
        builder.Add(keyword);
        builder.CloseTag("Keyword");
        if (0U < dataSize)
        {
            builder.OpenTag("Data");
            builder.Add(data, dataSize);
            builder.CloseTag("Data");
        }
        builder.OpenTag("Timeout");
        builder.Add(static_cast<int>(timeout.GetMilliSeconds()));
        builder.CloseTag("Timeout");
        builder.CloseTag("SyncRequest");
        *pOutSize = builder.GetDataSize();
    }

    void BuildLeaveRequest(
        void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(buffer);
        NN_ASSERT_NOT_NULL(pOutSize);
        XmlBuilder builder(buffer, bufferSize);
        builder.OpenTag("LeaveRequest");
        builder.CloseTag("LeaveRequest");
        *pOutSize = builder.GetDataSize();
    }

    bool ParseCreateGroupResponse(
        SynchronizationErrorCode* pOutError, char* xml) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(xml);

        // XML をパースします。
        XmlParser parser;
        if (!parser.Parse(xml))
        {
            NNT_LDN_LOG_WARN("failed to parse: %s\n", xml);
            return false;
        }

        // 該当のレスポンスであることを確認します。
        const XmlNode* root = parser.GetRoot();
        if (root->GetName().compare("CreateGroupResponse") != 0)
        {
            NNT_LDN_LOG_WARN("invalid response: %s\n", xml);
            return false;
        }

        // レスポンスから処理の結果を取得します。
        if (!GetResult(pOutError, root->FindChild("Result")))
        {
            NNT_LDN_LOG_WARN("failed to get result: %s\n", xml);
            return false;
        }
        return true;
    }

    bool ParseJoinGroupResponse(
        SynchronizationErrorCode* pOutError, char* xml,
        int* pOutClientCount, int* pOutClientIndex) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(xml);
        NN_ASSERT_NOT_NULL(pOutClientCount);
        NN_ASSERT_NOT_NULL(pOutClientIndex);

        // XML をパースします。
        XmlParser parser;
        if (!parser.Parse(xml))
        {
            NNT_LDN_LOG_WARN("failed to parse: %s\n", xml);
            return false;
        }

        // 該当のレスポンスであることを確認します。
        const XmlNode* root = parser.GetRoot();
        if (root->GetName().compare("JoinGroupResponse") != 0)
        {
            NNT_LDN_LOG_WARN("invalid response: %s\n", xml);
            return false;
        }

        // レスポンスから処理の結果を取得します。
        if (!GetResult(pOutError, root->FindChild("Result")))
        {
            NNT_LDN_LOG_WARN("failed to get result: %s\n", xml);
            return false;
        }

        // 成功した場合には詳細を取得します。
        if (*pOutError == SynchronizationErrorCode_Success)
        {
            if (!GetInteger(pOutClientCount, root->FindChild("ClientCount")))
            {
                NNT_LDN_LOG_WARN("failed to get client count: %s\n", xml);
                return false;
            }
            if (!GetInteger(pOutClientIndex, root->FindChild("ClientIndex")))
            {
                NNT_LDN_LOG_WARN("failed to get client index: %s\n", xml);
                return false;
            }
        }
        return true;
    }

    bool ParseSyncResponse(
        SynchronizationErrorCode* pOutError, char* xml,
        void* buffer, size_t* dataSize, size_t bufferSize) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(xml);

        // XML をパースします。
        XmlParser parser;
        if (!parser.Parse(xml))
        {
            NNT_LDN_LOG_WARN("failed to parse: %s\n", xml);
            return false;
        }

        // 該当のレスポンスであることを確認します。
        const XmlNode* root = parser.GetRoot();
        if (root->GetName().compare("SyncResponse") != 0)
        {
            NNT_LDN_LOG_WARN("invalid response: %s\n", xml);
            return false;
        }

        // レスポンスから処理の結果を取得します。
        if (!GetResult(pOutError, root->FindChild("Result")))
        {
            NNT_LDN_LOG_WARN("failed to get result: %s\n", xml);
            return false;
        }

        // 成功した場合には詳細を取得します。
        if (*pOutError == SynchronizationErrorCode_Success)
        {
            if (buffer != nullptr && dataSize != nullptr && 0U < bufferSize)
            {
                if (!GetData(buffer, dataSize, bufferSize, root->FindChild("Data")))
                {
                    *dataSize = 0;
                }
            }
        }
        return true;
    }

}}} // namespace nnt::ldn::detail
