﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "../Common/Params.h"
#include "../Common/Precondition.h"

namespace
{
    // テストの設定値です。
    nnt::lcs::TestConfig g_TestConfig;

    // LCS/Socket の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // EULA 取得用のバッファです。
    char g_EulaBuffer[1 * 1024 * 1024];

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronizationClient* g_pClient;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pClient->Synchronize(keyword))

    // テストを実施する条件を指定するためのマクロです。
    #define NNT_LDN_REQUIRES_NODE_COUNT(count) \
        if (g_TestConfig.nodeCount < (count))\
        {\
            return;\
        }

} // namespace <unnamed>

//
// ホストしかセッションに参加していない状態でスキャンします。
//
TEST(Scan, OnlyHost)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Scan.OnlyHost.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching05.value; id += 0x1000)
    {
        // テストの事前条件を取得します。
        const auto& cond = GetPrecondition("Scan", "OnlyHost", id);
        const auto& host = cond.nodes[0];

        // セッションの構築を待機します。
        NNT_LDN_SYNC("Scan.OnlyHost.SessionOpened");

        // セッションをスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        NNT_LDN_SYNC("Scan.OnlyHost.ScanCompleted");

        // スキャン結果の正当性を確認します。
        const auto& session = sessions[index];
        const auto& content = session.contents[0];
        ASSERT_STREQ(nnt::lcs::DefaultHostUserName, session.hostUserName);
        ASSERT_EQ(1, session.contentsCount);
        ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::System>());
        ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::Application>());
        ASSERT_EQ(true, content.contentsFlag.Test<nn::lcs::ContentsType::Patch>());
        ASSERT_EQ(0U, content.attributeFlag);
        ASSERT_STREQ(host.application.displayVersion, content.displayVersion);
        ASSERT_EQ(id, content.applicationId.value);
        ASSERT_EQ(nn::lcs::NodeCountMax, session.nodeCountMax);
        ASSERT_EQ(1, session.nodeCount);
        ASSERT_GE(session.linkLevel, nn::lcs::LinkLevelMin);
        ASSERT_LE(session.linkLevel, nn::lcs::LinkLevelMax);
    }
}

//
// ホストとクライアント 1 台がセッションに参加している状態でスキャンします。
//
TEST(Scan, HostAndClient)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(3);
    NNT_LDN_SYNC("Scan.HostAndClient.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching04.value; id += 0x1000)
    {
        // テストの事前条件を取得します。
        const auto& cond = GetPrecondition("Scan", "HostAndClient", id);

        // ホストとクライアントのうち、最新のバージョンを計算します。
        char displayVersion[16];
        const int version = std::max(
            cond.nodes[0].application.version, cond.nodes[1].application.version);
        nnt::lcs::ConvertToDisplayVersion(displayVersion, sizeof(displayVersion), version);

        // セッションの構築を待機します。
        NNT_LDN_SYNC("Scan.HostAndClient.SessionOpened");

        // 先頭のクライアントであれば接続し、それ以外のクライアントはスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        if (g_TestConfig.nodeIndex == 1)
        {
            // 接続先のセッションを見つけるためにスキャンします。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                sessions, &count, &index, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));
            NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedBeforeJoin");

            // セッションに接続します。
            auto& session = sessions[index];
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            NNT_LDN_SYNC("Scan.HostAndClient.Joined");

            // 他のクライアントによるスキャンを待機します。
            NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedAfterJoin");
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        }
        else
        {
            // クライアントの接続が完了するまで待機します。
            NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedBeforeJoin");
            NNT_LDN_SYNC("Scan.HostAndClient.Joined");

            // 即座にアドバータイズに反映されない可能性があるため、一定時間待機します。
            nn::os::SleepThread(StateChangeTimeout);

            // セッションをスキャンします。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                sessions, &count, &index, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));
            NNT_LDN_SYNC("Scan.HostAndClient.ScanCompletedAfterJoin");

            // スキャン結果の正当性を確認します。
            const auto& session = sessions[index];
            const auto& content = session.contents[0];
            ASSERT_STREQ(nnt::lcs::DefaultHostUserName, session.hostUserName);
            ASSERT_EQ(1, session.contentsCount);
            ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::System>());
            ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::Application>());
            ASSERT_EQ(true, content.contentsFlag.Test<nn::lcs::ContentsType::Patch>());
            ASSERT_EQ(0U, content.attributeFlag);
            ASSERT_STREQ(displayVersion, content.displayVersion);
            ASSERT_EQ(id, content.applicationId.value);
            ASSERT_EQ(nn::lcs::NodeCountMax, session.nodeCountMax);
            ASSERT_EQ(2, session.nodeCount);
            ASSERT_GE(session.linkLevel, nn::lcs::LinkLevelMin);
            ASSERT_LE(session.linkLevel, nn::lcs::LinkLevelMax);
        }
    }
}

//
// 複数のセッションが存在する状態で、バッファ数 1 でスキャンします。
// このテストだけホスト、クライアントの役割が逆転します。
//
TEST(Scan, BufferSizeMin)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Scan.BufferSizeMin.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 対象のアプリケーションを指定します。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);

    // セッションを構築します。
    nnt::lcs::HostStarter starter(settings);
    NNT_LDN_SYNC("Scan.BufferSizeMin.SessionOpened");

    // クライアントによるスキャンが完了するまで待機します。
    NNT_LDN_SYNC("Scan.BufferSizeMin.ScanCompleted");
}

//
// 最大端末数による接続制限を確認します。
//
TEST(Restrict, NodeCountMax)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Restrict.NodeCountMax.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Restrict.NodeCountMax.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    NNT_LDN_SYNC("Restrict.NodeCountMax.ScanCompleted");

    // セッションへの接続を試行します。
    // 接続台数制限により、接続できないクライアントもあります。
    auto& session = sessions[index];
    nnt::lcs::TimedJoin(session, JoinTimeout);
    NNT_LDN_SYNC("Restrict.NodeCountMax.Joined");

    // テストの終了まで待機します。
    NNT_LDN_SYNC("Restrict.NodeCountMax.SessionVerified");
    NNT_LDN_SYNC("Restrict.NodeCountMax.Finished");
}

//
// 最大端末数による接続制限を確認します。
//
TEST(Restrict, AcceptPolicyAlwaysReject)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.ScanCompleted");

    // セッションへの接続を試行します。AcceptPoilcy により失敗します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultCommunicationError,
        nnt::lcs::TimedJoin(session, JoinTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Joined");

    // テストの終了まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Finished");
}

//
// ホストのフライトモードによるマッチング失敗を確認します。
//
TEST(MatchingFailure, FlightMode)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("MatchingFailure.FlightMode.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("MatchingFailure.FlightMode.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Joined");

    // セッションが破棄されるまで待機します。
    while (nn::lcs::GetState() == nn::lcs::State_Joined)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved,
              nn::lcs::GetContentsShareFailureReason());
    NNT_LDN_SYNC("MatchingFailure.FlightMode.Destroyed");
}

//
// ホストのセッション破棄によるマッチング失敗を確認します。
//
TEST(MatchingFailure, DestroySession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.DestroySession.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("MatchingFailure.DestroySession.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("MatchingFailure.DestroySession.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("MatchingFailure.DestroySession.Joined");

    // セッションが破棄されるまで待機します。
    while (nn::lcs::GetState() == nn::lcs::State_Joined)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved,
              nn::lcs::GetContentsShareFailureReason());

    // テストの終了まで待機します。
    NNT_LDN_SYNC("MatchingFailure.DestroySession.Destroyed");
}

//
// ホストの Finalize() によるマッチング失敗を確認します。
//
TEST(MatchingFailure, HostFinalize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("MatchingFailure.HostFinalize.Joined");

    // セッションが破棄されるまで待機します。
    while (nn::lcs::GetState() == nn::lcs::State_Joined)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved,
              nn::lcs::GetContentsShareFailureReason());
}

//
// LeaveSession によるクライアントの離脱を検証します。
//
TEST(ClientLeave, LeaveSession)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("ClientLeave.LeaveSession.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("ClientLeave.LeaveSession.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Joined");

    // セッションから離脱します。
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    NNT_LDN_SYNC("ClientLeave.LeaveSession.Disconnected");

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None,
              nn::lcs::GetContentsShareFailureReason());
}

//
// Finalize によるクライアントの離脱を検証します。
//
TEST(ClientLeave, Finalize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.Finalize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::Initialize(g_Buffer, sizeof(g_Buffer), config));

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("ClientLeave.Finalize.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("ClientLeave.Finalize.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("ClientLeave.Finalize.Joined");

    // セッションから離脱します。
    nn::lcs::Finalize();

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_EQ(nn::lcs::State_None, nn::lcs::GetState());
    NNT_LDN_SYNC("ClientLeave.Finalize.Disconnected");
}

//
// Finalize によるクライアントの離脱を検証します。
//
TEST(ClientLeave, FlightMode)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("ClientLeave.FlightMode.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer initializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("ClientLeave.FlightMode.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("ClientLeave.FlightMode.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("ClientLeave.FlightMode.Joined");

    // セッションから離脱します。
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nnt::ldn::ScopedWirelessCommunicationSwitchSetter wifi(false);

    // 失敗理由を正しく取得できることを確認します。
    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_FlightMode,
              nn::lcs::GetContentsShareFailureReason());
    NNT_LDN_SYNC("ClientLeave.FlightMode.Disconnected");
}

//
// GetNodes() が正しく機能することを検証します。
//
TEST(Monitor, GetNodes)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Monitor.GetNodes.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer initializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Monitor.GetNodes.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("Monitor.GetNodes.ScanCompleted");

    // インデックスが若い順に 1 台ずつセッションに参加します。
    for (int expectedCount = 2; expectedCount <= g_TestConfig.nodeCount; ++expectedCount)
    {
        if (g_TestConfig.nodeIndex < expectedCount)
        {
            // インデックスによって接続するか待機するかを決めます。
            if (g_TestConfig.nodeIndex == expectedCount - 1)
            {
                const auto& session = sessions[index];
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            }
            else
            {
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(
                    &stateChangeEvent, JoinTimeout + StateChangeTimeout));
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // 現在接続中のクライアントを確認します。
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(
                nodes, &actualCount, nn::lcs::NodeCountMax));
            ASSERT_EQ(expectedCount, actualCount);
            ASSERT_NE(0, nodes[0].index);
            ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
            for (int i = 1; i < actualCount; ++i)
            {
                ASSERT_NE(0, nodes[i].index);
                ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
            }
            ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[g_TestConfig.nodeIndex].index);
        }
        NNT_LDN_SYNC("Monitor.GetNodes.Joined");
    }

    // インデックスが大きい順に 1 台ずつセッションから離脱します。
    for (int expectedCount = g_TestConfig.nodeCount - 1; expectedCount >= 1; --expectedCount)
    {
        // インデックスによって離脱するか待機するかを決めます。
        if (g_TestConfig.nodeIndex <= expectedCount)
        {
            if (g_TestConfig.nodeIndex == expectedCount)
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            }
            else
            {
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(
                    &stateChangeEvent, JoinTimeout + StateChangeTimeout));

                // 現在接続中のクライアントを確認します。
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount;
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(
                    nodes, &actualCount, nn::lcs::NodeCountMax));
                ASSERT_EQ(expectedCount, actualCount);
                ASSERT_NE(0, nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[g_TestConfig.nodeIndex].index);
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
        }
        NNT_LDN_SYNC("Monitor.GetNodes.Disconnected");
    }
}

//
// GetNodes() が正しく機能することを検証します。
//
TEST(Monitor, GetRequiredStorageSize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer initializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 複数のアプリケーションに対してテストを繰り返します。
    for (uint64_t id = nnt::lcs::Application::Matching01.value;
         id <= nnt::lcs::Application::Matching04.value; id += 0x1000)
    {
        // テストの事前条件を取得します。
        const auto& cond = GetPrecondition("Monitor", "GetRequiredStorageSize", id);
        const auto& app = cond.application;

        // セッションの構築を待機します。
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.SessionOpened");

        // 接続先のセッションをスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.ScanCompleted");

        // セッションに接続します。
        const auto& session = sessions[index];
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.Joined");

        // すぐには情報が更新されない可能性があるため、少し待機します。
        nn::os::SleepThread(StateChangeTimeout);

        // 自身が最新のパッチをもっていない場合には必要サイズが 1 以上になります。
        size_t requiredSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredSize));
        int version = cond.nodes[g_TestConfig.nodeIndex].application.version;
        if (version == app.latestVersion)
        {
            ASSERT_EQ(0U, requiredSize);
        }
        else
        {
            ASSERT_GT(requiredSize, 0U);
        }
        NNT_LDN_SYNC("Monitor.GetRequiredStorageSize.ReadyToDestroy");

        // セッションから離脱します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    }
}

//
// GetProgress() が正しく機能することを検証します。
//
TEST(InvalidState, Joined)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("InvalidState.Joined.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer initializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("InvalidState.Joined.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_LDN_SYNC("InvalidState.Joined.ScanCompleted");

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
    NNT_LDN_SYNC("InvalidState.Joined.Joined");

    // Joined 状態では GetProgress() は実行できません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetProgress(&progress));

    // Joined 状態では GetNodeProgress() は実行できません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetNodeProgress(&nodeProgress, nn::lcs::GetMyIndex()));

    // Joined 状態では GetDownloadedContents() は実行できません。
    nn::lcs::ContentsInfo content;
    int contentCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedContents(&content, &contentCount, 1));

    // Joined 状態では GetDownloadedEulaDataSize() は実行できません。
    size_t eulaSize;
    const char* path = "JPja/Eula.msbt.szs";
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaDataSize(&eulaSize, path));

    // Joined 状態では GetDownloadedEulaData() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaData(&eulaSize, g_EulaBuffer, sizeof(g_EulaBuffer), path));

    // Joined 状態では GetSessionContext() は実行できません。
    nn::lcs::SessionContext context;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetSessionContext(&context));

    // Joined 状態では ResumeContentsShare() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeContentsShare());

    // Joined 状態では ResumeSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeSession(context));

    // Joined 状態では GetContentsShareFailureReason() は実行できません。
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None, nn::lcs::GetContentsShareFailureReason());

    // Joined 状態では GetSuspendedReason() は実行できません。
    ASSERT_EQ(nn::lcs::SuspendedReason_None, nn::lcs::GetSuspendedReason());

    // Joined 状態では OpenSession() は実行できません。
    const auto id = DefaultApplicationId;
    const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::OpenSession(settings));

    // Joined 状態では SetClientAcceptPolicy() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysReject));

    // Joined 状態では StartContentsShare() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::StartContentsShare());

    // Joined 状態では Scan() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::Scan(sessions, &count, nn::lcs::ScanResultCountMax));

    // Joined 状態では JoinSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::JoinSession(session));

    NNT_LDN_SYNC("InvalidState.Joined.ReadyToDestroy");
}

//
// ホストのプロトコルに互換性がある場合の挙動を確認します。
//
TEST(Join, CompatibleProtocolVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.Start");

    // NS, LDN ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.SessionOpened");

    // テストで使用するプロトコルバージョンを定義します。
    int versions[][3] = { {2, 0}, {2, 1}, {2, 15} };
    for (size_t i = 0; i < sizeof(versions) / sizeof(versions[0]); ++i)
    {
        // プロトコルバージョンを設定します。
        const int majorVersion = versions[i][0];
        const int minorVersion = versions[i][1];
        nnt::lcs::ProtocolVersionSetter protocolVersionSetter(majorVersion, minorVersion);

        // LCS ライブラリを初期化します。
        nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
        nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

        // イベントを取得します。
        nn::os::SystemEventType stateChangeEvent;
        nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 接続先のセッションをスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // セッションに接続します。
        const auto& session = sessions[index];
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());

        // すぐに離脱します。
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    }
    NNT_LDN_SYNC("Join.CompatibleProtocolVersion.ReadyToDestroy");
}

//
// ホストのプロトコルに互換性が無い場合の挙動を確認します。
//
TEST(Join, IncompatibleProtocolVersion)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.Start");

    // NS, LDN ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.SessionOpened");

    // テストで使用するプロトコルバージョンを定義します。
    int versions[][2] = { {1, 15}, {3,1}, {4, 0} };
    for (size_t i = 0; i < sizeof(versions) / sizeof(versions[0]); ++i)
    {
        // プロトコルバージョンを設定します。
        const int majorVersion = versions[i][0];
        const int minorVersion = versions[i][1];
        nnt::lcs::ProtocolVersionSetter protocolVersionSetter(majorVersion, minorVersion);

        // LCS ライブラリを初期化します。
        nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
        nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

        // イベントを取得します。
        nn::os::SystemEventType stateChangeEvent;
        nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 接続先のセッションをスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // セッションに接続します。
        const auto& session = sessions[index];
        if (majorVersion < 2)
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::lcs::ResultLowerVersion,
                nnt::lcs::TimedJoin(session, JoinTimeout));
        }
        else if (2 < majorVersion)
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::lcs::ResultHigherVersion,
                nnt::lcs::TimedJoin(session, JoinTimeout));
        }
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    }
    NNT_LDN_SYNC("Join.IncompatibleProtocolVersion.ReadyToDestroy");
}

//
// クライアントがアプリケーションを所有していない場合の挙動を確認します。
//
TEST(Join, ClientDoesNotHaveApplication)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションの構築を待機します。
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.SessionOpened");

    // 接続先のセッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションに接続します。
    const auto& session = sessions[index];
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultNoSharableContentsSession,
        nnt::lcs::TimedJoin(session, JoinTimeout));
    NNT_LDN_SYNC("Join.ClientDoesNotHaveApplication.ReadyToDestroy");
}

//
// クライアントがアプリケーションを所有していない場合の挙動を確認します。
//
TEST(Join, MultipleHost)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Join.MultipleHost.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 先頭の端末はセッションを構築し、他の端末はクライアントとして動作します。
    if (g_TestConfig.nodeIndex == 1)
    {
        // 対象のアプリケーションを指定します。
        const auto& app = nnt::lcs::Application::Matching04;
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, app);

        // ホストとしてセッションを構築し、接続を待ち受けます。
        nnt::lcs::HostStarter starter(settings);
        nn::lcs::SessionInfo session;
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_LDN_SYNC("Join.MultipleHost.SessionOpened");
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin1");
        NNT_LDN_SYNC("Join.MultipleHost.Joined1");
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin2");
        NNT_LDN_SYNC("Join.MultipleHost.Joined2");
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
        ASSERT_EQ(g_TestConfig.nodeCount - 1, session.nodeCount);

        // テストの完了まで待機します。
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToDestroy");
    }
    else
    {
        // セッションの構築を待機します。
        NNT_LDN_SYNC("Join.MultipleHost.SessionOpened");

        // 接続先のセッションをスキャンします。
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin1");
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 1 つ目のセッションに接続します。
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_LDN_SYNC("Join.MultipleHost.Joined1");

        // セッションから離脱します。
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToJoin2");
        NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);

        // 接続先のセッションをスキャンします。
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultClientUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        // 2 つ目のセッションに接続します。
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_LDN_SYNC("Join.MultipleHost.Joined2");
        NNT_LDN_SYNC("Join.MultipleHost.ReadyToDestroy");
    }
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_SceneId);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&g_TestConfig, setting, argc, argv);

    // socket ライブラリと ns ライブラリを初期化しておきます。
    nn::socket::Initialize(g_SocketConfig);
    nnt::lcs::NsInitializer nsInitiailzer;

    // テストの準備です。
    nnt::lcs::Setup(g_TestConfig.sceneId);

    // 他のインスタンスと同期するためにテストサーバに接続します。
    nnt::ldn::HtcsSynchronizationClient client(
        g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
    int result = client.Connect("nn::lcs::Integration::Matching");

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        g_pClient = &client;
        if (g_TestConfig.nodeIndex == 0)
        {
            g_TestConfig.nodeIndex = static_cast<int8_t>(client.GetClientIndex());
        }
        if (g_TestConfig.nodeCount == 0)
        {
            g_TestConfig.nodeCount = static_cast<int8_t>(client.GetClientCount() + 1);
        }
        exitCode = RUN_ALL_TESTS();
        client.Disconnect();
    }
    else
    {
        exitCode = result;
    }
    nn::socket::Finalize();
    nnt::Exit(exitCode);
}
