﻿Param(
    [Parameter(Mandatory=$TRUE)][string] $TestName,
    [string] $Build = "Develop",
    [string] $Configuration = "NX-NXFP2-a64",
    [int] $SceneId = -1
)

# スクリプトをロードします。
$SCRIPT_DIR = (Split-Path -Parent $MyInvocation.MyCommand.Path)
$LDN_SCRIPT_DIR = (Convert-Path $SCRIPT_DIR\..\..\..\Ldn\Sources\Scripts)
. $LDN_SCRIPT_DIR\Siglo\Bundle
. $LDN_SCRIPT_DIR\Runner\RunParallel

# パスを設定します。
$SETUP = "$SCRIPT_DIR\Setup"

# テストの NSP ファイルのパスを取得します。
Function TestPath($Name)
{
    $FileName = "testLcs_$Name"
    return "$SIGLO_ROOT\Tests\Outputs\$Configuration\Tests\$FileName\$Build\$FileName.nsp"
}

# テストのセットアップと実行をします。
Function RunTest($ConfigFiles, $HostNspPath, $ClientNspPath)
{
    # SIGLO-70370? のワークアラウンド。
    # テスト結果が失敗かつターゲットへの接続ができなければテスト中に不具合が発生したとみなし、テストを始めからやり直します。
    for ($RetryCount = 1; $RetryCount-- -gt -1;)
    {
        &$SETUP $ConfigFiles
        $ErrorCount = RunParallel $Targets $SceneId $HostNspPath $ClientNspPath @Args

        if ($ErrorCount -eq 0)
        {
            break
        }

        # 接続できないターゲットのリストを取得します。
        $FailedTargets = $Targets | ? { ConnectToTarget $_.Name }
        if (-not $FailedTargets)
        {
            break
        }

        Write-Host "Workaround for SIGLO-70370"

        # 接続できないターゲットの HostBridge を再起動します。
        $FailedTargets | % {
            Stop-SaveHostBridgeLog -HostBridgeAddress $_.Address
            Restart-HostBridge $_.Address
        }
        $FailedTargets | % {
            Wait-HostBridge $_.Address | Write-Host
            Start-SaveHostBridgeLog -HostBridgeAddress $_.Address
        }
    }

    return $ErrorCount
}

# LupEula テストを実行します。
Function LupEula($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationLupEula")
    $ErrorCount = 0

    $ErrorCount += RunTest LupEula.HostToAllClients $Path $Path `
        --gtest_filter=LupEula.HostToAllClients

    $ErrorCount += RunTest LupEula.HostToAllClients $Path $Path `
        --gtest_filter=LupEula.Leave

    &$SETUP LupEula.RestoreEula

    return $ErrorCount
}

# ApplicationRecord テストを実行します。
Function ApplicationRecord($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationApplicationRecord")
    $ErrorCount = 0

    $ErrorCount += RunTest ApplicationRecord.GamecardRemovedWithoutLaunch.DownloadPatchSuccess $Path $Path `
        --gtest_filter=GamecardRemovedWithoutLaunch.DownloadPatchSuccess

    $ErrorCount += RunTest ApplicationRecord.GamecardRemovedAfterLaunch.DownloadPatchSuccess1 $Path $Path `
        --gtest_filter=GamecardRemovedAfterLaunch.DownloadPatchSuccess1

    $ErrorCount += RunTest ApplicationRecord.GamecardRemovedAfterLaunch.DownloadPatchSuccess2 $Path $Path `
        --gtest_filter=GamecardRemovedAfterLaunch.DownloadPatchSuccess2

    $ErrorCount += RunTest ApplicationRecord.GamecardRemovedAfterLaunch.DownloadPatchFailure1 $Path $Path `
        --gtest_filter=GamecardRemovedAfterLaunch.DownloadPatchFailure1

    $ErrorCount += RunTest ApplicationRecord.GamecardRemovedAfterLaunch.DownloadPatchFailure2 $Path $Path `
        --gtest_filter=GamecardRemovedAfterLaunch.DownloadPatchFailure2

    return $ErrorCount
}

# ExFat テストを実行します。
Function ExFat($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationExFat")
    $ErrorCount = 0

    $ErrorCount += RunTest ExFat.MasterHasExFat.PatchRequiredSystemVersion $Path $Path `
        --gtest_filter=MasterHasExFat.PatchRequiredSystemVersion

    $ErrorCount += RunTest ExFat.MasterHasExFat.LcsRequiredSystemVersion $Path $Path `
        --gtest_filter=MasterHasExFat.LcsRequiredSystemVersion

    $ErrorCount += RunTest ExFat.MasterHasExFat.ApplicationDeliveryProtocol $Path $Path `
        --gtest_filter=MasterHasExFat.ApplicationDeliveryProtocol

    $ErrorCount += RunTest ExFat.EveryoneHasExFat.PatchRequiredSystemVersion $Path $Path `
        --gtest_filter=EveryoneHasExFat.PatchRequiredSystemVersion

    $ErrorCount += RunTest ExFat.EveryoneHasExFat.LcsRequiredSystemVersion $Path $Path `
        --gtest_filter=EveryoneHasExFat.LcsRequiredSystemVersion

    $ErrorCount += RunTest ExFat.EveryoneHasExFat.ApplicationDeliveryProtocol $Path $Path `
        --gtest_filter=EveryoneHasExFat.ApplicationDeliveryProtocol

    $ErrorCount += RunTest ExFat.NoOneHasExFat.PatchRequiredSystemVersion $Path $Path `
        --gtest_filter=NoOneHasExFat.PatchRequiredSystemVersion

    $ErrorCount += RunTest ExFat.NoOneHasExFat.LcsRequiredSystemVersion $Path $Path `
        --gtest_filter=NoOneHasExFat.LcsRequiredSystemVersion

    $ErrorCount += RunTest ExFat.NoOneHasExFat.ApplicationDeliveryProtocol $Path $Path `
        --gtest_filter=NoOneHasExFat.ApplicationDeliveryProtocol

    $ErrorCount += RunTest ExFat.SlaveHasExFat.PatchRequiredSystemVersion1 $Path $Path `
        --gtest_filter=SlaveHasExFat.PatchRequiredSystemVersion1

    $ErrorCount += RunTest ExFat.SlaveHasExFat.PatchRequiredSystemVersion2 $Path $Path `
        --gtest_filter=SlaveHasExFat.PatchRequiredSystemVersion2

    $ErrorCount += RunTest ExFat.SlaveHasExFat.PatchRequiredSystemVersion3 $Path $Path `
        --gtest_filter=SlaveHasExFat.PatchRequiredSystemVersion3

    $ErrorCount += RunTest ExFat.SlaveHasExFat.LcsRequiredSystemVersion1 $Path $Path `
        --gtest_filter=SlaveHasExFat.LcsRequiredSystemVersion1

    $ErrorCount += RunTest ExFat.SlaveHasExFat.LcsRequiredSystemVersion2 $Path $Path `
        --gtest_filter=SlaveHasExFat.LcsRequiredSystemVersion2

    $ErrorCount += RunTest ExFat.SlaveHasExFat.LcsRequiredSystemVersion3 $Path $Path `
        --gtest_filter=SlaveHasExFat.LcsRequiredSystemVersion3

    $ErrorCount += RunTest ExFat.SlaveHasExFat.LcsRequiredSystemVersion4 $Path $Path `
        --gtest_filter=SlaveHasExFat.LcsRequiredSystemVersion4

    $ErrorCount += RunTest ExFat.SlaveHasExFat.ApplicationDeliveryProtocol1 $Path $Path `
        --gtest_filter=SlaveHasExFat.ApplicationDeliveryProtocol1

    $ErrorCount += RunTest ExFat.SlaveHasExFat.ApplicationDeliveryProtocol2 $Path $Path `
        --gtest_filter=SlaveHasExFat.ApplicationDeliveryProtocol2

    $ErrorCount += RunTest ExFat.SlaveHasExFat.ApplicationDeliveryProtocol3 $Path $Path `
        --gtest_filter=SlaveHasExFat.ApplicationDeliveryProtocol3

    $ErrorCount += RunTest ExFat.SlaveHasExFat.ApplicationDeliveryProtocol4 $Path $Path `
        --gtest_filter=SlaveHasExFat.ApplicationDeliveryProtocol4

    $ErrorCount += RunTest ExFat.SlaveHasExFat.PatchShare $Path $Path `
        --gtest_filter=SlaveHasExFat.PatchShare

    return $ErrorCount
}

# Fork テストを実行します。
Function Fork($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationFork")
    $ErrorCount = 0

    $ErrorCount += RunTest Fork.Lup.PatchRequiredSystemVersion1 $Path $Path `
        --gtest_filter=ForkLup.PatchRequiredSystemVersion1

    $ErrorCount += RunTest Fork.Lup.PatchRequiredSystemVersion2 $Path $Path `
        --gtest_filter=ForkLup.PatchRequiredSystemVersion2

    $ErrorCount += RunTest Fork.Lup.LcsRequiredSystemVersion1 $Path $Path `
        --gtest_filter=ForkLup.LcsRequiredSystemVersion1

    $ErrorCount += RunTest Fork.Lup.LcsRequiredSystemVersion2 $Path $Path `
        --gtest_filter=ForkLup.LcsRequiredSystemVersion2

    $ErrorCount += RunTest Fork.Lup.ApplicationDeliveryProtocol1 $Path $Path `
        --gtest_filter=ForkLup.ApplicationDeliveryProtocol1

    $ErrorCount += RunTest Fork.Lup.ApplicationDeliveryProtocol2 $Path $Path `
        --gtest_filter=ForkLup.ApplicationDeliveryProtocol2

    $ErrorCount += RunTest Fork.PatchShare.HostIsMaster $Path $Path `
        --gtest_filter=ForkPatchShare.HostIsMaster

    $ErrorCount += RunTest Fork.PatchShare.ClientIsMaster $Path $Path `
        --gtest_filter=ForkPatchShare.ClientIsMaster

    return $ErrorCount
}

# HostMigration テストを実行します。
Function HostMigration($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationHostMigration")
    $ErrorCount = 0

    $ErrorCount += RunTest HostMigrationA $Path $Path `
       --gtest_filter=HostMigration.HostHasLatestSystem

    $ErrorCount += RunTest HostMigrationB $Path $Path `
        --gtest_filter=HostMigration.HostAndFirstClientHaveLatestSystem

    $ErrorCount += RunTest HostMigrationC $Path $Path `
        --gtest_filter=HostMigration.FirstClientHasLatestSystem

    $ErrorCount += RunTest HostMigrationD $Path $Path `
        --gtest_filter=HostMigration.SomeClientsHaveLatestSystem

    return $ErrorCount
}

# KeyGeneration テストを実行します。
Function KeyGeneration($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationKeyGeneration")
    $ErrorCount = 0

    $ErrorCount += RunTest KeyGeneration $Path $Path

    return $ErrorCount
}

# Lup テストを実行します。
Function Lup($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationLup")
    $ErrorCount = 0

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.HostToAllClients $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.HostToAllClients

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.HostToSomeClients $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.HostToSomeClients

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.ClientToHostAndAllOtherClients $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.ClientToHostAndAllOtherClients

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.ClientToHost $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.ClientToHost

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.ClientToHostAndOneClient $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.ClientToHostAndOneClient

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.ClientToOneClient $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.ClientToOneClient

    $ErrorCount += RunTest Lup.PatchRequiredSystemVersion.NotRequired $Path $Path `
        --gtest_filter=LupPatchRequiredSystemVersion.NotRequired

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.HostToAllClients $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.HostToAllClients

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.HostToSomeClients $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.HostToSomeClients

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.ClientToHostAndAllOtherClients $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.ClientToHostAndAllOtherClients

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.ClientToHost $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.ClientToHost

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.ClientToHostAndOneClient $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.ClientToHostAndOneClient

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.ClientToOneClient $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.ClientToOneClient

    $ErrorCount += RunTest Lup.LcsRequiredSystemVersion.NotRequired $Path $Path `
        --gtest_filter=LupLcsRequiredSystemVersion.NotRequired

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.HostToAllClients $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.HostToAllClients

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.HostToSomeClients $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.HostToSomeClients

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.ClientToHostAndAllOtherClients $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.ClientToHostAndAllOtherClients

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.ClientToHost $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.ClientToHost

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.ClientToHostAndOneClient $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.ClientToHostAndOneClient

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.ClientToOneClient $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.ClientToOneClient

    $ErrorCount += RunTest Lup.ApplicationDeliveryProtocol.NotRequired $Path $Path `
        --gtest_filter=LupApplicationDeliveryProtocol.NotRequired

    $ErrorCount += RunTest Lup.Discontinue.LeaveSuspendedWithRebootRequired $Path $Path `
        --gtest_filter=LupDiscontinue.LeaveSuspendedWithRebootRequired

    $ErrorCount += RunTest Lup.Discontinue.ResumeWithoutReboot $Path $Path `
        --gtest_filter=LupDiscontinue.ResumeWithoutReboot

    return $ErrorCount
}

# Matching テストを実行します。
Function Matching($Targets, $SceneId)
{
    $HostPath = (TestPath "IntegrationMatchingHost")
    $ClientPath = (TestPath "IntegrationMatchingClient")
    $ErrorCount = 0

    $ErrorCount += RunTest Matching $HostPath $ClientPath

    return $ErrorCount
}

# PatchShare テストを実行します。
Function PatchShare($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationPatchShare")
    $ErrorCount = 0

    $ErrorCount += RunTest PatchShare $Path $Path

    return $ErrorCount
}

# PatchShareFailure テストを実行します。
Function PatchShareFailure($Targets)
{
    $Path = (TestPath "IntegrationPatchShareFailure")
    $ErrorCount = 0

    $ErrorCount += RunTest PatchShareFailure $Path $Path `
        --gtest_filter=-*.SuspendedStorageSpaceNotEnough*:Leave.SuspendedIncompatibleContentsInfo

    $ErrorCount += RunTest PatchShareFailure.StorageSpaceNotEnough $Path $Path `
        --gtest_filter=Leave.SuspendedStorageSpaceNotEnough*

    $ErrorCount += RunTest PatchShareFailure.IncompatibleContentsInfo $Path $Path `
        --gtest_filter=Leave.SuspendedIncompatibleContentsInfo*

    return $ErrorCount
}

# Standalone テストを実行します。
Function Standalone($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationStandalone")
    $ErrorCount = 0

    $ErrorCount += RunTest Standalone $Path $Path

    return $ErrorCount
}

# SystemDeliveryProtocol テストを実行します。
Function SystemDeliveryProtocol($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationSystemDeliveryProtocol")
    $ErrorCount = 0

    $ErrorCount += RunTest SystemDeliveryProtocol.HostIsHigherVersion $Path $Path `
        --gtest_filter=SystemDeliveryProtocol.HostIsHigherVersion

    $ErrorCount += RunTest SystemDeliveryProtocol.HostIsLowerVersion $Path $Path `
        --gtest_filter=SystemDeliveryProtocol.HostIsLowerVersion

    return $ErrorCount
}

# Timing テストを実行します。
Function Timing($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationTiming")
    $ErrorCount = 0

    $ErrorCount += RunTest Timing $Path $Path `
       --gtest_filter=-*Lup*:*HostMigration*

    # TODO: 親交代中に切断するとアボートするので実行しない
    #&$SETUP Timing.HostMigration
    #$ErrorCount += (RunParallel $Targets $SceneId $Path $Path `
    #   --gtest_filter=AfterStart.SimultaneousHostMigration*)

    $ErrorCount += RunTest Timing.Lup $Path $Path `
        --gtest_filter=Lup.SimultaneousSuspendAndResume

    foreach ($TestCount in 0..3)
    {
        $ErrorCount += RunTest Timing.Lup $Path $Path `
            "-t" $TestCount --gtest_filter=Lup.SimultaneousLeaveSuspend
    }

    foreach ($TestCount in 0..3)
    {
        $ErrorCount += RunTest Timing.Lup $Path $Path `
            "-t" $TestCount --gtest_filter=Lup.SimultaneousFinalizeSuspend
    }

    foreach ($TestCount in 0..3)
    {
        $ErrorCount += RunTest Timing.Lup $Path $Path `
            "-t" $TestCount --gtest_filter=Lup.SimultaneousLeaveResume
    }

    foreach ($TestCount in 0..3)
    {
        $ErrorCount += RunTest Timing.Lup $Path $Path `
            "-t" $TestCount --gtest_filter=Lup.SimultaneousFinalizeResume
    }

    return $ErrorCount
}

# StorageSpace テストを実行します。
Function StorageSpace($Targets, $SceneId)
{
    $Path = (TestPath "IntegrationStorageSpace")
    $ErrorCount = 0

    $ErrorCount += RunTest StorageSpace.NotEnough $Path $Path `
            --gtest_filter=StorageSpace.NotEnoughAllClients

    &$SETUP StorageSpace.PatchSpace
    $ErrorCount += (RunParallel $Targets $SceneId $Path $Path `
            --gtest_filter=StorageSpace.NotEnoughAfterStartedAllClients)

    return $ErrorCount
}

# 全てのテストを実行します。
Function All($Targets, $SceneId)
{
    $ErrorCount = 0
    $ErrorCount += (Standalone $Targets $SceneId)
    $ErrorCount += (Matching $Targets $SceneId)
    $ErrorCount += (PatchShare $Targets $SceneId)
    $ErrorCount += (PatchShareFailure $Targets $SceneId)
    $ErrorCount += (KeyGeneration $Targets $SceneId)
    $ErrorCount += (SystemDeliveryProtocol $Targets $SceneId)
    $ErrorCount += (HostMigration $Targets $SceneId)
    $ErrorCount += (Lup $Targets $SceneId)
    $ErrorCount += (ExFat $Targets $SceneId)
    $ErrorCount += (Fork $Targets $SceneId)
    $ErrorCount += (Timing $Targets $SceneId)
    $ErrorCount += (LupEula $Targets $SceneId)
    $ErrorCount += (StorageSpace $Targets $SceneId)

    return $ErrorCount
}

# 指定されたテストを実行します。
Function Run($SceneId)
{
    # テスト実行の開始を通知するログを表示します。
    Write-Host "Run: $($TestName)"

    # LdnTestBridge を終了しておきます。
    if (Get-Process | Where-Object { $_.Name -eq "LdnTestBridge" })
    {
        Stop-Process -Name "LdnTestBridge"
    }

    # PC に接続されている開発機を取得します。
    $Targets = GetTargetList

    # Scene Id が指定されなかった場合、ランダムに生成します。
    if ($SceneId -lt 0)
    {
        $SceneId = (Get-Random 65536)
    }

    # 引数で指定されたテストを実行します。
    if ($TestName -eq "All")
    {
        return (All $Targets $SceneId)
    }
    elseif ($TestName -eq "LupEula")
    {
        return (LupEula $Targets $SceneId)
    }
    elseif ($TestName -eq "ApplicationRecord")
    {
        return (ApplicationRecord $Targets $SceneId)
    }
    elseif ($TestName -eq "ExFat")
    {
        return (ExFat $Targets $SceneId)
    }
    elseif ($TestName -eq "Fork")
    {
        return (Fork $Targets $SceneId)
    }
    elseif ($TestName -eq "HostMigration")
    {
        return (HostMigration $Targets $SceneId)
    }
    elseif ($TestName -eq "KeyGeneration")
    {
        return (KeyGeneration $Targets $SceneId)
    }
    elseif ($TestName -eq "Lup")
    {
        return (Lup $Targets $SceneId)
    }
    elseif ($TestName -eq "Matching")
    {
        return (Matching $Targets $SceneId)
    }
    elseif ($TestName -eq "PatchShare")
    {
        return (PatchShare $Targets $SceneId)
    }
    elseif ($TestName -eq "PatchShareFailure")
    {
        return (PatchShareFailure $Targets $SceneId)
    }
    elseif ($TestName -eq "Standalone")
    {
        return (Standalone $Targets $SceneId)
    }
    elseif ($TestName -eq "SystemDeliveryProtocol")
    {
        return (SystemDeliveryProtocol $Targets $SceneId)
    }
    elseif ($TestName -eq "Timing")
    {
        return (Timing $Targets $SceneId)
    }
    elseif ($TestName -eq "StorageSpace")
    {
        return (StorageSpace $Targets $SceneId)
    }
    else
    {
        throw "Unknown Test: $TestName"
    }
}

$ExitCode = (Run $SceneId)
if ($ExitCode -ne 0)
{
    throw "Test Failed: $TestName ($ExitCode)"
}
return $ExitCode
