﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nnt/lcs/testLcs_CommandLineParser.h>

namespace nnt { namespace lcs { namespace
{
    const size_t LongNameLengthMax = 16;

    bool IsOption(const char* arg, char shortName, const char* longName) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(arg);
        NN_ASSERT_NOT_EQUAL(shortName, '-');
        return (shortName != 0 && arg[0] == '-' && arg[1] == shortName) ||
            (longName != nullptr && std::strncmp(arg, longName, LongNameLengthMax) == 0);
    }

    int GetInteger(const char* name, const char* str, int min, int max) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(name);
        NN_ASSERT_NOT_NULL(str);
        NN_ASSERT(min <= max);

        // 文字列を整数値に変換します。
        char* end;
        int n = std::strtol(str, &end, 10);

        // 変換に失敗した場合にはエラーメッセージを出力して終了します。
        if (end == str || *end != 0)
        {
            NN_LOG("FAILED: %s should be decimal number\n", name);
            std::exit(-1);
        }
        else if (n < min)
        {
            NN_LOG("FAILED: %s (=%d) should be greater than or equal to %d\n", name, n, min);
            std::exit(-1);
        }
        else if (max < n)
        {
            NN_LOG("FAILED: %s (=%d) should be less than or equal to %d\n", name, n, max);
            std::exit(-1);
        }
        return n;
    }

    int GetTimeSpan(const char* name, const char* str, int min, int max) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(name);
        NN_ASSERT_NOT_NULL(str);
        NN_ASSERT(min <= max);

        // 文字列を整数値に変換します。
        char* end;
        int n = std::strtol(str, &end, 10);
        if (end == str)
        {
            NN_LOG("FAILED: %s should be decimal number\n", name);
            std::exit(-1);
        }

        // 単位を取得します。
        char unit = *end;
        if (*end == 0)
        {
            // 単位を指定しなかった場合には秒単位の指定とします。
        }
        else if (unit == 's' || unit == 'S')
        {
            // 秒単位です。
        }
        else if (unit == 'm' || unit == 'M')
        {
            // 分単位です。
            n *= 60;
        }
        else if (unit == 'h' || unit == 'H')
        {
            // 時間単位です。
            n *= 60 * 60;
        }
        else if (unit == 'd' || unit == 'D')
        {
            // 日単位です。
            n *= 24 * 60 * 60;
        }
        else
        {
            NN_LOG("FAILED: %c is invalid unit\n", name);
            std::exit(-1);
        }

        // 範囲チェックです。
        if (n < min)
        {
            NN_LOG("FAILED: %s (=%d) should be greater than or equal to %d seconds\n", name, n, min);
            std::exit(-1);
        }
        else if (max < n)
        {
            NN_LOG("FAILED: %s (=%d) should be less than or equal to %d seconds\n", name, n, max);
            std::exit(-1);
        }

        return n;
    }

}}} // namespace nnt::lcs::<unnamed>

namespace nnt { namespace lcs
{
    void Parse(TestConfig* pOutConfig, const CommandLineParserSetting& setting,
               int argc, char* argv[]) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutConfig);
        NN_SDK_ASSERT(0 < argc);
        NN_SDK_ASSERT_NOT_NULL(argv);

        // デフォルト値を設定します。
        std::memset(pOutConfig, 0, sizeof(TestConfig));
        pOutConfig->duration = 364 * 24 * 3600;
        nn::os::GenerateRandomBytes(&pOutConfig->seed, sizeof(pOutConfig->seed));
        pOutConfig->seed &= 0x7FFF;
        pOutConfig->threshold = 0;

        // コマンドライン引数を最後まで解析します。
        int i = 1;
        int outIndex;
        while (i < argc)
        {
            // ノードの数です。
            if ((setting.flag & CommandLineOptionFlag_NodeCount) != 0 &&
                IsOption(argv[i], 'n', "node") && i + 1 < argc)
            {
                pOutConfig->nodeCount = static_cast<int8_t>(
                    GetInteger("the number of nodes", argv[i + 1],
                    setting.nodeCountMin, setting.nodeCountMax));
                i += 2;
            }

            // ノードのインデックスです。アクセスポイントは自動的に 0 です。
            else if ((setting.flag & CommandLineOptionFlag_NodeIndex) != 0 &&
                     IsOption(argv[i], 'i', "index") && i + 1 < argc)
            {
                pOutConfig->nodeIndex = static_cast<int8_t>(
                    GetInteger("node index", argv[i + 1], 0, setting.nodeCountMax - 1));
                i += 2;
            }

            // エイジング時間です。
            else if ((setting.flag & CommandLineOptionFlag_Duration) != 0 &&
                     IsOption(argv[i], 'd', "duration") && i + 1 < argc)
            {
                pOutConfig->duration = static_cast<int32_t>(GetTimeSpan(
                    "Duration", argv[i + 1], 1, 365 * 24 * 3600));
                i += 2;
            }

            // シード値です。
            else if ((setting.flag & CommandLineOptionFlag_Seed) != 0 &&
                     IsOption(argv[i], 0, "seed") && i + 1 < argc)
            {
                pOutConfig->seed = static_cast<int32_t>(GetInteger(
                    "Seed", argv[i + 1], 0, 32767));
                i += 2;
            }

            // テスト成功・失敗の判定に仕様する閾値です。
            else if ((setting.flag & CommandLineOptionFlag_Threshold) != 0 &&
                     IsOption(argv[i], 't', "threshold") && i + 1 < argc)
            {
                pOutConfig->threshold = GetInteger(
                    "Threshold", argv[i + 1], 0, 100) * 1.0f;
                i += 2;
            }

            // 再起動時に指定する引数です。
            else if ((setting.flag & CommandLineOptionFlag_Resume) != 0 &&
                     IsOption(argv[i], 'r', "resume"))
            {
                pOutConfig->resume = true;
                ++i;
            }

            // シーン識別子です。
            else if ((setting.flag & CommandLineOptionFlag_SceneId) != 0 &&
                     IsOption(argv[i], 's', "scene") && i + 1 < argc)
            {
                pOutConfig->sceneId = static_cast<uint16_t>(GetInteger(
                    "Scene ID", argv[i + 1], 0, 65535));
                i += 2;
            }

            // カスタムオプションです。
            else if (setting.parser && setting.parser(&outIndex, argv, argc, i))
            {
                i = outIndex;
            }

            // 無効な引数です。
            else
            {
                NN_LOG("FAILED: %s is an invalid argument\n", argv[i]);
                std::exit(-1);
            }
        }
    } // NOLINT(impl/function_size)

}} // namespace nnt::lcs
