﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include "../Common/testLbl_Helper.h"
#include "../Common/testLbl_Util-hardware.nx.h"
#include "../Common/testLbl_AutoControlEmulation.h"
#include "../Common/testLbl_Config.h"

namespace {

const float HighAmbientLightSensorValue = 816.147339f;  // 蛍光灯下でのセンサー値

const size_t TestSwitchBacklightStackSize = 4096;
NN_ALIGNAS(4096) char s_TestSwitchBacklightStack[TestSwitchBacklightStackSize];

void TestApplyingBrightness(int degrees, int64_t intervalMilliSeconds)
{
    nn::TimeSpan timeSpan = nn::TimeSpan::FromMilliSeconds(intervalMilliSeconds);
    for (int i1=0; i1<degrees; ++i1)
    {
        // 自動輝度調整off or VRモードのとき、SetAmbientLightSensorValue を呼び出しても何も起こらない。
        lbldfc::SetAmbientLightSensorValue((float)i1 / degrees * HighAmbientLightSensorValue);
        if ( lbldfc::IsVrModeEnabled() )
        {
            lbldfc::SetCurrentBrightnessSettingForVrMode((float)i1 / degrees);
        }
        else
        {
            lbldfc::SetCurrentBrightnessSetting((float)i1 / degrees);
        }
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        nn::os::SleepThread(timeSpan);
    }
    for (int i1=degrees; i1>0; --i1)
    {
        // 自動輝度調整off or VRモードのとき、SetAmbientLightSensorValue を呼び出しても何も起こらない。
        lbldfc::SetAmbientLightSensorValue((float)i1 / degrees * HighAmbientLightSensorValue);
        if ( lbldfc::IsVrModeEnabled() )
        {
            lbldfc::SetCurrentBrightnessSettingForVrMode((float)i1 / degrees);
        }
        else
        {
            lbldfc::SetCurrentBrightnessSetting((float)i1 / degrees);
        }
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        nn::os::SleepThread(timeSpan);
    }
}

void TestApplyingBrightnessWithDelay(int degrees, int64_t intervalMilliSeconds)
{
    // 輝度設定値が0.0の状態で安定させる。
    nnt::lbl::GetStableAutoAppliedBrightnessSetting(0.0f, 0.0f);

    nn::TimeSpan timeSpan = nn::TimeSpan::FromMilliSeconds(intervalMilliSeconds);
    lbldfc::SetCurrentBrightnessSetting(1.0f);
    lbldfc::SetAmbientLightSensorValue(HighAmbientLightSensorValue);
    for (int i1=0; i1<degrees; ++i1)
    {
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        nn::os::SleepThread(timeSpan);
    }
    lbldfc::SetAmbientLightSensorValue(0.0f);
    for (int i1=degrees; i1>0; --i1)
    {
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        nn::os::SleepThread(timeSpan);
    }
}

void TestSwitchingBacklight(int degrees, int64_t fadeMilliSeconds)
{
    nn::TimeSpan longFadeTime = nn::TimeSpan::FromMilliSeconds(fadeMilliSeconds);
    nn::TimeSpan partFadeTime = nn::TimeSpan::FromMilliSeconds(fadeMilliSeconds / degrees);
    int shortDegrees = degrees / 4;

    // off にするが途中で on に切り替える。
    lbldfc::SwitchBacklightOff(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<shortDegrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    lbldfc::SwitchBacklightOn(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<shortDegrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    // 観測のための待ち時間
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    // off に切り替える。
    lbldfc::SwitchBacklightOff(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<degrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    // 観測のための待ち時間
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    // on にするが途中で off に切り替える。
    lbldfc::SwitchBacklightOn(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<shortDegrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    lbldfc::SwitchBacklightOff(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<shortDegrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    // 観測のための待ち時間
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    // on に切り替える。
    lbldfc::SwitchBacklightOn(longFadeTime);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    for (int i1=0; i1<degrees; ++i1)
    {
        nn::os::SleepThread(partFadeTime);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    }
    // 観測のための待ち時間
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
}

void TestSwitchingBacklightLoop(void* args)
{
    bool* pIsRunning = reinterpret_cast<bool*>(args);
    while (*pIsRunning)
    {
        TestSwitchingBacklight(2, 200);
    }
}

}

/**
 * @brief   LCDバックライトの輝度を自動輝度調整offで変化させるケースで動作するか(粗め設定)。
 */
TEST(Practice, ApplyBrightnessRoughly)
{
    nnt::lbl::LblInitializer initializer;

    // 自動輝度調整offで粗めに輝度を変化。
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(10, 500);
}

/**
 * @brief   LCDバックライトの輝度を自動輝度調整offで変化させるケースで動作するか(細かめ設定)。
 */
TEST(Practice, ApplyBrightnessProperly)
{
    nnt::lbl::LblInitializer initializer;

    // 自動輝度調整offで細かめに輝度を変化。
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(100, 50);
}

/**
 * @brief   LCDバックライトの輝度を自動輝度調整onで変化させるケースで動作するか。
 */
TEST(Practice, AutoApplyBrightness)
{
    nnt::lbl::LblInitializer initializer;

    // 自動輝度調整onで輝度を適当に変化。
    lbldfc::EnableAutoBrightnessControl();

    // 現在の輝度設定値をチラつきながら変化する。
    TestApplyingBrightness(100, 50);
}

/**
 * @brief   LCDバックライトの輝度を自動輝度調整onかつ反応遅延ありで変化させるケースで動作するか。
 */
TEST(Practice, AutoApplyBrightnessWithDelay)
{
    nnt::lbl::LblInitializer initializer;

    // 自動輝度調整onで輝度を適当に変化。
    lbldfc::EnableAutoBrightnessControl();

    // 徐々に輝度変化する。
    TestApplyingBrightnessWithDelay(100, 50);
}

/**
 * @brief   簡単なケースでバックライトの on/off の制御が動作するか。
 */
TEST(Practice, SwitchBacklight)
{
    nnt::lbl::LblInitializer initializer;

    // 0ms から 1600ms まで。
    TestSwitchingBacklight(1, 0);
    int64_t fadeMilliSeconds = 50LL;
    while (fadeMilliSeconds <= 1600LL)
    {
        TestSwitchingBacklight(static_cast<int>(fadeMilliSeconds / 50LL), fadeMilliSeconds);
        fadeMilliSeconds *= 2L;
    }
}

/**
 * @brief   他の機能が働いているケースでバックライトの on/off の制御が動作するか。
 */
TEST(Practice, SwitchBacklightWithChangingBrightness)
{
    nnt::lbl::LblInitializer initializer;

    // バックライト制御を適当にするスレッドを立てる。
    nn::os::ThreadType thread;
    bool isRunning = true;
    nn::os::CreateThread(&thread, TestSwitchingBacklightLoop, &isRunning,
        s_TestSwitchBacklightStack, TestSwitchBacklightStackSize, nn::os::DefaultThreadPriority);
    nn::os::StartThread(&thread);

    // 自動輝度調整offで細かめに輝度を変化。
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(10, 40);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(4000));

    // 自動輝度調整onで輝度を適当に変化。
    lbldfc::EnableAutoBrightnessControl();

    // VR モードにして適当に変化。
    lbldfc::EnableVrMode();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);
    lbldfc::DisableVrMode();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);

    // 現在の輝度設定値をチラつきながら変化する。
    TestApplyingBrightness(40, 20);

    // スレッドの終了処理
    isRunning = false;
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
}

/**
 * @brief   アプリ側で自動輝度調整のエミュレートができるか。
 */
TEST(Practice, AutoControlEmulation)
{
    nnt::lbl::LblInitializer initializer;

    // アプリ側でエミュレートするために自動輝度調整をoffにする。
    lbldfc::DisableAutoBrightnessControl();

    // 5秒間テスト。
    float level = 1.0f;
    for (int i1=0; i1<300; ++i1)
    {
        // lbl プロセスが回っていないので適当に照度センサーの値を設定。
        lbldfc::SetAmbientLightSensorValue(3.0f * i1);

        bool isOverflown = false;
        float sensor = lbldfc::GetAmbientLightSensorValue(&isOverflown);
        float brightness = nnt::lbl::CalculateAutoControlledBrightness(sensor, level);
        //NN_LOG("[%d] %s() sensor:[%f] brightness:[%f]\n", __LINE__, __FUNCTION__, sensor, brightness);

        lbldfc::SetCurrentBrightnessSetting(brightness);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        // 60fps で更新。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

/**
 * @brief   低輝度化が動作するか。
 */
TEST(Practice, Dimming)
{
    nnt::lbl::LblInitializer initializer;

    lbldfc::EnableDimming();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);
    lbldfc::DisableDimming();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);
}

/**
 * @brief   VRモード切替が動作するか。
 */
TEST(Practice, VrMode)
{
    nnt::lbl::LblInitializer initializer;

    lbldfc::EnableVrMode();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);
    lbldfc::DisableVrMode();
    lbldfc::DisableAutoBrightnessControl();
    TestApplyingBrightness(50, 50);
}
