﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nnt/nntest.h>
#include <algorithm>
#include <cmath>

#include "../Common/testLbl_Helper.h"
#include "../Common/testLbl_Util-hardware.nx.h"
#include "../Common/testLbl_Config.h"

namespace {

    bool IsAlsEnabledEnvironment() NN_NOEXCEPT
    {
        // 照度センサが有効なプラットフォームかどうかを取得
        // fwdbg で明示的に指示されていたら、プラットフォーム構成情報は無視して強制無効化
        bool isAlsEnabled = false;
        bool forceDisableAls = false;
        nn::settings::fwdbg::GetSettingsItemValue(&forceDisableAls, sizeof(forceDisableAls), "lbl", "force_disable_als");
        if ( forceDisableAls )
        {
            isAlsEnabled = false;
        }
        else
        {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            NN_ABORT_UNLESS_EQUAL(sizeof(isAlsEnabled), nn::settings::fwdbg::GetSettingsItemValue(&isAlsEnabled, sizeof(isAlsEnabled), "platformconfig", "has_als"));
#else
            // Win 環境はプラットフォーム構成情報が読めることを期待せず、常時使用可能とする
            isAlsEnabled = true;
#endif
        }

        return isAlsEnabled;
    }

    float GetVrBrightnessScale() NN_NOEXCEPT
    {
        int vrBrightnessScalePermil = 593;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS_EQUAL(sizeof(vrBrightnessScalePermil),
            nn::settings::fwdbg::GetSettingsItemValue(&vrBrightnessScalePermil, sizeof(vrBrightnessScalePermil), "lbl", "vr_brightness_scale"));
#else
        // Win 環境はプラットフォーム構成情報が読めることを期待しない
#endif
        return static_cast<float>(vrBrightnessScalePermil) / 1000.0f;
    }

    float GetVrLowerBrightness() NN_NOEXCEPT
    {
        int vrLowerBrightnessPermil = 20;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS_EQUAL(sizeof(vrLowerBrightnessPermil),
            nn::settings::fwdbg::GetSettingsItemValue(&vrLowerBrightnessPermil, sizeof(vrLowerBrightnessPermil), "lbl", "vr_lower_brightness"));
#else
        // Win 環境はプラットフォーム構成情報が読めることを期待しない
#endif
        return static_cast<float>(vrLowerBrightnessPermil) / 1000.0f;
    }

    float GetDimmingBrightness() NN_NOEXCEPT
    {
        int dimmingBrightnessPermil = 150;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_ABORT_UNLESS_EQUAL(sizeof(dimmingBrightnessPermil),
            nn::settings::fwdbg::GetSettingsItemValue(&dimmingBrightnessPermil, sizeof(dimmingBrightnessPermil), "lbl", "dimming_brightness"));
#else
        // Win 環境はプラットフォーム構成情報が読めることを期待しない
#endif
        return static_cast<float>(dimmingBrightnessPermil) / 1000.0f;
    }

    void ApplyCurrentBrightnessSettingUntilVrTransitionFinished()
    {
        int transitionframe = 10;
        nn::settings::fwdbg::GetSettingsItemValue(&transitionframe, sizeof(transitionframe), "lbl", "vr_brightness_transition_frame");
        for(int i = 0; i < transitionframe; i++)
        {
            lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        }
    }
}

/**
 * @brief   照度の有効・無効状態に沿った結果が得られるか
 */
TEST(Basic, IsAmbientLightSensorAvailable)
{
    nnt::lbl::LblInitializer initializer;

    if ( IsAlsEnabledEnvironment() )
    {
        EXPECT_TRUE(lbldfc::IsAmbientLightSensorAvailable());
    }
    else
    {
        EXPECT_FALSE(lbldfc::IsAmbientLightSensorAvailable());
    }
}

/**
 * @brief   一般的な変数の境界値で正常動作するか
 *          定義された変数の境界値で正常動作するか
 */
TEST(Basic, BoundaryArgs)
{
    nnt::lbl::LblInitializer initializer;

    nnt::lbl::SetMaxValueToSetting();
    nnt::lbl::SetMinValueToSetting();
}

/**
 * @brief   一般的な変数の異常値で正常にエラー処理されるか
 *          定義された変数の異常値で正常にエラー処理されるか
 */
TEST(Basic, IrregularArgs)
{
    nnt::lbl::LblInitializer initializer;

    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSetting(- std::numeric_limits<float>::min()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSetting(std::numeric_limits<float>::lowest()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSettingForVrMode(-std::numeric_limits<float>::min()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSettingForVrMode(std::numeric_limits<float>::lowest()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetAmbientLightSensorValue(- std::numeric_limits<float>::min()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetAmbientLightSensorValue(std::numeric_limits<float>::lowest()), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOn(nn::TimeSpan::FromNanoSeconds(- std::numeric_limits<int64_t>::min())), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOn(nn::TimeSpan::FromNanoSeconds(std::numeric_limits<int64_t>::lowest())), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOff(nn::TimeSpan::FromNanoSeconds(- std::numeric_limits<int64_t>::min())), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOff(nn::TimeSpan::FromNanoSeconds(std::numeric_limits<int64_t>::lowest())), "");
}

/**
 * @brief   初期化せずに実行したときにアボートするか
 */
TEST(Basic, NotInitialzed)
{
    // 初期化せずに呼び出す。
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSetting(0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::GetCurrentBrightnessSetting(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetCurrentBrightnessSettingForVrMode(0.0f), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::GetCurrentBrightnessSettingForVrMode(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::ApplyCurrentBrightnessSettingToBacklight(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::GetBrightnessSettingAppliedToBacklight(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOn(0LL), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SwitchBacklightOff(0LL), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::GetBacklightSwitchStatus(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::EnableDimming(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::DisableDimming(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::IsDimmingEnabled(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::EnableVrMode(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::DisableVrMode(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::IsVrModeEnabled(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::EnableAutoBrightnessControl(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::DisableAutoBrightnessControl(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::IsAutoBrightnessControlEnabled(), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::SetAmbientLightSensorValue(0.0f), "");
    bool isOverflown = false;
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::GetAmbientLightSensorValue(&isOverflown), "");
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::IsAmbientLightSensorAvailable(), "");
}

/**
 * @brief   一般的な内部変数の境界値で正常動作するか
 *          定義された内部変数の境界値で正常動作するか
 */
TEST(Basic, BoundaryState)
{
    nnt::lbl::LblInitializer initializer;

    // 内部変数を最大値側の境界値にしてテスト。
    lbldfc::EnableAutoBrightnessControl();
    nnt::lbl::SetMaxValueToSetting();

    // 書いて読む。
    lbldfc::SaveCurrentSetting();
    lbldfc::LoadCurrentSetting();

    // 読みだした内部変数が一致するかチェック。
    EXPECT_EQ(std::numeric_limits<float>::max(), lbldfc::GetCurrentBrightnessSetting());
    EXPECT_EQ(std::numeric_limits<float>::max(), lbldfc::GetCurrentBrightnessSettingForVrMode());

    // 自動輝度調整がonになったときは照度センサー値も影響するので、on/off両方のケースでチェック。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_LE(std::numeric_limits<float>::max() / 2.0f, lbldfc::GetBrightnessSettingAppliedToBacklight());
    // 照度センサー値は SetMaxValueToSetting() した時点で最大値になる。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_LT(0.98f, lbldfc::GetBrightnessSettingAppliedToBacklight());

    lbldfc::SetAmbientLightSensorValue(std::numeric_limits<float>::max());
    bool isOverflown = false;
    EXPECT_EQ(std::numeric_limits<float>::max(), lbldfc::GetAmbientLightSensorValue(&isOverflown));

    // 内部変数を最小値側の境界値にしてテスト。
    lbldfc::DisableAutoBrightnessControl();
    nnt::lbl::SetMinValueToSetting();

    // 書いて読む。
    lbldfc::SaveCurrentSetting();
    lbldfc::LoadCurrentSetting();

    // 読みだした内部変数が一致するかチェック。
    EXPECT_EQ(0.0f, lbldfc::GetCurrentBrightnessSetting());
    EXPECT_EQ(0.0f, lbldfc::GetCurrentBrightnessSettingForVrMode());

    // 自動輝度調整がonになったときは照度センサー値も影響するので、on/off両方のケースでチェック。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(nnt::lbl::MinManualControlBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());
    // 照度センサー値は SetMinValueToSetting() した時点で最小値になる。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_GT(0.2f, lbldfc::GetBrightnessSettingAppliedToBacklight());

    lbldfc::SetAmbientLightSensorValue(0.0f);
    EXPECT_EQ(0.0f, lbldfc::GetAmbientLightSensorValue(&isOverflown));
}

/**
 * @brief   settings を使った設定値の Save/Load が正常に動作するか
 */
TEST(Basic, SaveAndLoadSetting)
{
    nnt::lbl::LblInitializer initializer;

    // 適当な値を書く。
    float reference = 0.5f;
    float overwrite = reference + 0.1f;
    nnt::lbl::SetUniformedValueToSetting(reference);
    lbldfc::SaveCurrentSetting();

    // 照合用データとして自動輝度調整onのときに適用した輝度設定値を取得。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    float autoAppliedBrightness = nnt::lbl::GetStableAutoAppliedBrightnessSetting(reference, reference);
    lbldfc::DisableAutoBrightnessControl();

    // ちゃんとLoadされるかチェックするために内部変数を適当な値でクリアー。
    nnt::lbl::SetUniformedValueToSetting(overwrite);
    lbldfc::LoadCurrentSetting();

    // 照合
    EXPECT_EQ(reference, lbldfc::GetCurrentBrightnessSetting());
    EXPECT_EQ(reference, lbldfc::GetCurrentBrightnessSettingForVrMode());

    // 自動輝度調整がoffのときは照度センサー値は影響しない。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(autoAppliedBrightness, nnt::lbl::GetStableAutoAppliedBrightnessSetting(reference, reference));
    lbldfc::DisableAutoBrightnessControl();
}

/**
 * @brief   期待する処理時間に収まっているか
 */
TEST(Basic, ProcessingTime)
{
    nnt::lbl::LblInitializer initializer;

    // 処理時間に影響はなさそうだが、とりあえず自動輝度調整offかつ設定値の最大値を使用。
    lbldfc::DisableAutoBrightnessControl();
    nnt::lbl::SetMaxValueToSetting();

    nn::TimeSpan time;
    int64_t start = 0LL;
    int64_t end = 0LL;

    // とりあえず 100ms 以内で処理されるかをチェック。
    start = time.GetMilliSeconds();
    lbldfc::SaveCurrentSetting();
    end = time.GetMilliSeconds();
    EXPECT_GE(100, end - start);

    start = time.GetMilliSeconds();
    lbldfc::LoadCurrentSetting();
    end = time.GetMilliSeconds();
    EXPECT_GE(100, end - start);

    start = time.GetMilliSeconds();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    end = time.GetMilliSeconds();
    EXPECT_GE(100, end - start);
}

/**
 * @brief   自動輝度調整の切り替えが正しくされるか
 */
TEST(Basic, AutoControl)
{
    nnt::lbl::LblInitializer initializer;

    // 輝度設定値は中レベル、照度センサー値は最小レベル。
    float brightnessReference = 0.5f;
    float sensorReference = 0.0f;
    float margin = 0.05f;
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    lbldfc::SetAmbientLightSensorValue(sensorReference);

    lbldfc::EnableAutoBrightnessControl();
    EXPECT_TRUE(lbldfc::IsAutoBrightnessControlEnabled());
    // 自動輝度調整を有効にした直後ではすぐに輝度が変化する。
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    float brightnessStabled = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);
    EXPECT_EQ(brightnessStabled, lbldfc::GetBrightnessSettingAppliedToBacklight());

    lbldfc::DisableAutoBrightnessControl();
    EXPECT_FALSE(lbldfc::IsAutoBrightnessControlEnabled());
    // 自動輝度調整を無効にした直後でもすぐに輝度が変化する。
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 現在は手動輝度調整時の輝度レベルと現在の輝度設定値が同じではないが、
    // 近い値にはなるので、マージンを持たせてチェックする。
    EXPECT_LT(brightnessReference - margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
    EXPECT_GT(brightnessReference + margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
}

/**
 * @brief   基本的な点灯/消灯の切り替えが正しくされるか
 */
TEST(Basic, SwitchBacklight)
{
    // 初期化直後は消灯状態になっているか。
    lbldfc::Initialize();
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    lbldfc::Finalize();

    nnt::lbl::LblInitializer initializer;

    // 切り替え完了までの時間を0としたときは瞬時に切り替えがされる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    nn::TimeSpan fadeShortTime = nn::TimeSpan::FromMilliSeconds(20LL);

    // 消灯開始後と完了後で状態は異なる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    // 点灯開始後と完了後で状態は異なる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    // 低輝度化が有効なときも適切に振る舞う。
    {
        lbldfc::EnableDimming();
        NN_UTIL_SCOPE_EXIT{
            lbldfc::DisableDimming();
        };
        lbldfc::SetCurrentBrightnessSetting(1.0f);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        FAKE_ENV_TEST(EXPECT_GT(255, nn::pwm::GetDuty(nullptr)));

        APPLY_BRIGHTNESS_SETTING(lbldfc::SwitchBacklightOff(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
        APPLY_BRIGHTNESS_SETTING(nn::os::SleepThread(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)));
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)));

        APPLY_BRIGHTNESS_SETTING(lbldfc::SwitchBacklightOn(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
        APPLY_BRIGHTNESS_SETTING(nn::os::SleepThread(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_LT(0, nn::pwm::GetDuty(nullptr)));
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
    }

    // VR モードが有効なときも適切に振る舞う。
    {
        lbldfc::EnableVrMode();
        NN_UTIL_SCOPE_EXIT{
            lbldfc::DisableVrMode();
        };
        lbldfc::SetCurrentBrightnessSettingForVrMode(1.0f);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        FAKE_ENV_TEST(EXPECT_GT(255, nn::pwm::GetDuty(nullptr)));

        APPLY_BRIGHTNESS_SETTING(lbldfc::SwitchBacklightOff(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
        APPLY_BRIGHTNESS_SETTING(nn::os::SleepThread(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)));
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)));

        APPLY_BRIGHTNESS_SETTING(lbldfc::SwitchBacklightOn(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
        APPLY_BRIGHTNESS_SETTING(nn::os::SleepThread(fadeShortTime));
        EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
        FAKE_ENV_TEST(EXPECT_LT(0, nn::pwm::GetDuty(nullptr)));
        FAKE_ENV_TEST(EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)));
    }
}

/**
 * @brief   低輝度化の切り替えが正しくされるか
 */
TEST(Basic, Dimming)
{
    nnt::lbl::LblInitializer initializer;
    ASSERT_FALSE(lbldfc::IsVrModeEnabled()); // テスト事前条件

    EXPECT_FALSE(lbldfc::IsDimmingEnabled());
    lbldfc::SetCurrentBrightnessSetting(0.0f);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    const auto lowestBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    const auto dimmingBrightness = GetDimmingBrightness();

    // 輝度設定値が低いときは低輝度化されない(既に低輝度だから)
    // その他での低輝度時の輝度は一律。
    lbldfc::EnableDimming();
    EXPECT_TRUE(lbldfc::IsDimmingEnabled());
    lbldfc::SetCurrentBrightnessSetting(1.0f);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    float currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSetting(0.3f);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSetting(0.0f);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(lowestBrightness, currentBrightness);

    lbldfc::DisableDimming();
    EXPECT_FALSE(lbldfc::IsDimmingEnabled());
    lbldfc::SetCurrentBrightnessSetting(1.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSetting(0.3f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSetting(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_EQ(lowestBrightness, currentBrightness);

    // 自動輝度調整が有効なときも適切に振る舞う。
    // 現仕様では自動輝度調整時の低輝度時の輝度は一律。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::SetAmbientLightSensorValue(1000.0f);

    lbldfc::EnableDimming();
    EXPECT_TRUE(lbldfc::IsDimmingEnabled());
    lbldfc::SetCurrentBrightnessSetting(1.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSetting(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();

    if (IsAlsEnabledEnvironment())
    {
        currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
        EXPECT_EQ(dimmingBrightness, currentBrightness);

        // 自動輝度調整時の最低輝度を取得。
        lbldfc::SetCurrentBrightnessSetting(0.0f);
        lbldfc::SetAmbientLightSensorValue(0.0f);
        lbldfc::ApplyCurrentBrightnessSettingToBacklight();
        const auto lowestAutoBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();

        // 現仕様ではdimming時の輝度と自動輝度調整時の最低輝度は一致する。
        EXPECT_EQ(lowestAutoBrightness, dimmingBrightness);
    }

    // VR モードに切り替わっても適切に振る舞う。
    // 輝度設定値が低いときは低輝度化されない(既に低輝度だから)
    lbldfc::EnableVrMode();
    ASSERT_TRUE(lbldfc::IsVrModeEnabled());

    lbldfc::DisableDimming();
    lbldfc::SetCurrentBrightnessSettingForVrMode(0.0f);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    const auto lowestVrBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();

    lbldfc::EnableDimming();
    EXPECT_TRUE(lbldfc::IsDimmingEnabled());
    lbldfc::SetCurrentBrightnessSettingForVrMode(1.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSettingForVrMode(0.3f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(dimmingBrightness, currentBrightness);
    lbldfc::SetCurrentBrightnessSettingForVrMode(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    currentBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_FLOAT_EQ(lowestVrBrightness, currentBrightness);
}

/**
 * @brief   VRモードの切り替えが正しくされるか
 */
TEST(Basic, VrMode)
{
    const auto VrBrightnessScale = GetVrBrightnessScale();
    const auto VrLowerBrightness = GetVrLowerBrightness();

    nnt::lbl::LblInitializer initializer;

    // 輝度設定
    const auto BrightnessSettingForNormalMode = 0.0f;
    const auto BrightnessSettingForVrMode = 1.0f;
    lbldfc::SetCurrentBrightnessSetting(BrightnessSettingForNormalMode);
    lbldfc::SetCurrentBrightnessSettingForVrMode(BrightnessSettingForVrMode);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    // 初期状態は非 VR モード
    EXPECT_FALSE(lbldfc::IsVrModeEnabled());
    EXPECT_EQ(nnt::lbl::MinManualControlBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());

    // VRモードに切り替えると、VR モード用の輝度設定が使用される
    lbldfc::EnableVrMode();
    EXPECT_TRUE(lbldfc::IsVrModeEnabled());;
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    auto baseVrBrightness = static_cast<double>(BrightnessSettingForVrMode) * static_cast<double>(VrBrightnessScale);
    EXPECT_FLOAT_EQ( static_cast<float>(std::sqrt( baseVrBrightness * baseVrBrightness + VrLowerBrightness )),
                     lbldfc::GetBrightnessSettingAppliedToBacklight() );
    lbldfc::SetCurrentBrightnessSettingForVrMode(0.5f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    baseVrBrightness = 0.5f * VrBrightnessScale;
    EXPECT_FLOAT_EQ( static_cast<float>(std::sqrt( baseVrBrightness * baseVrBrightness + VrLowerBrightness )),
               lbldfc::GetBrightnessSettingAppliedToBacklight() );

    // 解除すると、通常モードの輝度に即座に戻る
    lbldfc::DisableVrMode();
    EXPECT_FALSE(lbldfc::IsVrModeEnabled());
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    EXPECT_FLOAT_EQ(nnt::lbl::MinManualControlBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());

    // 自動輝度調整が有効なときも適切に振る舞う。
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::SetAmbientLightSensorValue(1000.0f);

    lbldfc::EnableVrMode();
    EXPECT_TRUE(lbldfc::IsVrModeEnabled());
    lbldfc::SetCurrentBrightnessSettingForVrMode(BrightnessSettingForVrMode);
    ApplyCurrentBrightnessSettingUntilVrTransitionFinished();

    baseVrBrightness = BrightnessSettingForVrMode * VrBrightnessScale;
    EXPECT_FLOAT_EQ(static_cast<float>(std::sqrt(baseVrBrightness * baseVrBrightness + VrLowerBrightness)), lbldfc::GetBrightnessSettingAppliedToBacklight());
    lbldfc::SetCurrentBrightnessSettingForVrMode(0.3f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    baseVrBrightness = 0.3f * VrBrightnessScale;
    EXPECT_FLOAT_EQ(static_cast<float>(std::sqrt(baseVrBrightness * baseVrBrightness + VrLowerBrightness)), lbldfc::GetBrightnessSettingAppliedToBacklight());
}

/**
 * @brief   写像の係数の切り替えが正しくされるか
 */
//TEST(Basic, Mapping)
//{
//    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
//}
