﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/la/la_Api.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>

#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/la/la_NifmToWifiWebAuthArgumentsReader.h>
#include <nn/la/la_AppletToNifmArgumentsWriter.h>

#include <nn/nifm/nifm_Result.h>

namespace
{

    void CheckCommonArguments(const nn::la::CommonArgumentsReader& commonArgumentsReader) NN_NOEXCEPT
    {
        EXPECT_GE(commonArgumentsReader.GetLaVersion(), 0);
        EXPECT_EQ(nn::oe::ThemeColorType_Default, commonArgumentsReader.GetColorIndex());
        EXPECT_TRUE(commonArgumentsReader.IsPlayStartupSound());
    }

    void CheckNifmToWifiWebAuthArguments(const nn::la::NifmToWifiWebAuthArgumentsReader wifiWebAuthArgumentsReader) NN_NOEXCEPT
    {
        EXPECT_EQ(0, nn::util::Strncmp("http://example.com", wifiWebAuthArgumentsReader.GetAuthUrlPointer(), nn::la::detail::NifmToWifiWebAuthArgumentsDataV0::AuthUrlSize));
        EXPECT_EQ(0, nn::util::Strncmp("http://ctest.cdn.nintendo.net/", wifiWebAuthArgumentsReader.GetConnTestUrlPointer(), nn::la::detail::NifmToWifiWebAuthArgumentsDataV0::ConnTestUrlSize));
        nn::util::Uuid id;
        id.FromString("C842618B-41FC-40AB-9A55-EE35452B39B3");
        EXPECT_EQ(id, wifiWebAuthArgumentsReader.GetProfileId());
        EXPECT_EQ(100, wifiWebAuthArgumentsReader.GetSubmitId());
    }
}

void LibraryAppletNetConnectMain(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    NN_LOG("===== DummyWifiWebAuth Start =====\n");

    nn::la::CommonArgumentsReader commonArgumentsReader;

    // CommonArgumentsReader::ReadFromStorage() で読み込み Unpop
    {
        nn::applet::StorageHandle storageHandle;
        ASSERT_TRUE(nn::ae::TryPopFromInChannel(&storageHandle));
        ASSERT_TRUE(commonArgumentsReader.ReadFromStorage(storageHandle));

        CheckCommonArguments(commonArgumentsReader);

        nn::ae::UnpopToInChannel(storageHandle);
    }

    // CommonArgumentsReader::TryPopFromInChannel() で読み込み
    {
        ASSERT_TRUE(commonArgumentsReader.TryPopFromInChannel());

        CheckCommonArguments(commonArgumentsReader);

        NN_LOG("CommonArguments:\n");
        NN_LOG("  laVersion = %d\n", commonArgumentsReader.GetLaVersion());
        NN_LOG("  colorIndex = %d\n", commonArgumentsReader.GetColorIndex());
        NN_LOG("  isPlayStartupSound = %s\n", commonArgumentsReader.IsPlayStartupSound() ? "true" : "false");
    }

    nn::la::NifmToWifiWebAuthArgumentsReader wifiWebAuthArgumentsReader;

    // NifmToWifiWebAuthArgumentsReader::ReadFromStorage() で読み込み Unpop
    {
        nn::applet::StorageHandle storageHandle;
        ASSERT_TRUE(nn::ae::TryPopFromInChannel(&storageHandle));
        ASSERT_TRUE(wifiWebAuthArgumentsReader.ReadFromStorage(storageHandle));

        CheckNifmToWifiWebAuthArguments(wifiWebAuthArgumentsReader);

        nn::ae::UnpopToInChannel(storageHandle);
    }

    // NifmToWifiWebAuthArgumentsReader::TryPopFromInChannel() で読み込み
    {
        ASSERT_TRUE(wifiWebAuthArgumentsReader.TryPopFromInChannel());

        CheckNifmToWifiWebAuthArguments(wifiWebAuthArgumentsReader);

        char uuidStr[nn::util::Uuid::StringSize];

        NN_LOG("WifiWebAuthArguments:\n");
        NN_LOG("  authUrl = %s\n", wifiWebAuthArgumentsReader.GetAuthUrlPointer());
        NN_LOG("  connTestUrl = %s\n", wifiWebAuthArgumentsReader.GetConnTestUrlPointer());
        NN_LOG("  profileId = %s\n", wifiWebAuthArgumentsReader.GetProfileId().ToString(uuidStr, nn::util::Uuid::StringSize));
        NN_LOG("  submitId = %d\n", wifiWebAuthArgumentsReader.GetSubmitId());
    }

    // 疎通確認
    // ...

    // （テストとして）SubmitId に応じて、将来バージョンの応答を返す
    static const uint32_t SubmitIdForFutureVersion = 100;
    if (SubmitIdForFutureVersion == wifiWebAuthArgumentsReader.GetSubmitId())
    {
        nn::la::detail::AppletToNifmArgumentsDataV0 appletToNifmArgumentsDataV0;
        appletToNifmArgumentsDataV0.result = nn::nifm::ResultErrorHandlingCompleted();

        static const uint32_t AppletToNifmVersionFuture = 1;
        static const uint32_t DummyFutureParameter = 1234;

        {
            nn::applet::StorageHandle storageHandle;
            NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(AppletToNifmVersionFuture) + sizeof(appletToNifmArgumentsDataV0) + sizeof(DummyFutureParameter)));

            int size = 0;
            NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &AppletToNifmVersionFuture, sizeof(AppletToNifmVersionFuture)));
            size += sizeof(AppletToNifmVersionFuture);
            NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &appletToNifmArgumentsDataV0, sizeof(appletToNifmArgumentsDataV0)));
            size += sizeof(appletToNifmArgumentsDataV0);
            NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &DummyFutureParameter, sizeof(DummyFutureParameter)));
            nn::ae::PushToOutChannel(storageHandle);
        }
    }
    else
    {
        // 通常の成功を返す
        nn::la::AppletToNifmArgumentsWriter appletToNifmArgumentsWriter(true);
        appletToNifmArgumentsWriter.PushToOutChannel();
    }
}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、自前で nninitStartup() を用意しています。
}

extern "C" void nnMain()
{
    nn::ae::InvokeLibraryAppletMain(LibraryAppletNetConnectMain);
}

