﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lmem/lmem_UnitHeap.h>

#include "testLMem_ThreadSafe.h"

namespace {

/**
 * @brief   ユニットヒープのテストフィクスチャです。
 */
class UnitHeapThreadSafeTest : public LMemThreadSafeTest
{
protected:
    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        const size_t UnitSize = sizeof(nn::Bit64);
        const size_t HeapSize = nn::lmem::GetRequiredUnitHeapSize(UnitSize, MaxThreadCount, nn::lmem::DefaultAlignment, true);

        InitializeCommmon();

        m_HeapHandle = nn::lmem::CreateUnitHeap(m_Heap, HeapSize, UnitSize, nn::lmem::CreationOption_ThreadSafe | nn::lmem::CreationOption_DebugFill);
        ASSERT_NOT_NULL(m_HeapHandle);
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        nn::lmem::DestroyUnitHeap(m_HeapHandle);
        std::free(m_Heap);
    }
};

// =============
// スレッド関数
// =============

/**
 * @brief   メモリブロックの確保と解放を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 */
void AllocateFreeUnitHeap(void* arg)
{
    CheckStruct* ptr = reinterpret_cast<CheckStruct*>(arg);

    const int BlockCount = 1024;        // 確保するユニットの数
    nn::Bit64* pMemoryBlock;            // 確保したメモリブロックのアドレス

    for(int i = 0; i < BlockCount; ++i)
    {
        pMemoryBlock = reinterpret_cast<nn::Bit64*>(nn::lmem::AllocateFromUnitHeap(ptr->handle));
        if(pMemoryBlock != nullptr)
        {
            *pMemoryBlock = ptr->fill;
            // 確保済みの領域が誤って別スレッドから上書きされていないか確認
            ASSERT_EQ(*pMemoryBlock, ptr->fill);
            nn::lmem::FreeToUnitHeap(ptr->handle, pMemoryBlock);
        }
    }
}

// =======
// テスト
// =======

/**
 * @brief   複数スレッドを作成し、確保と解放を繰り返します。
 */
TEST_F(UnitHeapThreadSafeTest, AllocateFreeUnitHeap)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap,
                                                        AllocateFreeUnitHeap };

    DoMultiThreadTestWithCheckStruct(functions, FunctionCount);
}

}   // unnamed namespace
